(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

(** Bound ASTs. *)

open Ident
open Ast

type region =
  | RVar of Ident.t
  | RSub of region * string

type base_type =
  | TUnit
  | TInt
  | TBool
  | TString

type type_expr = type_node node

and type_node =
  | TETuple of type_expr list
  | TEPointer of class_expr * region
  | TEIdent of Ident.t
  | TEBase of base_type
  | TESum of type_expr * type_expr
  | TELogic of type_expr list * Ident.t

and class_expr = region list * type_expr list * Ident.t

type permission = region Ast.permission

type deref_kind =
  | DKUnavailable (* no permission available on the region which is read *)
  | DKSingleton
  | DKGroup
  | DKFocus of region (* DKFocus(s): s -> r is available *)

type adopt_kind =
  | AKSingletonToEmpty
  | AKSingletonToGroup
  | AKGroupToGroup
  | AKSingletonToFocus of region
  | AKGroupToFocus of region (* region s: s -> r is available *)

type weaken_kind =
  | WKEmptyToGroup
  | WKClosedToGroup

type unfocus_kind =
  | UKClosed
  | UKGroup

type expr = expr_node node

and expr_node =
  | Const of const
  | Unop of un_op * expr
  | Binop of bin_op * expr * expr
  | Tuple of expr list
  | Proj of expr * int * int (* last int = tuple length *)
  | Left of expr
  | Right of expr
  | Var of Ident.t
  | Let of Ident.t * expr * expr
  | Seq of expr * expr
  | Call of
      Ident.t (* called value name *)
      * expr list (* parameters *)
      * permission list (* consumes (after substitution) *)
      * permission list (* produces (after substitution) *)
      * (Ident.t * region) list (* applied substitution *)
      * permission list
        (* available permissions for the call
           (only valid after inference, includes consumes) *)
  | If of expr * expr * expr
  | While of expr * expr * assertion * permission list
  | Assign of expr * expr * region (* region = region of first expr *)
  | Deref of expr * region * deref_kind * permission list (* region = region of expr *)
  | New of class_expr * region
  | Pack of expr * region (* region = region of expr *)
  | Unpack of expr * region (* region = region of expr *)
  | Adopt of expr * region * region * adopt_kind
      (* last region = region of expr *)
  | Focus of expr * region * region (* last region = region of expr *)
  | Unfocus of expr * region * region (* last region = region of expr *)
  | Region of Ident.t * expr * class_expr option
  | Print of string * expr

  | LogicApp of Ident.t * expr list * region list * permission list
      (* fourth arguments (perms) is used for expressions (not terms)
         to know which permissions are available at this point *)

  | AdoptRegion of region * region * adopt_kind
  | UnfocusRegion of region * region * unfocus_kind
  | PackRegion of region
  | UnpackRegion of region
  | WeakenRegion of region * weaken_kind

  | BlackBox of permission list * permission list
  | Typed of expr * type_expr
  | Old of expr
  | Assert of assertion * permission list

(** We only use parts of the AST *)
and term = expr

and assertion = assertion_node node

and assertion_node =
  | PTrue
  | PFalse
  | PIff of assertion * assertion
  | PImplies of assertion * assertion
  | PAnd of assertion * assertion
  | POr of assertion * assertion
  | PNot of assertion
  | PEqual of term * term
  | PDiff of term * term
  | PLt of term * term
  | PGt of term * term
  | PLe of term * term
  | PGe of term * term
  | PTerm of term
  | PForall of Ident.t * type_expr * assertion
  | PExists of Ident.t * type_expr * assertion
  | PApp of Ident.t * term list * region list
  | POld of assertion

type class_def = {
  c_region_params: Ident.t list;
  c_type_params: Ident.t list;
  c_name: Ident.t;
  c_owned_regions: Ident.t list;
  c_type: type_expr;
  c_invariant: Ident.t * assertion * region list;
}

type value_def = {
  v_name: Ident.t;
  v_params: (Ident.t * type_expr) list;
  v_return_type: type_expr;
  v_consumes: permission list;
  v_produces: permission list;
  v_pre: assertion;
  v_post: assertion;
  v_body: expr option;
  v_regions: Ident.t list;
  v_type_variables: Ident.t list;
  v_result: Ident.t; (* result in post-condition *)
}

type logic_type_def = {
  lt_name: Ident.t;
  lt_type_params: Ident.t list;
}

type logic_function_def = {
  lf_name: Ident.t;
  lf_params: type_expr list;
  lf_return_type: type_expr;
  lf_type_variables: Ident.t list;
  lf_region_variables: Ident.t list;
}

type axiom_def = {
  ax_name: Ident.t;
  ax_assertion: assertion;
  ax_type_variables: Ident.t list;
  ax_region_variables: Ident.t list;
}

type predicate_def = {
  p_name: Ident.t;
  p_params: (Ident.t * type_expr) list;
  p_body: assertion option;
  p_type_variables: Ident.t list;
  p_region_variables: Ident.t list;
}

type def =
  | Class of class_def
  | Value of value_def
  | LogicType of logic_type_def
  | LogicFunction of logic_function_def
  | Axiom of axiom_def
  | Predicate of predicate_def

type file = def list
