(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

(** Transform a parsed AST into a bound AST. *)

open Misc
open Ast
module P = Past
module B = Bast

type environment = {
  variables: unit Env.t;
  type_variables: unit Env.t;
  values: unit Env.t;
  regions: unit Env.t;
  classes: unit Env.t;
  logic_types: unit Env.t;
  logic_functions: unit Env.t;
  predicates: unit Env.t;
}

let empty_env = {
  variables = Env.empty;
  type_variables = Env.empty;
  values = Env.empty;
  regions = Env.empty;
  classes = Env.empty;
  logic_types = Env.empty;
  logic_functions = Env.empty;
  predicates = Env.empty;
}

let dummy_region = B.RVar (Ident.create "dummy_region")

let lookup loc env s =
  try
    Env.lookup env s
  with Not_found ->
    Loc.locate_error loc "unbound identifier: %s" s

let rec region env r =
  match r.node with
    | P.RVar s -> B.RVar (lookup r.loc env.regions s)
    | P.RSub (r, s) -> B.RSub (region env r, s)

let permission env = function
  | PEmpty r -> PEmpty (region env r)
  | POpen r -> POpen (region env r)
  | PClosed r -> PClosed (region env r)
  | PGroup r -> PGroup (region env r)
  | PArrow (s, r) -> PArrow (region env s, region env r)
  | PSub (s, r) -> PSub (region env s, region env r)

let rec typ env t =
  let typ = typ env in
  let region = region env in
  let lookup = lookup t.loc in
  let node = match t.node with
    | P.TETuple l ->
        B.TETuple (List.map typ l)
    | P.TEPointer (([], [], "int"), None) ->
        B.TEBase B.TInt
    | P.TEPointer (([], [], "bool"), None) ->
        B.TEBase B.TBool
    | P.TEPointer (([], [], "unit"), None) ->
        B.TEBase B.TUnit
    | P.TEPointer (([], [], "string"), None) ->
        B.TEBase B.TString
    | P.TEPointer ((rl, tl, c), r) ->
        begin try
          let c = Env.lookup env.classes c in
          let r = match r with
            | Some r -> region r
            | None -> B.RVar (Ident.create "_auto")
          in
          B.TEPointer ((List.map region rl, List.map typ tl, c), r)
        with Not_found ->
          try
            B.TEIdent (Env.lookup env.type_variables c)
          with Not_found -> begin
            let c = lookup env.logic_types c in
            if rl <> [] || r <> None then
              Loc.locate_error t.loc
                "%a is a logic type,@ not a class type;@ it cannot have regions"
                Pp.identifier c;
            B.TELogic (List.map typ tl, c)
          end
        end
    | P.TESum (a, b) ->
        B.TESum (typ a, typ b)
  in
  { node = node;
    loc = t.loc }

and class_expr env (rl, tl, c) =
  (* TODO: reuse in function typ to factorize *)
  let c = Env.lookup env.classes c in
  let rl = List.map (region env) rl in
  let tl = List.map (typ env) tl in
  rl, tl, c

let add_list env l = List.fold_left (fun a b -> Env.add a b ()) env l

let selectors = Hashtbl.create 16

let make_selector names =
  Misc.list_iteri (fun i name -> Hashtbl.add selectors name (i + 1)) names

let selector loc name =
  try
    Hashtbl.find selectors name
  with Not_found ->
    Loc.locate_error loc "unknown selector: %s" name

let selector loc = function
  | P.SInt i -> i
  | P.SString s -> selector loc s

let rec expr env e =
  let fresh_regions = ref [] in
  let expr ?(env = env) e =
    let e, fr = expr env e in
    fresh_regions := fr @ !fresh_regions;
    e
  in
  let region ?(env = env) = region env in
  let typ ?(env = env) = typ env in
  let permission ?(env = env) = permission env in
  let fresh_region s =
    let r = Ident.create s in
    fresh_regions := r :: !fresh_regions;
    r
  in
  let lookup = lookup e.loc in
  let node = match e.node with
    | P.Const c ->
        B.Const c
    | P.Unop (o, a) ->
        B.Unop (o, expr a)
    | P.Binop (o, a, b) ->
        B.Binop (o, expr a, expr b)
    | P.Tuple l ->
        B.Tuple (List.map expr l)
    | P.Proj (a, s) ->
        B.Proj (expr a, selector e.loc s, -1)
    | P.Left a ->
        B.Left (expr a)
    | P.Right a ->
        B.Right (expr a)
    | P.Var s ->
        begin try
          let v = Env.lookup env.variables s in
          B.Var v
        with Not_found ->
          B.LogicApp (lookup env.logic_functions s, [], [], [])
        end
    | P.Let (s, a, b) ->
        let x = Ident.create s in
        let env = { env with variables = Env.add env.variables x () } in
        B.Let (x, expr a, expr ~env b)
    | P.Seq (a, b) ->
        B.Seq (expr a, expr b)
    | P.Old x ->
        B.Old (expr x)
    | P.Call (s, l) ->
        let l = List.map expr l in
        begin try
          let x = Env.lookup env.values s in
          B.Call (x, l, [], [], [], [])
        with Not_found ->
          B.LogicApp (lookup env.logic_functions s, l, [], [])
        end
    | P.If (a, b, c) ->
        B.If (expr a, expr b, expr c)
    | P.While (a, b, c) ->
        B.While (expr a, expr_and_bind env b, assertion env c, [])
    | P.Assign (a, b) ->
        B.Assign (expr a, expr b, dummy_region)
    | P.Deref a ->
        B.Deref (expr a, dummy_region, B.DKUnavailable, [])
    | P.New ((rl, tl, s), r) ->
        let c = lookup env.classes s in
        let r = match r with
          | Some r -> region r
          | None -> B.RVar (fresh_region "_auto_new")
        in
        B.New ((List.map (region ~env) rl, List.map (typ ~env) tl, c), r)
    | P.Pack a ->
        B.Pack (expr a, dummy_region)
    | P.Unpack a ->
        B.Pack (expr a, dummy_region)
    | P.Adopt (a, r) ->
        B.Adopt (expr a, region r, dummy_region, B.AKSingletonToEmpty)
    | P.Focus (a, s) ->
        let r = match s with
          | Some r -> region r
          | None -> B.RVar (fresh_region "_auto_focus")
        in
        B.Focus (expr a, r, dummy_region)
    | P.Unfocus (a, r) ->
        B.Unfocus (expr a, region r, dummy_region)
    | P.Region (s, a, c) ->
        let x = Ident.create s in
        let env = { env with regions = Env.add env.regions x () } in
        Opt.iter (fun c -> Regiontypes.add x (class_expr env c)) c;
        B.Region (x, expr ~env a, Opt.map (class_expr env) c)
    | P.Print (s, a) ->
        B.Print (s, expr a)
    | P.AdoptRegion (s, r) ->
        B.AdoptRegion (region s, region r, B.AKSingletonToEmpty)
    | P.UnfocusRegion (s, r) ->
        B.UnfocusRegion (region s, region r, B.UKClosed)
    | P.PackRegion r ->
        B.PackRegion (region r)
    | P.UnpackRegion r ->
        B.UnpackRegion (region r)
    | P.BlackBox (c, p) ->
        B.BlackBox (List.map (permission ~env) c, List.map (permission ~env) p)
    | P.Typed (e, t) ->
        B.Typed (expr e, typ t)
    | P.Assert a ->
        B.Assert (assertion env a, [])
  in
  { node = node;
    loc = e.loc }, !fresh_regions

and bind_region e r =
  { node = B.Region (r, e, None);
    loc = e.loc }

and expr_and_bind env e =
  let e, r = expr env e in
  List.fold_left bind_region e r

and assertion env a =
  let assertion ?(env = env) a = assertion env a in
  let term ?(env = env) e = expr_and_bind env e in
  let typ ?(env = env) t = typ env t in
  let node = match a.node with
    | P.PTrue ->
        B.PTrue
    | P.PFalse ->
        B.PFalse
    | P.PIff (a, b) ->
        B.PIff (assertion a, assertion b)
    | P.PImplies (a, b) ->
        B.PImplies (assertion a, assertion b)
    | P.PAnd (a, b) ->
        B.PAnd (assertion a, assertion b)
    | P.POr (a, b) ->
        B.POr (assertion a, assertion b)
    | P.PNot a ->
        B.PNot (assertion a)
    | P.PEqual (a, b) ->
        B.PEqual (term a, term b)
    | P.PDiff (a, b) ->
        B.PDiff (term a, term b)
    | P.PGt (a, b) ->
        B.PGt (term a, term b)
    | P.PLt (a, b) ->
        B.PLt (term a, term b)
    | P.PGe (a, b) ->
        B.PGe (term a, term b)
    | P.PLe (a, b) ->
        B.PLe (term a, term b)
    | P.PTerm e ->
        B.PTerm (term e)
    | P.PForall (x, t, a) ->
        let x = Ident.create x in
        let env = { env with variables = Env.add env.variables x () } in
        B.PForall (x, typ t, assertion ~env a)
    | P.PExists (x, t, a) ->
        let x = Ident.create x in
        let env = { env with variables = Env.add env.variables x () } in
        B.PExists (x, typ t, assertion ~env a)
    | P.POld a ->
        B.POld (assertion a)
    | P.PApp (x, tl) ->
        let x = lookup a.loc env.predicates x in
        let tl = List.map term tl in
        B.PApp (x, tl, [])
  in
  { node = node;
    loc = a.loc }

let invariant env (x, a) =
  let x = Ident.create x in
  let env = { env with variables = Env.add env.variables x () } in
  x, assertion env a, []

let class_def env x =
  let rp = List.map Ident.create x.P.c_region_params in
  let ro = List.map Ident.create x.P.c_owned_regions in
  let tp = List.map Ident.create x.P.c_type_params in
  let name = Ident.create x.P.c_name in
  let env = { env with classes = Env.add env.classes name () } in
  let iregions = add_list env.regions rp in
  let iregions = add_list iregions ro in
  let ienv =
    { env with
        regions = iregions;
        type_variables = add_list env.regions tp }
  in
  env,
  { B.c_region_params = rp;
    B.c_type_params = tp;
    B.c_name = name;
    B.c_owned_regions = ro;
    B.c_type = typ ienv x.P.c_type;
    B.c_invariant = invariant ienv x.P.c_invariant }

let rec add_rvar acc r =
  match r.node with
    | P.RVar s -> StringSet.add s acc
    | P.RSub (r, _) -> add_rvar acc r

let rec free_regions_of_type_expr acc x =
  match x.node with
    | P.TEPointer ((rl, tl, _), r) ->
        let acc = Opt.fold add_rvar acc r in
        let acc = List.fold_left add_rvar acc rl in
        List.fold_left free_regions_of_type_expr acc tl
    | P.TETuple tl ->
        List.fold_left free_regions_of_type_expr acc tl
    | P.TESum (a, b) ->
        free_regions_of_type_expr (free_regions_of_type_expr acc a) b

let rec free_variables_of_type_expr env acc x =
  match x.node with
    | P.TEPointer (([], [], c), None) -> (* a little hackish *)
        if Env.mem_string env.classes c then
          acc
        else
          StringSet.add c acc
    | P.TEPointer ((_, tl, _), _)
    | P.TETuple tl ->
        List.fold_left (free_variables_of_type_expr env) acc tl
    | P.TESum (a, b) ->
        free_variables_of_type_expr env
          (free_variables_of_type_expr env acc a) b

let rec free_type_variables_of_assertion env acc x =
  match x.node with
    | P.PTrue
    | P.PFalse ->
        acc
    | P.PIff (a, b)
    | P.PImplies (a, b)
    | P.PAnd (a, b)
    | P.POr (a, b) ->
        free_type_variables_of_assertion env
          (free_type_variables_of_assertion env acc a) b
    | P.POld a
    | P.PNot a ->
        free_type_variables_of_assertion env acc a
    | P.PEqual _
    | P.PDiff _
    | P.PLt _
    | P.PGt _
    | P.PLe _
    | P.PGe _
    | P.PTerm _
    | P.PApp _ ->
        acc
    | P.PForall (_, t, a)
    | P.PExists (_, t, a) ->
        free_type_variables_of_assertion env
          (free_variables_of_type_expr env acc t) a

let rec free_region_variables_of_assertion acc x =
  match x.node with
    | P.PTrue
    | P.PFalse ->
        acc
    | P.PIff (a, b)
    | P.PImplies (a, b)
    | P.PAnd (a, b)
    | P.POr (a, b) ->
        free_region_variables_of_assertion
          (free_region_variables_of_assertion acc a) b
    | P.POld a
    | P.PNot a ->
        free_region_variables_of_assertion acc a
    | P.PEqual _
    | P.PDiff _
    | P.PLt _
    | P.PGt _
    | P.PLe _
    | P.PGe _
    | P.PTerm _
    | P.PApp _ ->
        acc
    | P.PForall (_, t, a)
    | P.PExists (_, t, a) ->
        free_region_variables_of_assertion
          (free_regions_of_type_expr acc t) a

let type_and_region_variables fr ft l =
  let regions =
    List.fold_left
      fr
      StringSet.empty
      l
    |> StringSet.elements
    |> List.map Ident.create
  in
  let type_vars =
    List.fold_left
      ft
      StringSet.empty
      l
    |> StringSet.elements
    |> List.map Ident.create
  in
  type_vars, regions

let enter_type_and_region (t, r) env =
  { env with
      type_variables = add_list env.type_variables t;
      regions = add_list env.regions r },
  t,
  r

let type_and_region_variables_of_types env types =
  enter_type_and_region
    (type_and_region_variables
       free_regions_of_type_expr
       (free_variables_of_type_expr env)
       types)
    env

let type_and_region_variables_of_assertion env body =
  enter_type_and_region
    (type_and_region_variables
       free_region_variables_of_assertion
       (free_type_variables_of_assertion env)
       [ body ])
    env

let result = Ident.create "result"

let value_def env x =
  let name = Ident.create x.P.v_name in
  let env = { env with values = Env.add env.values name () } in
  let ienv, type_vars, regions =
    type_and_region_variables_of_types env
      (x.P.v_return_type :: List.map snd x.P.v_params)
  in
  let params =
    List.map (fun (s, t) -> Ident.create s, typ ienv t) x.P.v_params in
  let ienv =
    { ienv with variables = add_list env.variables (List.map fst params) }
  in
  let body = Opt.map (expr_and_bind ienv) x.P.v_body in
  let ienv_post = { ienv with variables = Env.add ienv.variables result () } in
  env,
  { B.v_name = name;
    B.v_params = params;
    B.v_regions = regions;
    B.v_type_variables = type_vars;
    B.v_return_type = typ ienv x.P.v_return_type;
    B.v_consumes = List.map (permission ienv) x.P.v_consumes;
    B.v_produces = List.map (permission ienv) x.P.v_produces;
    B.v_pre = assertion ienv x.P.v_pre;
    B.v_post = assertion ienv_post x.P.v_post;
    B.v_body = body;
    B.v_result = result }

let logic_type_def env x =
  let name = Ident.create x.P.lt_name in
  let tp = List.map Ident.create x.P.lt_type_params in
  let env = { env with logic_types = Env.add env.logic_types name () } in
  env,
  { B.lt_name = name;
    B.lt_type_params = tp }

let logic_function_def env x =
  let name = Ident.create x.P.lf_name in
  let ienv, type_vars, regions =
    type_and_region_variables_of_types env
      (x.P.lf_return_type :: x.P.lf_params)
  in
  let params = List.map (typ ienv) x.P.lf_params in
  let rty = typ ienv x.P.lf_return_type in
  let env =
    { env with logic_functions = Env.add env.logic_functions name () } in
  env,
  { B.lf_name = name;
    B.lf_params = params;
    B.lf_return_type = rty;
    B.lf_type_variables = type_vars;
    B.lf_region_variables = regions }

let axiom_def env x =
  let ienv, type_vars, regions =
    type_and_region_variables_of_assertion env x.P.ax_assertion in
  env,
  { B.ax_name = Ident.create x.P.ax_name;
    B.ax_assertion = assertion ienv x.P.ax_assertion;
    B.ax_type_variables = type_vars;
    B.ax_region_variables = regions }

let predicate_def env x =
  let name = Ident.create x.P.p_name in
  let ienv, type_vars, regions =
    let ienv1, t1, r1 =
      type_and_region_variables_of_types env (List.map snd x.P.p_params) in
    let ienv2, t2, r2 =
      match x.P.p_body with
        | None ->
            ienv1, [], []
        | Some body ->
            type_and_region_variables_of_assertion ienv1 body
    in
    ienv2, t1 @ t2, r1 @ r2
  in
  let params =
    List.map (fun (s, t) -> Ident.create s, typ ienv t) x.P.p_params in
  let ienv =
    { ienv with variables = add_list env.variables (List.map fst params) } in
  { env with predicates = Env.add env.predicates name () },
  { B.p_name = name;
    B.p_params = params;
    B.p_body = Opt.map (assertion ienv) x.P.p_body;
    B.p_type_variables = type_vars;
    B.p_region_variables = regions }

let def env = function
  | P.Class x -> let env, x = class_def env x in env, [B.Class x]
  | P.Value x -> let env, x = value_def env x in env, [B.Value x]
  | P.LogicType x -> let env, x = logic_type_def env x in env, [B.LogicType x]
  | P.LogicFunction x ->
      let env, x = logic_function_def env x in env, [B.LogicFunction x]
  | P.Axiom x -> let env, x = axiom_def env x in env, [B.Axiom x]
  | P.Selector l -> make_selector l; env, []
  | P.Predicate x -> let env, x = predicate_def env x in env, [B.Predicate x]

let file ?(env = empty_env) x =
  let a, b = list_fold_map def env x in
  a, List.flatten b
