(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Misc
open Ast
open Bast
open Asttools

type expr_result = {
  expr: expr;
  produces: permission list;
}

let eq_permission a b =
  Unify.expand_permission a = Unify.expand_permission b

let eq_region a b =
  Unify.expand_region a = Unify.expand_region b

let consume from perm =
  list_removef (eq_permission perm) from

let produce into perm =
  perm :: into

let rec is_parent_region_of region r =
  eq_region region r ||
    match r with
      | RVar _ -> false
      | RSub (r, _) -> is_parent_region_of region r

let permission_mentions region = function
  | PEmpty r
  | POpen r
  | PClosed r
  | PGroup r ->
      is_parent_region_of region r
  | PArrow (s, r)
  | PSub (s, r) ->
      is_parent_region_of region r || is_parent_region_of region s

let consume_and_produce perms c p =
  let perms = List.fold_left consume perms c in
  List.fold_left produce perms p

let classes_environment = ref Env.empty
let find_class c = Env.find !classes_environment c
let class_expr_of_region r = Regiontypes.get find_class r

let owned_regions r =
  let (_, _, c) = class_expr_of_region r in
  let c = find_class c in
  List.map (fun s -> RSub (r, Ident.name s)) c.c_owned_regions

let unpack r =
(*  log "@[<hov 2>OWNED REGIONS OF@ %a@ ARE@ %a@]@."
    Pp.region r
    (Pp.list ~sep: Pp.comma_space Pp.region) (owned_regions r);*)
  let owns = List.map (fun r -> PGroup r) (owned_regions r) in
  UnpackRegion r, [ PClosed r ], POpen r :: owns

let pack r =
(*  log "@[<hov 2>OWNED REGIONS OF@ %a@ ARE@ %a@]@."
    Pp.region r
    (Pp.list ~sep: Pp.comma_space Pp.region) (owned_regions r);*)
  let owns = List.map (fun r -> PGroup r) (owned_regions r) in
  PackRegion r, POpen r :: owns, [ PClosed r ]

let operations_producing context perm =
  let find_arrows r =
    list_filterf
      (function
         | PArrow (s, r') when eq_region r r' ->
             Some s
         | PEmpty _ | POpen _ | PClosed _ | PGroup _ | PSub _
         | PArrow _ ->
             None)
      context
  in
  match perm with
    | PEmpty r ->
        []
    | POpen r ->
        [ unpack r ]
    | PClosed r ->
        [ pack r ]
    | PGroup r ->
        let simples =
          [ WeakenRegion (r, WKEmptyToGroup), [ PEmpty r ], [ PGroup r ];
            WeakenRegion (r, WKClosedToGroup), [ PClosed r ], [ PGroup r ] ]
        in
        let arrow_contextuals s =
          [ UnfocusRegion (s, r, UKClosed),
            [ PClosed s; PArrow (s, r) ],
            [ PGroup r ];
            UnfocusRegion (s, r, UKGroup),
            [ PGroup s; PArrow (s, r) ],
            [ PGroup r ] ]
        in
        let unpacks =
          match Unify.expand_region r with
            | RVar _ ->
                [ ]
            | RSub (r', _) ->
                [ unpack r' ]
        in
        let arrows = find_arrows r in
        simples @ (List.flatten (List.map arrow_contextuals arrows)) @ unpacks
    | PArrow (s, r) ->
        []
    | PSub (s, r) ->
        let simples =
          [ Const CUnit,
            [ PEmpty s; PEmpty r ],
            [ PEmpty r; PSub (s, r) ];
            Const CUnit,
            [ PEmpty s; POpen r ],
            [ POpen r; PSub (s, r) ];
            Const CUnit,
            [ PEmpty s; PClosed r ],
            [ PClosed r; PSub (s, r) ];
            Const CUnit,
            [ PEmpty s; PGroup r ],
            [ PGroup r; PSub (s, r) ];
            AdoptRegion (s, r, AKSingletonToEmpty),
            [ PClosed s; PEmpty r ],
            [ PClosed r; PSub (s, r) ];
            AdoptRegion (s, r, AKSingletonToGroup),
            [ PClosed s; PGroup r ],
            [ PGroup r; PSub (s, r) ];
            AdoptRegion (s, r, AKGroupToGroup),
            [ PGroup s; PGroup r ],
            [ PGroup r; PSub (s, r) ] ]
        in
        let arrow_contextuals t =
          [ AdoptRegion (s, r, AKSingletonToFocus t),
            [ PClosed s; PArrow (t, r) ],
            [ PArrow (t, r); PSub (s, r) ];
            AdoptRegion (s, r, AKGroupToFocus t),
            [ PGroup s; PArrow (t, r) ],
            [ PArrow (t, r); PSub (s, r) ]]
        in
        let arrows = find_arrows r in
        (* Warning: concatenation order matters *)
        (List.flatten (List.map arrow_contextuals arrows)) @ simples

exception Cannot_join

(** Construct an expression which uses [consumes] to produce at least
    [produces].

    We start from the unit expression [acc] with [acc.produces = consumes].
    For each permission [perm] in [produces], we remove from [acc.consumes]
    the permissions which are used to produce [perm], and we add [perm]
    along with other permissions which are produced as a side-effect.

    If a permission in [forbidden] appears in the consumptions of an
    operation, this operation will not be used. *)
let rec join forbidden consumes produces =
  List.fold_left
    (join_one forbidden)
    { expr = { loc = Loc.dummy_location; node = Const CUnit };
      produces = consumes }
    produces

and join_one forbidden acc perm =
  if List.exists (eq_permission perm) acc.produces then
    acc
  else try
    let j, o, c, p =
      list_findf
        (fun (o, c, p) ->
           let bad =
             List.exists
               (fun p -> List.exists (eq_permission p) forbidden)
               c
           in
           if bad then
             None
           else try
             Some (join (c @ forbidden) acc.produces c, o, c, p)
           with Cannot_join ->
             (*log "@[<hov 2>failed %a ===> %a@]@."
               (Pp.list ~sep: Pp.comma_space Pp.permission) c
               (Pp.list ~sep: Pp.comma_space Pp.permission) p;*)
             None)
        (operations_producing acc.produces perm)
    in
    (*log "@[<hov 2>applying %a@ for %a ===> %a@]@."
      Pp.expr (make_node o)
      (Pp.list ~sep: Pp.comma_space Pp.permission) c
      (Pp.list ~sep: Pp.comma_space Pp.permission) p;*)
    { expr = make_node (seq acc.expr (make_node (seq j.expr (make_node o))));
      produces = consume_and_produce j.produces c p }
  with Not_found ->
    raise Cannot_join

(** Same as [join] (below) but does not catch [Cannot_join]. *)
let join_ex consumes produces =
  let r = join produces consumes produces in
(*  log "@[<hv 2>INFERENCE:@ @[<hv 2>to produce:@ %a@]@ @[<hv 2>from permissions:@ %a@]@ @[<hv 2>execute: %a@]@ @[<hv 2>and obtain:@ %a@]@]@."
    Pp.permissions produces Pp.permissions consumes Pp.expr r.expr
    Pp.permissions r.produces;*)
  r

let join loc consumes produces =
  (*log
    "@[<hov 2>**** JOIN@ %a@ ===>@ %a@]@."
    (Pp.list ~sep: Pp.comma_space Pp.permission) consumes
    (Pp.list ~sep: Pp.comma_space Pp.permission) produces;*)
  try
    join_ex consumes produces
  with Cannot_join ->
    Loc.locate_error loc
      "@[<hov>permissions@ %a@ are@ needed@ here,@ but@ only@ %a@ are@ available@]"
      (Pp.list ~sep: Pp.comma_space Pp.permission) produces
      (Pp.list ~sep: Pp.comma_space Pp.permission) consumes

(** Decide what to do with [goal].
    If it is in [perms], add it to [acc].
    If not, try to find a permission that can potentially be produced both
    from [perms] and from [goal], and add that permission to [acc]. *)
let find_best_permission perms acc goal =
  match goal with
    | PEmpty r | POpen r | PClosed r | PGroup r | PArrow (_, r) ->
        begin try
          let conflict =
            List.find
              (function
                 | PEmpty r' | POpen r' | PClosed r' | PGroup r'
                 | PArrow (_, r') ->
                     eq_region r r'
                 | PSub _ ->
                     false)
              perms
          in
          if eq_permission goal conflict then
            goal :: acc
          else match goal, conflict with
            | PSub _, _ | _, PSub _ | PClosed _, PClosed _ | POpen _, POpen _ ->
                assert false (* impossible *)
            | (PEmpty _ | PGroup _ | PArrow _), _
            | _, (PEmpty _ | PGroup _ | PArrow _) ->
                PGroup r :: acc
            | POpen _, PClosed _ | PClosed _, POpen _ ->
                POpen r :: acc
        with Not_found ->
          acc
        end
    | PSub (s, r) ->
        goal :: acc

(** Try to produce [perm] from [j] and [k].
    If [join] succeeds, return the modified [j] and [k].
    If [join] fails, return [j] and [k] unchanged. *)
let produce_in_branches (j, k) perm =
  try
    let j' = join_ex j.produces [ perm ] in
    let k' = join_ex k.produces [ perm ] in
    let j = { j' with expr = make_node (seq j.expr j'.expr) } in
    let k = { k' with expr = make_node (seq k.expr k'.expr) } in
    j, k
  with Cannot_join ->
    j, k

let bag_intersection a b =
  let r, _ =
    List.fold_left
      (fun (acc, b) a ->
         try
           let b = list_removef (eq_permission a) b in
           a :: acc, b
         with Not_found ->
           acc, b)
      ([], b) a
  in r

(** Return [(j, k, p)] where [j] is an operation that consumes
    [a] and produces [p], and [k] is an operation that consumes [b]
    and produces [p]. As many productions as possible are kept in [p]. *)
let join_branches a b =
  let perms = List.fold_left (find_best_permission b) [] a in
  let j, k =
    List.fold_left
      produce_in_branches
      ({ expr = make_node (Const CUnit); produces = a },
       { expr = make_node (Const CUnit); produces = b })
      perms
  in
  j, k, bag_intersection a b

(** Change the [expr_result] [a] so that it produces at least [perms]. *)
let force_production a perms =
  let b = join a.expr.loc a.produces perms in
  { expr = { loc = a.expr.loc; node = seqf a.expr b.expr };
    produces = b.produces }

let rec find_deref_kind r = function
  | [] ->
      DKUnavailable
  | perm :: rem ->
      let k =
        match perm with
          | PEmpty _ | PSub _ ->
              DKUnavailable
          | POpen s
          | PClosed s ->
              if eq_region r s then DKSingleton else DKUnavailable
          | PGroup s ->
              if eq_region r s then DKGroup else DKUnavailable
          | PArrow (t, s) ->
              if eq_region r s then DKFocus t else DKUnavailable
      in
      match k with
        | DKUnavailable ->
            find_deref_kind r rem
        | k ->
            k

let rec expr e perms prods =
  let return ?(loc = e.loc) n produces =
    { expr = { loc = loc; node = n };
      produces = produces }
  in
  match e.node with
    | Const _ | Var _ as x ->
        force_production (return x perms) prods
    | Unop (o, a) ->
        let a = expr a perms prods in
        return (Unop (o, a.expr)) a.produces
    | Binop (o, a, b) ->
        let a = expr a perms [] in
        let b = expr b a.produces prods in
        return (Binop (o, a.expr, b.expr)) b.produces
    | Tuple l ->
        let p, l = expr_list l perms in
        force_production (return (Tuple l) p) prods
    | Proj (a, i, c) ->
        let a = expr a perms prods in
        return (Proj (a.expr, i, c)) a.produces
    | Left a ->
        let a = expr a perms prods in
        return (Left a.expr) a.produces
    | Right a ->
        let a = expr a perms prods in
        return (Right a.expr) a.produces
    | Let (x, a, b) ->
        let a = expr a perms [] in
        let b = expr b a.produces prods in
        return (Let (x, a.expr, b.expr)) b.produces
    | Seq (a, b) ->
        let a = expr a perms [] in
        let b = expr b a.produces prods in
        return (Seq (a.expr, b.expr)) b.produces
    | Call (x, l, c, p, subst, _) ->
        let perms, l = expr_list l perms in
        let perms, call = match List.rev l with
          | [] ->
              let j = join e.loc perms c in
              j.produces, (seq j.expr e)
          | last :: r ->
              let last = { expr = last; produces = perms } in
              let last = force_production last c in
              let l = List.rev (last.expr :: r) in
              last.produces, Call (x, l, c, p, subst, perms)
        in
(*        log "@[<hov 2>available:@ %a@]@."
          (Pp.list ~sep: Pp.comma_space Pp.permission) perms;
        log "@[<hov 2>consumed by %a:@ %a@]@."
          Pp.identifier x
          (Pp.list ~sep: Pp.comma_space Pp.permission) c;*)
        force_production (return call (consume_and_produce perms c p)) prods
    | LogicApp (x, l, rs, _) ->
        let p, l = expr_list l perms in
        (* same remark as for Assert *)
        force_production (return (LogicApp (x, l, rs, perms)) p) prods
    | If (a, b, c) ->
        let a = expr a perms [] in
        let b = expr b a.produces prods in
        let c = expr c a.produces prods in
        let j, k, p = join_branches b.produces c.produces in
        let b = { b with expr = { b.expr with node = seqf b.expr j.expr } } in
        let c = { c with expr = { c.expr with node = seqf c.expr k.expr } } in
        return (If (a.expr, b.expr, c.expr)) p
    | While (a, b, i, _) ->
        let a = expr a perms [] in
        let b = expr b a.produces a.produces in
        force_production
          (return (While (a.expr, b.expr, i, a.produces)) b.produces)
          prods
    | Assign (a, b, r) ->
        let a = expr a perms [] in
        let b = expr b a.produces [ POpen r ] in
        force_production (return (Assign (a.expr, b.expr, r)) b.produces) prods
    | Deref (a, r, _, _) ->
        let a = expr a perms prods in
        return
          (Deref (a.expr, r, find_deref_kind r a.produces, a.produces))
          a.produces
    | New ((rl, tl, c), r) ->
        let owns = List.map (fun r -> PEmpty r) (owned_regions r) in
        let j = join e.loc perms [ PEmpty r ] in
        force_production
          (return
             (seq j.expr e)
             (consume_and_produce j.produces [ PEmpty r ] (POpen r :: owns)))
          prods
    | Pack (a, r) ->
        let owns = List.map (fun r -> PGroup r) (owned_regions r) in
        let a = expr a perms (POpen r :: owns) in
        force_production
          (return
             (Pack (a.expr, r))
             (consume_and_produce a.produces (POpen r :: owns) [ PClosed r ]))
          prods
    | Unpack (a, r) ->
        let owns = List.map (fun r -> PGroup r) (owned_regions r) in
        let a = expr a perms [ PClosed r ] in
        force_production
          (return
             (Unpack (a.expr, r))
             (consume_and_produce a.produces [ PClosed r ] (POpen r :: owns)))
          prods
    | Adopt (a, r, s, _) ->
        (* TODO: other kinds of adoption *)
        let a = expr a perms [ PClosed s; PGroup r ] in
        force_production
          (return
             (Adopt (a.expr, r, s, AKSingletonToGroup))
             (consume_and_produce a.produces
                [ PClosed s; PGroup r ]
                [ PGroup r; PSub (s, r) ]))
          prods
    | Focus (a, s, r) ->
        let a = expr a perms [ PEmpty s; PGroup r ] in
        force_production
          (return
             (Focus (a.expr, s, r))
             (consume_and_produce
                a.produces
                [ PEmpty s; PGroup r ]
                [ PClosed s; PArrow (s, r); PSub (s, r) ]))
          prods
    | Unfocus (a, r, s) ->
        let a = expr a perms [ PClosed s; PArrow (s, r) ] in
        force_production
          (return
             (Unfocus (a.expr, r, s))
             (consume_and_produce
                a.produces
                [ PClosed s; PArrow (s, r) ]
                [ PGroup r ]))
          prods
    | Region (r, a, t) ->
        let perms = produce perms (PEmpty (RVar r)) in
        let a = expr a perms prods in
        let perms =
          List.filter
            (fun p -> not (permission_mentions (RVar r) p))
            a.produces
        in
        return (Region (r, a.expr, t)) perms
    | Print (s, a) ->
        let a = expr a perms prods in
        return (Print (s, a.expr)) a.produces
    | AdoptRegion (s, r, _) ->
        (* TODO: other kinds of adoption *)
        let j = join e.loc perms [ PClosed s; PGroup r ] in
        force_production
          (return
             (seq
                j.expr
                { e with node = AdoptRegion (s, r, AKSingletonToGroup) })
             (consume_and_produce j.produces
                [ PClosed s; PGroup r ]
                [ PGroup r; PSub (s, r) ]))
          prods
    | UnfocusRegion (s, r, _) ->
        (* TODO: other kinds of unfocus *)
        let j = join e.loc perms [ PClosed s; PArrow (s, r) ] in
        force_production
          (return
             (seq j.expr e)
             (consume_and_produce
                j.produces
                [ PClosed s; PArrow (s, r) ]
                [ PGroup r ]))
          prods
    | PackRegion r ->
        let owns = List.map (fun r -> PGroup r) (owned_regions r) in
        let j = join e.loc perms [ POpen r ] in
        force_production
          (return
             (seq j.expr e)
             (consume_and_produce j.produces (POpen r :: owns) [ PClosed r ]))
          prods
    | UnpackRegion r ->
        let owns = List.map (fun r -> PGroup r) (owned_regions r) in
        let j = join e.loc perms [ PClosed r ] in
        force_production
          (return
             (seq j.expr e)
             (consume_and_produce j.produces [ PClosed r ] (POpen r :: owns)))
          prods
    | BlackBox (c, p) ->
        let j = join e.loc perms c in
        force_production
          (return
             (seq j.expr e)
             (consume_and_produce j.produces c p))
          prods
    | WeakenRegion _ ->
        assert false (* the user cannot write a weakening annotation *)
    | Typed (a, t) ->
        let a = expr a perms prods in
        return (Typed (a.expr, t)) a.produces
    | Old a ->
        let a = expr a perms prods in
        return (Old a.expr) a.produces        
    | Assert (a, _) ->
        (* we assume the assert is before the force_production, but this
           could be discussed *)
        let x = Assert (a, perms) in
        force_production (return x perms) prods

and expr_list l perms =
  let p, l =
    list_fold_map
      (fun p a -> let a = expr a p [] in (* p *) a.produces, a)
      perms l
  in
  p, List.map (fun x -> x.expr) l

let value_def x =
  match x.v_body with
    | None ->
        Value x
    | Some body ->
        let body = expr body x.v_consumes x.v_produces in
        let x = { x with v_body = Some body.expr } in
        Value x

let def = function
  | Value x -> value_def x
  | Class _
  | LogicType _
  | LogicFunction _
  | Axiom _
  | Predicate _
      as x -> x

let file x =
  List.map def x
