(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

type const =
  | CUnit
  | CInt of int
  | CBool of bool

type region =
  | RVar of string
  | RSub of region * string
  | RUVar of int (* unifiable variable *)

type base_type =
  | TUnit
  | TInt
  | TBool

type typ =
  | TTuple of typ list
  | TPointer of region list * typ list * string * region
  | TVar of string
  | TUVar of int (* unifiable variable *)
  | TBase of base_type
  | TSum of typ * typ

type 'a node = {
  loc: Loc.t;
  node: 'a;
  mutable typ: typ;
}

type type_expr = type_node node

and type_node =
  | TETuple of type_expr list
  | TEPointer of class_expr * region
  | TEIdent of string
  | TESum of type_expr * type_expr

and class_expr = region list * type_expr list * string

type permission =
  | PEmpty of region
  | POpen of region
  | PClosed of region
  | PGroup of region
  | PArrow of region * region

type int_bin_op = [ `add | `sub | `mul | `div | `imod ]
type int_un_op = [ `neg ]
type bool_bin_op = [ `bor | `band | `bxor ]
type bool_un_op = [ `bnot ]

type int_op = [ int_bin_op | int_un_op ]
type bool_op = [ bool_bin_op | bool_un_op ]
type bin_op = [ int_bin_op | bool_bin_op ]
type un_op = [ int_un_op | bool_un_op ]
type op = [ bin_op | un_op ]

type expr = expr_node node

and expr_node =
  | Const of const
  | Unop of un_op * expr
  | Binop of bin_op * expr * expr
  | Tuple of expr list
  | Proj of expr * int
  | Left of expr
  | Right of expr
  | Var of string
  | Let of string * expr * expr
  | Seq of expr * expr
  | Call of string * expr list * (string * region) list ref
  | If of expr * expr * expr
  | While of expr * expr
  | Assign of expr * expr
  | Deref of expr
  | New of class_expr * region
  | Pack of expr
  | Unpack of expr
  | Adopt of expr * region
  | Focus of expr * region
  | FocusBind of expr * string * string * expr
  | Unfocus of expr * region
  | Region of string * expr
  | Print of string * expr

  | AdoptRegion of region * region
  | UnfocusRegion of region * region
  | PackRegion of region
  | UnpackRegion of region

  | BlackBox of permission list * permission list

type class_def = {
  c_region_params: string list;
  c_type_params: string list;
  c_name: string;
  c_owned_regions: string list;
  c_type: type_expr;
  c_self_region: string option;
}

type value_def = {
  v_name: string;
  v_params: (string * type_expr) list;
  v_return_type: type_expr;
  v_consumes: permission list;
  v_produces: permission list;
  v_body: expr option;
}

type def =
  | Class of class_def
  | Value of value_def

type file = def list
