(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Format
open Lexing

module StringSet = Set.Make(String)
module StringMap = Map.Make(String)

let error s =
  ksprintf (fun s -> eprintf "%s\n" s; exit 1) s

let lexbuf_location lexbuf =
  lexeme_start_p lexbuf, lexeme_end_p lexbuf

exception Parse_error of Loc.t * string

let rec list_fold_map f acc env = function
  | [] -> env, List.rev acc
  | x :: r ->
      let env, x = f env x in
      list_fold_map f (x :: acc) env r
let list_fold_map f env l = list_fold_map f [] env l

let rec list_remove acc x = function
  | [] -> raise Not_found
  | y :: rem when x = y -> List.rev_append acc rem
  | y :: rem -> list_remove (y :: acc) x rem
let list_remove x = list_remove [] x

let rec list_removef acc f = function
  | [] -> raise Not_found
  | y :: rem when f y -> List.rev_append acc rem
  | y :: rem -> list_removef (y :: acc) f rem
let list_removef f x = list_removef [] f x

let rec list_findf f = function
  | [] ->
      raise Not_found
  | x :: r ->
      match f x with
        | None ->
            list_findf f r
        | Some y ->
            y

let check loc b x =
  ksprintf (fun s -> if b then () else Loc.locate_error loc "%s" s) x

let (|>) x f = f x

module Opt = struct
  let map f = function
    | None -> None
    | Some x -> Some (f x)

  let fold f x = function
    | None -> x
    | Some y -> f x y
end

let log s = kfprintf (fun fmt -> ()) std_formatter s

let fresh =
  let i = ref (-1) in
  fun () ->
    incr i;
    !i
