(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Ast
open Format

let nothing fmt () = ()
let char c fmt () = fprintf fmt "%c" c
let string s fmt () = fprintf fmt "%s" s
let comma_space fmt () = fprintf fmt ",@ "
let todo s fmt () = fprintf fmt "(TODO: Pp.%s)" s

let list ?(sep = nothing) ?(left = nothing) ?(right = nothing) f fmt = function
  | [] -> ()
  | [x] ->
      left fmt ();
      f fmt x;
      right fmt ();
  | x :: rem ->
      left fmt ();
      f fmt x;
      List.iter
        (fun x ->
           sep fmt ();
           f fmt x)
        rem;
      right fmt ()

let option ?(left = nothing) ?(right = nothing) f fmt = function
  | None -> ()
  | Some x ->
      left fmt ();
      f fmt x;
      right fmt ()

let rec region fmt = function
  | RVar id -> fprintf fmt "%s" id
  | RUVar i -> fprintf fmt "?%d" i
  | RSub (r, id) -> fprintf fmt "%a.%s" region r id

let rec typ fmt = function
  | TVar id ->
      fprintf fmt "%s" id
  | TUVar i ->
      fprintf fmt "?%d" i
  | TTuple tl ->
      fprintf fmt "%a"
        (list ~sep: (char '*') ~left: (char '(') ~right: (char ')') typ) tl
  | TPointer (rl, tl, c, r) ->
      fprintf fmt "%s%a%a[%a]"
        c
        (list ~sep: (char ',') ~left: (char '{') ~right: (char '}') region) rl
        (list ~sep: (char ',') ~left: (char '(') ~right: (char ')') typ) tl
        region r
  | TBase TUnit ->
      fprintf fmt "unit"
  | TBase TInt ->
      fprintf fmt "int"
  | TBase TBool ->
      fprintf fmt "bool"
  | TSum (a, b) ->
      fprintf fmt "(%a | %a)" typ a typ b

let permission fmt = function
  | PEmpty r ->
      fprintf fmt "%a^e" region r
  | POpen r ->
      fprintf fmt "%a^o" region r
  | PClosed r ->
      fprintf fmt "%a^c" region r
  | PGroup r ->
      fprintf fmt "%a^g" region r
  | PArrow (s, r) ->
      fprintf fmt "%a -> %a" region s region r

let region_list_of_permission = function
  | PEmpty r
  | POpen r
  | PClosed r
  | PGroup r ->
      [r]
  | PArrow (s, r) ->
      [r; s]

(* sort by region first *)
let compare_permissions a b =
  let c =
    compare
      (region_list_of_permission a)
      (region_list_of_permission b)
  in
  if c = 0 then
    compare a b
  else
    c

let permissions fmt caps =
  let caps = List.sort compare_permissions caps in
  list ~sep: (string ", ") permission fmt caps

let const fmt = function
  | CUnit ->
      fprintf fmt "()"
  | CInt i ->
      fprintf fmt "%d" i
  | CBool b ->
      fprintf fmt "%B" b

let op fmt o =
  let s = match o with
    | `add -> "+"
    | `sub -> "-"
    | `mul -> "*"
    | `div -> "/"
    | `imod -> "mod"
    | `neg -> "-"
    | `bor -> "or"
    | `band -> "and"
    | `bxor -> "xor"
    | `bnot -> "not"
  in
  fprintf fmt "%s" s

let rec type_expr fmt e =
  let f x = fprintf fmt x in
  match e.node with
    | TETuple l ->
        f "@[<hv 2>(%a)@]" (list ~sep: comma_space type_expr) l
    | TEPointer ((rl, tl, s), r) ->
        f "%s%a%a[%a]"
          s
          (list ~left: (char '(') ~right: (char ')') ~sep: (string ", ")
             type_expr)
          tl
          (list ~left: (char '{') ~right: (char '}') ~sep: (string ", ")
             region)
          rl
          region r
    | TEIdent s ->
        f "%s" s
    | TESum (a, b) ->
        f "@[<hv>(%a@ | %a)@]" type_expr a type_expr b

let rec expr fmt e =
  let f x = fprintf fmt x in
  match e.node with
    | Const c ->
        const fmt c
    | Unop (o, a) ->
        f "(%a %a)" op o expr a
    | Binop (o, a, b) ->
        f "(@[<hv 2>%a@ %a@ %a@])" expr a op o expr b
    | Tuple l ->
        f "(@[<hv 2>%a@])" (list ~sep: comma_space expr) l
    | Proj (a, i) ->
        f "%a.%d" expr a i
    | Left a ->
        f "(@[<hv 2>Left@ %a@])" expr a
    | Right a ->
        f "(@[<hv 2>Right@ %a@])" expr a
    | Var s ->
        f "%s" s
    | Let (s, a, b) ->
        f "(@[<hv>@[<hv>@[<hv 2>let %s =@ %a@]@ in@]@ %a@])" s expr a expr b
    | Seq (a, b) ->
        f "(@[<hv>%a;@ %a@])" expr a expr b
    | Call (s, l, _) ->
        f "(@[<hv 2>%s(%a)@])" s (list ~sep: comma_space expr) l
    | If (a, b, c) ->
        f "(@[<hv>if %a@ then %a@ else %a@])" expr a expr b expr c
    | While (a, b) ->
        f "(@[<hv 2>while %a do@ %a@])" expr a expr b
    | Assign (a, b) ->
        f "(@[<hv 2>%a :=@ %a@])" expr a expr b
    | Deref a ->
        f "(!%a)" expr a
    | New ((rl, tl, s), r) ->
        f "(new %s%a%a[%a])" s
          (list ~sep: comma_space ~left: (char '{') ~right: (char '}') region)
          rl
          (list ~sep: comma_space ~left: (char '(') ~right: (char ')')
             type_expr)
          tl
          region r
    | Pack a ->
        f "(@[<hv 2>pack@ %a@])" expr a
    | Unpack a ->
        f "(@[<hv 2>unpack@ %a@])" expr a
    | Adopt (a, r) ->
        f "(@[<hv 2>adopt@ %a@ in %a@])" expr a region r
    | Focus (a, r) ->
        f "(@[<hv 2>focus@ %a@ in %a@])" expr a region r
    | FocusBind _ ->
        f "%a" (todo "expr.FocusBind") ()
    | Unfocus (a, r) ->
        f "(@[<hv 2>unfocus@ %a@ in %a@])" expr a region r
    | Region (s, e) ->
        f "(@[<hv>region %s in@ %a@])" s expr e
    | Print (s, e) ->
        f "(@[<hv>print \"%s\" %a@])" (String.escaped s) expr e
    | AdoptRegion (a, b) ->
        f "(@[adopt region %a in %a@])" region a region b
    | UnfocusRegion (a, b) ->
        f "(@[unfocus region %a in %a@])" region a region b
    | PackRegion r ->
        f "(@[pack region %a@])" region r
    | UnpackRegion r ->
        f "(@[unpack region %a@])" region r
    | BlackBox (c, p) ->
        f "(@[<hv 2>blackbox@ consumes %a@ produces %a@])"
          (list ~sep: comma_space permission) c
          (list ~sep: comma_space permission) p

let class_def fmt x =
  let f x = fprintf fmt x in
  f "@[<hv 2>class %s%a%a%a =@ %a@ end@]@.@."
    x.c_name
    (list ~left: (char '(') ~sep: comma_space ~right: (char ')')
       pp_print_string)
    x.c_type_params
    (list ~left: (char '{') ~sep: comma_space ~right: (char '}')
       pp_print_string)
    x.c_owned_regions
    (option ~left: (char '[') ~right: (char ']') pp_print_string)
    x.c_self_region
    type_expr x.c_type

let value_def fmt x =
  let f x = fprintf fmt x in
  f "@[<hv 2>val @[<hv 2>%s(%a)@]: %a"
    x.v_name
    (list ~sep: comma_space
       (fun _ (n, t) -> f "%s: %a" n type_expr t))
    x.v_params
    type_expr x.v_return_type;
  begin match x.v_consumes with
    | [] -> ()
    | l -> f "@ consumes %a" (list ~sep: (string ", ") permission) l
  end;
  begin match x.v_produces with
    | [] -> ()
    | l -> f "@ produces %a" (list ~sep: (string ", ") permission) l
  end;
  begin match x.v_body with
    | None -> ()
    | Some b -> f " =@ %a" expr b
  end;
  f "@]@.@."

let def fmt = function
  | Class x -> class_def fmt x
  | Value x -> value_def fmt x

let file fmt f =
  List.iter (def fmt) f
