(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

(** Associate a [class_def] to each region. *)

open Misc
open Ast
open Bast
open Typing_common

let tbl = Hashtbl.create 42

let eq_permission a b =
  Unify.expand_permission a = Unify.expand_permission b

let eq_region a b =
  Unify.expand_region a = Unify.expand_region b

let eq_class_expr (rl1, tl1, c1) (rl2, tl2, c2) =
  let acc = c1 = c2 in
  let acc =
    List.fold_left2
      (fun acc r1 r2 -> acc && eq_region r1 r2)
      acc rl1 rl2
  in
  let acc =
    List.fold_left2
      (fun acc t1 t2 -> acc (* TODO *))
      acc tl1 tl2
  in
  acc

let pp_class_expr fmt (rl, tl, c) =
  Format.fprintf fmt "%a%a%a"
    Pp.identifier c
    (Pp.list ~sep: Pp.comma_space ~left: (Pp.string " {") ~right: (Pp.char '}')
       Pp.region) rl
    (Pp.list ~sep: Pp.comma_space ~left: (Pp.string " [") ~right: (Pp.char '}')
       Pp.type_expr) tl

(** Should be called for each function parameter, allocation and focus. *)
let add (r: Ident.t) c =
  try
    let c' = Hashtbl.find tbl r in
    if not (eq_class_expr c c') then
      log "internal error: region %a contains pointers of class %a, and is now used with pointers of class %a@."
        Pp.identifier r pp_class_expr c' pp_class_expr c
  with Not_found ->
    Hashtbl.add tbl r c

(** Find the pointer type (i.e. class expr) for a given owned region.
    To do this we look in the pointed type (this function should be called
    after applying [pointed_type_of]) for a subterm of a pointer type
    of the searched region. *)
let rec find_subregion_class_expr (find_class: Ident.t -> class_def)
    t s: class_expr option =
  match t.node with
    | TEPointer (ce, RSub (_, r)) when r = s ->
        Some ce
    | TEPointer ((_, l, _), _)
    | TETuple l
    | TELogic (l, _) ->
        List.fold_left
          (fun a b -> Opt.if_none (find_subregion_class_expr find_class b s) a)
          None
          l
    | TEIdent id ->
(*        log "find_subregion_class_expr called with TEIdent %a" Pp.identifier id;
        assert false (* impossible because [get] calls [pointed_type] *)*)
        (* above assert is false, it may be a type variable *)
        None
    | TEBase _ ->
        None
    | TESum (a, b) ->
        Opt.if_none
          (find_subregion_class_expr find_class a s)
          (find_subregion_class_expr find_class b s)

let rec get find_class (r: Bast.region): class_expr =
  let r = Unify.expand_region r in
  match r with
    | RVar id ->
        begin try
          Hashtbl.find tbl id
        with Not_found ->
          log "internal error: don't know which class of pointers region %a is supposed to contain@."
            Pp.identifier id;
          exit 2
        end
    | RSub (r', s) ->
        let rl, tl, c = get find_class r' in
        let t = pointed_type_of rl tl (find_class c) r' in
        match find_subregion_class_expr find_class t s with
          | Some r' ->
              r'
          | None ->
              log "internal error: don't know which class of pointers owned region %a is supposed to contain@."
                Pp.region r;
              exit 2

let add_if_var r c =
  match r with
    | RVar r -> add r c
    | RSub _ -> ()
