(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Misc
open Ast
open Bast
open Asttools

type environment = {
  variables: type_expr Env.t;
  type_variables: unit Env.t;
  values: value_def Env.t; (** Warning: the [v_body] should be ignored. *)
  regions: unit Env.t;
  classes: class_def Env.t;
  logic_types: logic_type_def Env.t;
  logic_functions: logic_function_def Env.t;
  predicates: predicate_def Env.t;
}

let empty_env = {
  variables = Env.empty;
  type_variables = Env.empty;
  values = Env.empty;
  regions = Env.empty;
  classes = Env.empty;
  logic_types = Env.empty;
  logic_functions = Env.empty;
  predicates = Env.empty;
}

let find loc env i =
  try
    Env.find env i
  with Not_found ->
    Loc.locate_error loc "unbound identifier: %s" (Ident.name i)

(* substitution on TVar *)
let rec subst_type id t u =
  match u.node with
    | TEIdent v when id = v ->
        t
    | TEBase _ | TEIdent _ ->
        u
    | TETuple tl ->
        { u with node = TETuple (List.map (subst_type id t) tl) }
    | TESum (a, b) ->
        { u with node = TESum (subst_type id t a, subst_type id t b) }
    | TEPointer ((rl, tl, c), r) ->
        { u with node = TEPointer ((rl, List.map (subst_type id t) tl, c), r) }
    | TELogic (tl, x) ->
        { u with node = TELogic (List.map (subst_type id t) tl, x) }

(* substitution on RVar *)
let rec subst_regreg id r reg =
  match Unify.expand_region reg with
    | RVar v when id = v ->
        r
    | RVar _ as x -> x
    | RSub (a, b) ->
        RSub (subst_regreg id r a, b)

(* apply a full substitution to a single region *)
let subst_regreg_list s r =
  List.fold_left (fun r (id, s) -> subst_regreg id s r) r s

(* substitution on RVar *)
let rec subst_region id r t =
  let node = match t.node with
    | TEBase _ | TEIdent _ as x ->
        x
    | TETuple tl ->
        TETuple (List.map (subst_region id r) tl)
    | TESum (a, b) ->
        TESum (subst_region id r a, subst_region id r b)
    | TEPointer ((rl, tl, c), r') ->
        TEPointer (
          (List.map (subst_regreg id r) rl,
           List.map (subst_region id r) tl,
           c),
          subst_regreg id r r'
        )
    | TELogic (tl, x) ->
        TELogic (List.map (subst_region id r) tl, x)
  in
  { t with node = node }

let pointed_type_of rl tl c r =
  let t =
    List.fold_left2
      (fun acc id t -> subst_type id t acc)
      c.c_type
      c.c_type_params
      tl
  in
  let t =
    List.fold_left2
      (fun acc id r -> subst_region id r acc)
      t
      c.c_region_params
      rl
  in
  let t =
    List.fold_left
      (fun acc id -> subst_region id (RSub (r, Ident.name id)) acc)
      t
      c.c_owned_regions
  in
  t

let pointed_type env t =
  match t.node with
    | TEPointer ((rl, tl, c), r) ->
        let c = find t.loc env.classes c in
        pointed_type_of rl tl c r
    | _ ->
        Loc.locate_error t.loc "this should be a pointer"

module Region = struct
  type t = Bast.region
  let compare a b =
    compare (Unify.expand_region a) (Unify.expand_region b)
end
module RegionSet = Set.Make(Region)

let region_set_of_list s =
  List.fold_right RegionSet.add s RegionSet.empty

let rec root_of_region r =
  match Unify.expand_region r with
    | RVar r -> r
    | RSub (r, _) -> root_of_region r

let region_of_permission = function
  | PEmpty r | POpen r | PClosed r | PGroup r | PArrow (_, r) -> [ r ]
  | PSub _ -> []
