(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Ast
open Bast

let make_node ?(loc = Loc.dummy_location) n = { loc = loc; node = n }
let ttuple ?loc l = make_node ?loc (TETuple l)
let tpointer ?loc rl tl c r = make_node ?loc (TEPointer ((rl, tl, c), r))
let tident ?loc i = make_node ?loc (TEIdent i)
let tbase ?loc b = make_node ?loc (TEBase b)
let tsum ?loc a b = make_node ?loc (TESum (a, b))

(** Sequence node of two [expr]s. May do some simplifications. *)
let seq a b =
  match a.node, b.node with
    | Const CUnit, x | x, Const CUnit -> x
    | _ -> Seq (a, b)

(** Sequence node of two [expr]s. May do some simplifications.
    The first expression is returned, not the second one. *)
let seqf =
  let c = ref 0 in fun a b ->
    match a.node with
      | Const _ | Var _ ->
          (* no effect, including on permissions, so we might as well swap *)
          seq b a
      | Assign _ ->
          (* necessarily of unit type, so the let is not needed *)
          seq a b
      | _ ->
          match b.node with
            | Const CUnit -> a.node
            | _ ->
                incr c;
                let v = Ident.create ("_seqf_" ^ string_of_int !c) in
                let seq = make_node ~loc: b.loc (Seq (b, make_node (Var v))) in
                Let (v, a, seq)

(*
let rec iter_expr f e =
  f e;
  match e.node with
    | Const _
    | Var _
    | New _
    | AdoptRegion _
    | UnfocusRegion _
    | PackRegion _
    | UnpackRegion _
    | WeakenRegion _
    | BlackBox _
    | Assert _
      -> ()
    | Unop (_, a)
    | Proj (a, _, _)
    | Left a
    | Right a
    | Deref (a, _, _)
    | Pack (a, _)
    | Unpack (a, _)
    | Adopt (a, _, _, _)
    | Focus (a, _, _)
    | Unfocus (a, _, _)
    | Region (_, a, _)
    | Print (_, a)
    | Typed (a, _)
    | Old a
      -> iter_expr f a
    | Binop (_, a, b)
    | Let (_, a, b)
    | Seq (a, b)
    | While (a, b, _, _)
    | Assign (a, b, _)
      -> iter_expr f a; iter_expr f b
    | If (a, b, c)
      -> iter_expr f a; iter_expr f b; iter_expr f c
    | Tuple l
    | Call (_, l, _, _, _)
    | LogicApp (_, l, _, _)
      -> List.iter (iter_expr f) l

let fold_expr f acc e =
  let r = ref acc in
  iter_expr (fun e -> r := f acc e) e;
  !r
*)
