(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

(* Compute the set of regions which are read or written. *)

open Misc
open Ast
open Bast
open Typing_common (* for RegionSet and subst_regreg_list *)

let rec term_reads ?(acc = RegionSet.empty) t =
  let tr ?(acc = acc) t = term_reads ~acc t in
  match t.node with
    | Var _
    | Const _ ->
        acc
    | Old a
    | Typed (a, _)
    | Proj (a, _, _)
    | Left a
    | Right a
    | Unop (_, a) ->
        tr a
    | Binop (_, a, b)
    | Let (_, a, b) ->
        let acc = tr a in
        tr ~acc b
    | Tuple l ->
        List.fold_left (fun acc a -> tr ~acc a) acc l
    | LogicApp (_, l, r, _) ->
        let acc = List.fold_right RegionSet.add r acc in
        List.fold_left (fun acc a -> tr ~acc a) acc l
    | If (a, b, c) ->
        let acc = tr a in
        let acc = tr ~acc b in
        tr ~acc c
    | Deref (a, r, _, _) ->
        let acc = tr a in
        RegionSet.add r acc
    | Assert (a, _) ->
        assertion_reads ~acc a

    | Call _ ->
        Loc.locate_error t.loc "calls are not implemented in the logic"

    | Seq _ | While _ | Assign _ | New _ | Pack _ | Unpack _ | Adopt _ | Focus _
    | Unfocus _ | Region _ | Print _ | AdoptRegion _ | UnfocusRegion _
    | PackRegion _ | UnpackRegion _ | BlackBox _ | WeakenRegion _ ->
        Loc.locate_error t.loc "this operation is not usable in the logic"

and assertion_reads ?(acc = RegionSet.empty) a =
  match a.node with
    | PTrue
    | PFalse ->
        acc
    | PIff (a, b)
    | PImplies (a, b)
    | PAnd (a, b)
    | POr (a, b) ->
        let acc = assertion_reads ~acc a in
        assertion_reads ~acc b
    | POld a
    | PNot a ->
        assertion_reads ~acc a
    | PEqual (a, b)
    | PDiff (a, b)
    | PLt (a, b)
    | PGt (a, b)
    | PLe (a, b)
    | PGe (a, b) ->
        let acc = term_reads ~acc a in
        term_reads ~acc b
    | PTerm a ->
        term_reads ~acc a
    | PForall (id, t, a)
    | PExists (id, t, a) ->
        assertion_reads ~acc a
    | PApp (id, tl, rs) ->
        let acc = List.fold_right RegionSet.add rs acc in
        List.fold_left (fun acc t -> term_reads ~acc t) acc tl

(* WARNING: this does not return bound regions *)
let rec expr_writes ?(bound = IdentSet.empty) ?(acc = RegionSet.empty)
    writes_of_function_id e =
  let writes ?(bound = bound) acc e =
    expr_writes ~bound ~acc writes_of_function_id e
  in
  let add acc r =
(*    log "@ add %a (root = %a) (bound = %a)" Pp.region r Pp.identifier (root_of_region r) (Pp.list ~sep: Pp.comma_space Pp.identifier) (IdentSet.elements bound);*)
    if IdentSet.mem (root_of_region r) bound then
      ((*log "@ YES"; *)acc)
    else
      ((*log "@ NO"; *)RegionSet.add r acc)
  in
  match e.node with
    | Const _
    | Var _
    | BlackBox (_, _)
    | Assert (_, _)
    | PackRegion _
    | UnpackRegion _
      -> acc

    | Unop (_, a)
    | Proj (a, _, _)
    | Left a
    | Right a
    | Deref (a, _, _, _)
    | Typed (a, _)
    | Old a
    | Print (_, a)
    | Pack (a, _)
    | Unpack (a, _)
      -> writes acc a

    | Binop (_, a, b)
    | Let (_, a, b)
    | Seq (a, b)
    | While (a, b, _, _)
      -> writes (writes acc a) b

    | If (a, b, c)
      -> writes (writes (writes acc a) b) c

    | Tuple l
    | LogicApp (_, l, _, _)
      -> List.fold_left (fun x -> writes x) acc l

    | New (_, r)
    | WeakenRegion (r, _)
      -> add acc r

    | AdoptRegion (r, s, _)
    | UnfocusRegion (r, s, _)
      -> add (add acc r) s

    | Adopt (a, r, s, _)
    | Focus (a, r, s)
    | Unfocus (a, r, s)
      -> writes (add (add acc s) r) a

    | Region (rid, a, _)
      -> writes ~bound: (IdentSet.add rid bound) acc a

    | Assign (a, b, r)
      -> writes (writes (add acc r) a) b

    | Call (function_id, l, _, _, subst, _) ->
        log "CALL %a: effects before subst: %a@."
          Pp.identifier function_id
          (Pp.list ~sep: Pp.comma_space Pp.region)
          (RegionSet.elements (writes_of_function_id function_id));
        log "substitution: %a@."
          (Pp.list ~sep: Pp.comma_space
             (fun fmt (a, b) ->
                Format.fprintf fmt "%a -> %a" Pp.identifier a Pp.region b))
          subst;
        let function_effects =
          List.map
            (subst_regreg_list subst)
            (RegionSet.elements (writes_of_function_id function_id))
        in
        log "effects of %a(%a) after substitution: %a@.@."
          Pp.identifier function_id
          (Pp.list ~sep: Pp.comma_space Pp.expr) l
          (Pp.list ~sep: Pp.comma_space Pp.region) function_effects;
        let acc = List.fold_left add acc function_effects in
        List.fold_left writes acc l

let value_writes: (Ident.t, RegionSet.t) Hashtbl.t = Hashtbl.create 16

let find_in_value_writes fid =
  try
    Hashtbl.find value_writes fid
  with Not_found ->
    RegionSet.empty

(* Take a list of mutually recursive values, compute their write effects, and
   enter them in the [value_effects] table. *)
let fixpoint values =
  log "Computing effect fixpoint for %d value(s)...@." (List.length values);
  let changed = ref true in
  let pass = ref 0 in
  while !changed do
    changed := false;
    incr pass;
    log "* Pass %d@." !pass;
    List.iter
      (fun v ->
         log "  * Value %a...@?" Pp.identifier v.v_name;
         let old_effects = find_in_value_writes v.v_name in
         let new_effects = match v.v_body with
           | None ->
               region_set_of_list
                 (List.flatten (List.map region_of_permission v.v_consumes))
           | Some body ->
               expr_writes find_in_value_writes body
         in
         Hashtbl.replace value_writes v.v_name new_effects;
         let value_changed = not (RegionSet.equal old_effects new_effects) in
         if value_changed then
           log " changed.@."
         else
           log " unchanged.@.";
         changed := !changed || value_changed)
      values
  done;
  log "Done.@."

let writes_of_value v =
  let fid = v.v_name in
  try
    RegionSet.elements (Hashtbl.find value_writes fid)
  with Not_found ->
    error "effects.ml: writes_of_value: The writes effects for value %a have not been computed yet." Pp.identifier fid
