(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

(** Why interpretation of Capucine programs. *)

(*
How regions are translated:
  There are 3 references (R when group, V for value and P for pointer when
  singleton) which may actually sometimes not be references. Moreover,
  the fact that there is a region tree and region substitutions when
  calling complicates stuff a bit.

It depends on how the region is bound.
  1) The region is bound using the region ... in operation. The three references
     are created.
  2) The region is bound at the level of the function.
    a) No parent* of the region has a positive permission
       (read-only: only a group version of the region, with no reference).
    b) A parent* has a positive permission.
       (read-write: all three references are created)
  3) In logic predicates (predicate definitions, axioms and invariants),
     all regions are considered group read-only regions.

So, when calling a predicate, one must find a group region (not a reference).
Depending on how the region is bound you may either already have it,
already have it but under a reference, or you have to build it from the
singleton references, or if the region is being focused you have to
use an if-then-else.

Finally, when calling a function, one first have to find out whether the
region is read-only or not in the called function. If read-only, then its
as if calling a predicate, and if not, then the three references are
available anyway in the caller.

Finding out whether a region is read-only has a trap because of the region
tree: if A is read-only, A.B may not be. To find out whether R.r is read-only,
first look for permissions on R.r, then for permissions on R, and so on.
When calling a function, these have to be looked both in the callee and
in the caller.

Another trap when checking whether a region is read-only appears with
function calls where some regions are unified. If a non-read-only region A
and a read-only region B are unified, then something bad happens. This
actually may happen in real programs if B is later unified with A.B or
something (see an older version of subobs).
*)

open Typing_common
open Ast
open Bast
open Misc
open Why
open Whylib

let classes_environment: class_def Env.t ref = ref Env.empty
let values_environment: value_def Env.t ref = ref Env.empty
let find_class c = Env.find !classes_environment c
let find_value c = Env.find !values_environment c
let class_expr_of_region r = Regiontypes.get find_class r
let class_of_region r =
  let _, _, c = class_expr_of_region r in
  find_class c

let dummy_region = RVar (Ident.create "dummy_region_for_interp")

(** Translate a type expression into a Why logic type. *)
let rec translate_type t =
  match t.node with
    | TETuple l ->
        tuple (List.map translate_type l)
    | TEPointer _ ->
        pointer
    | TEIdent x ->
        type_variable x
    | TEBase TUnit ->
        unit
    | TEBase TInt ->
        int
    | TEBase TBool ->
        bool
    | TEBase TString ->
        string
    | TESum (a, b) ->
        either (translate_type a) (translate_type b)
    | TELogic (tl, id) ->
        logic_type ~args: (List.map translate_type tl) id

and pointed_type tl c =
  (* we use dummy values for regions, as the translation does not care *)
  let t =
    pointed_type_of
      (List.map (fun _ -> dummy_region) c.c_region_params)
      tl c dummy_region
  in
  translate_type t

let pointed_type_of_region r =
  let _, tl, c = class_expr_of_region r in
  pointed_type tl (find_class c)

let rec region_name = function
  | RVar id -> Ident.name id
  | RSub (r, s) -> region_name r ^ "_" ^ s

let rec set_of_region_tree ?(acc = RegionSet.empty) r =
  if RegionSet.mem r acc then
    error "you have a cycle in the region tree of region %a"
      Pp.region r;
  List.fold_left
    (fun acc id -> set_of_region_tree ~acc (RSub (r, Ident.name id)))
    (RegionSet.add r acc)
    (List.map fst (class_of_region r).c_owned_regions)

let list_of_region_tree r =
  let set = set_of_region_tree r in
  let l = RegionSet.elements set in
  List.sort Region.compare l

(* This function is used to determine whether [region] is read-only according
   to permissions [perms]. A region is not read-only if it or one of its
   transitive owner has a permission on it. *)
let is_region_root_positive_in perms region =
  let rec an_owner_is r o =
    Infer.eq_region r o ||
      match o with
        | RVar _ -> false
        | RSub (o', _) -> an_owner_is r o'
  in
  List.exists
    (function
       | PSub _ ->
           false
       | PEmpty r | PArrow (_, r) | PGroup r | POpen r | PClosed r ->
           let r = Unify.expand_region r in
           let region = Unify.expand_region region in
           (* return whether an owner of region is r *)
           an_owner_is r region)
(*           root_of_region (Unify.expand_region r)
           = root_of_region (Unify.expand_region region))*)
    perms

(* for debugging only *)
let is_region_root_positive_in p r =
  let res = is_region_root_positive_in p r in
  log "is_region_root_positive_in %a %a = %b@."
    (Pp.list ~sep: Pp.comma_space Pp.permission) p
    Pp.region r
    res;
  res

let is_singleton_in perms region =
  List.exists
    (function
       | PEmpty _ | PArrow _ | PSub _ | PGroup _ ->
           false
       | POpen r | PClosed r ->
           Unify.expand_region r = Unify.expand_region region)
    perms

type region_variable = G | P | V

(* (deref_of_perms perms) computes whether a region should be dereferenced
   to read into it. It is used in Deref expressions and terms. *)
let deref_of_perms = is_region_root_positive_in
let always_deref _ = true
let never_deref _ = false

module INeedToHideNonGeneralizableStuff = struct
  let region_singleton_pointer_id r = Ident.create (region_name r ^ "_p")
  let region_singleton_pointer_id = memoize region_singleton_pointer_id
  let region_singleton_pointer_id r =
    region_singleton_pointer_id (Unify.expand_region r)

  let region_singleton_value_id r = Ident.create (region_name r ^ "_s")
  let region_singleton_value_id = memoize region_singleton_value_id
  let region_singleton_value_id r =
    region_singleton_value_id (Unify.expand_region r)

  let region_group_id r = Ident.create (region_name r ^ "_g")
  let region_group_id = memoize region_group_id
  let region_group_id r = region_group_id (Unify.expand_region r)

  let region_var r which =
    match which with
      | G -> region_group_id r
      | P -> region_singleton_pointer_id r
      | V -> region_singleton_value_id r

  let read_region context ?at deref r which =
    let read ?at =
      if deref r then
        context.deref ?at
      else
        (assert (at = None); context.var)
    in
    let var = region_var r which in
    read ?at var
end

let region_var = INeedToHideNonGeneralizableStuff.region_var
let read_region = INeedToHideNonGeneralizableStuff.read_region

let region_singleton_pointer_type =
  type_ref (type_base pointer)

let region_singleton_value_type r =
  type_ref (type_base (pointed_type_of_region r))

let region_group_logic_type r =
  region (pointed_type_of_region r)

let region_group_type r =
  type_ref (type_base (region_group_logic_type r))

(* used everywhere when needed to represent the three references *)
let region_references_of_region r =
  [
    region_var r G, region_group_type r;
    region_var r P, region_singleton_pointer_type;
    region_var r V, region_singleton_value_type r;
  ]

(* used by axioms, predicates, logic functions
   ==> we assume the region is group
   so this cannot be used for application
   (see logic_app_region_arguments and logic_region_arguments_of_region_app) *)
let logic_region_arguments_of_region r =
  [ region_var r G, region (pointed_type_of_region r) ]

let logic_region_arguments_of_region_app (context: 'a context) ?at perms r =
  let read_region: region_variable -> 'a =
    read_region ?at context (deref_of_perms perms) r in
  if is_singleton_in perms r then
    context.set
      context.empty
      (read_region P)
      (read_region V)
  else
    read_region G

let call_regions x =
  List.flatten
    (List.map (fun rid -> list_of_region_tree (RVar rid)) x.v_regions)

let logic_app_regions x =
  List.flatten
    (List.map (fun rid -> list_of_region_tree (RVar rid)) x)

(* rs is given by the third argument of LogicApp *)
(* used to translate LogicApp (term apps) and PApp (predicate apps) *)
(* permissions are given to know whether the region is singleton, in which
   case we put it in a group *)
let logic_app_region_arguments context ?at perms rs =
  List.map list_of_region_tree rs
  |> List.flatten
  |> List.map (logic_region_arguments_of_region_app context ?at perms)

let region_references_of_region_tree r =
  list_of_region_tree (RVar r)
  |> List.map region_references_of_region
  |> List.flatten

(* for value declarations *)
let region_arguments_of_region perms r =
  if is_region_root_positive_in perms r then
    region_references_of_region r
  else
    [ region_var r G, type_base (region_group_logic_type r) ]

(* when calling values (must be compatible with the above function) *)
let region_arguments_of_region_app ~perms ~consumes r =
  log "@[<hv 2>REGION_ARGUMENTS_OF_REGION_APP@ @[<hv 2>perms =@ %a@]@ @[<hv 2>consumes =@ %a@]@ @[r = %a@]@]@."
    (Pp.list ~sep: Pp.comma_space Pp.permission) perms
    (Pp.list ~sep: Pp.comma_space Pp.permission) consumes
    Pp.region r;
  if is_region_root_positive_in consumes r then
    (log "==> I'll use the three references because the callee needs so.@.";
     List.map var (List.map fst (region_references_of_region r)))
  else begin
    match Infer.find_deref_kind r perms with
      | DKSingleton ->
          log "==> I'll construct a non-ref map from a singleton.@.";
          [ expr_set
              expr_empty
              (read_region in_expr always_deref r P)
              (read_region in_expr always_deref r V) ]
      | DKUnavailable ->
          log "==> No permission is available, but maybe an owner is?@.";
          if is_region_root_positive_in perms r then begin
            log "==> Found an owner: region is not read-only yet, \
I'll deref the map.@.";
            [ read_region in_expr always_deref r G ]
          end else begin
            log "==> No owner found: region is already read-only, \
I'll use the non-ref map.@.";
            [ read_region in_expr never_deref r G ]
          end
      | DKGroup ->
          log "==> Region is group, I'll convert the ref into a read-only.@.";
          [ read_region in_expr always_deref r G ]
      | DKFocus s ->
          (* TODO *)
          assert false
  end

let region_arguments_of_value x =
  call_regions x
  |> List.map (region_arguments_of_region x.v_consumes)
  |> List.flatten

let region_arguments_of_logic x =
  logic_app_regions x.lf_region_variables
  |> List.map logic_region_arguments_of_region
  |> List.flatten

let region_arguments_of_predicate x =
  logic_app_regions x.p_region_variables
  |> List.map logic_region_arguments_of_region
  |> List.flatten

let region_arguments_of_axiom x =
  logic_app_regions x.ax_region_variables
  |> List.map logic_region_arguments_of_region
  |> List.flatten

let normal_arguments_of_value x =
  List.map (fun (id, t) -> id, type_base (translate_type t)) x.v_params

let arguments_of_value x =
  region_arguments_of_value x @ normal_arguments_of_value x

let rec reroot_region from_root to_root r =
  if Region.compare r from_root = 0 then
    to_root
  else match r with
    | RVar _ ->
        raise (Invalid_argument "reroot_region")
    | RSub (r', s) ->
        RSub (reroot_region from_root to_root r', s)

(** Return a list of [(s0, r0)] where [s0] and [r0] are equivalent nodes
    of their respective trees. The couple [(s, r)] is not included. *)
let prepare_tree_copy s r =
  let r_tree = list_of_region_tree r in
  let r_tree = List.filter (fun t -> Region.compare t r <> 0) r_tree in
  let s_tree = List.map (reroot_region r s) r_tree in
  List.combine s_tree r_tree

(** Copy the tree of [s] into the tree of [r]. Root nodes are not copied. *)
let copy_region_trees s r =
  seq
    (List.map
       (fun (s, r) ->
          assign
            (region_var r G)
            (read_region in_expr always_deref s G))
       (prepare_tree_copy s r))

(** Copy the tree of [s] into the tree of [r]. Root nodes are not copied.
    Nodes of [s] are added to the nodes of [r] using disjoint set unions. *)
let append_region_trees s r =
  seq
    (List.map
       (fun (s, r) ->
          assign
            (region_var r G)
            (expr_disjoint_union
               (read_region in_expr always_deref s G)
               (read_region in_expr always_deref r G)))
       (prepare_tree_copy s r))

let invariant_id c =
  Ident.create (Ident.name c.c_name ^ "_inv")
let invariant_id = memoize invariant_id

let rec reroot_invariant_region root r =
  match Unify.expand_region r with
    | RVar x ->
        RSub (root, Ident.name x)
    | RSub (r, s) ->
        RSub (reroot_invariant_region root r, s)

(** Assertion of the invariant of class [c] for value [x] of region [root]. *)
let apply_invariant_predicate c x root =
  let _, _, regs = c.c_invariant in
  let regs =
    List.map
      (fun s ->
         let s = reroot_invariant_region root s in
         read_region in_term always_deref s G)
      regs
  in
  a_app
    (invariant_id c)
    (regs @ [ x ])

(** Assertion of all the invariants of region [r]. *)
let apply_invariant_predicate_group r =
  let c = class_of_region r in
  let p = Ident.create "p" in
  a_forall
    p
    pointer
    (apply_invariant_predicate
       c
       (get (read_region in_term always_deref r G) (term_var p))
       r)

(** Assertions of the invariants of the subtree of [r], not including [r]. *)
let apply_invariant_predicate_subtree r =
  let r_tree = list_of_region_tree r in
  let r_tree = List.filter (fun t -> Region.compare t r <> 0) r_tree in
  List.map apply_invariant_predicate_group r_tree

(** If the region is singleton, call [apply_invariant_predicate], else
    call [apply_invariant_predicate_group], with the right arguments.
    If the permission is not closed, then do nothing (return true). *)
let apply_invariant_predicate_perm = function
  | PEmpty _
  | POpen _
  | PSub _
  | PArrow _ -> (* TODO: do something for PArrow *)
      [ a_true ]
  | PClosed r ->
      apply_invariant_predicate
        (class_of_region r)
        (read_region in_term always_deref r V)
        r
      :: apply_invariant_predicate_subtree r
  | PGroup r ->
      apply_invariant_predicate_group r
      :: apply_invariant_predicate_subtree r

let assume_permissions l =
  List.map apply_invariant_predicate_perm l
  |> List.flatten
  |> List.map assume
  |> seq

let find_pointers_of_region r args =
  list_filterf
    (function
       | id, { node = TEPointer (_, s) } when Region.compare r s = 0 ->
           Some id
       | _ ->
           None)
    args

let pointers_if_singleton args = function
  | PEmpty _ | PArrow _ | PSub _ | PGroup _ ->
      None
  | POpen r | PClosed r ->
      try
        Some (find_pointers_of_region r args, r)
      with Not_found ->
        None

let assume_singletons args perms =
  list_filterf (pointers_if_singleton args) perms
  |> List.map (fun (pl, r) -> List.map (fun p -> p, r) pl)
  |> List.flatten
  |> List.map
      (fun (p, r) ->
         assume (a_eq (read_region in_term always_deref r P) (term_var p)))
  |> seq

(* In cases other than Deref, there is no need to replace always_deref
   with (deref_of_perms perms) as permission typing ensures we do have a
   permission on the region and we actually need a deref. *)
let rec expr x =
  match x.node with
    | Const CUnit ->
        const_void
    | Const (CInt i) ->
        const_int i
    | Const (CBool b) ->
        const_bool b
    | Unop (`bnot, a) ->
        expr_not (expr a)
    | Unop (`neg, a) ->
        expr_neg (expr a)
    | Binop (`band, a, b) ->
        expr_and (expr a) (expr b)
    | Binop (`bor, a, b) ->
        expr_or (expr a) (expr b)
    | Binop (`bxor, a, b) ->
        expr_xor (expr a) (expr b)
    | Binop (`add, a, b) ->
        expr_add (expr a) (expr b)
    | Binop (`sub, a, b) ->
        expr_sub (expr a) (expr b)
    | Binop (`mul, a, b) ->
        expr_mul (expr a) (expr b)
    | Binop (`div, a, b) ->
        expr_div (expr a) (expr b)
    | Binop (`imod, a, b) ->
        expr_mod (expr a) (expr b)
    | Binop (`lt, a, b) ->
        expr_lt (expr a) (expr b)
    | Binop (`gt, a, b) ->
        expr_gt (expr a) (expr b)
    | Binop (`le, a, b) ->
        expr_le (expr a) (expr b)
    | Binop (`ge, a, b) ->
        expr_ge (expr a) (expr b)
    | Binop (`eq, a, b) ->
        expr_eq (expr a) (expr b)
    | Tuple l ->
        expr_tuple (List.map expr l)
    | LogicApp (x, l, rs, perms) ->
        let regions = logic_app_region_arguments in_expr perms rs in
        app_list (var x) (regions @ List.map expr l)
    | Proj (e, i, n) ->
        expr_proj n (i - 1) (expr e)
    | Left a ->
        expr_left (expr a)
    | Right a ->
        expr_right (expr a)
    | Var v ->
        var v
    | Let (v, a, b) ->
        expr_let v (expr a) (expr b)
    | Seq (a, b) ->
        seq2 (expr a) (expr b)
    | If (a, b, c) ->
        expr_if (expr a) (expr b) (expr c)
    | While (a, b, i, perms) ->
        loop ~invariant: 
          (assertion perms (Some at_init) None i) (expr a) (expr b)
    | Assign (a, b, r) ->
        expr_ignore
          (expr a)
          (assign (region_var r V) (expr b))
          (*    | Deref (a, r, DKUnavailable) ->
                expr_ignore
                (expr a)
                (black_box (type_base (pointed_type_of_region r)))*)
    | Deref (a, r, DKSingleton, perms) ->
        expr_ignore
          (expr a)
          (read_region in_expr (deref_of_perms perms) r V)
    | Deref (a, r, (DKUnavailable | DKGroup), perms) ->
        let x = Ident.create "p" in
        expr_let
          x
          (expr a)
          (expr_get (read_region in_expr (deref_of_perms perms) r G) (var x))
    | Deref (a, r, DKFocus s, perms) ->
        let x = Ident.create "p" in
        let result_is x = a_eq term_result x in
        let read_region = read_region in_term (deref_of_perms perms) in
        let post =
          a_if
            (term_eq (term_var x) (read_region s P))
            (result_is (read_region s V))
            (result_is (get (read_region r G) (term_var x)))
        in
        expr_let
          x
          (expr a)
          (black_box
             (type_annot
                ~post
                (type_base (pointed_type_of_region r))))
    | New (_, r) ->
        read_region in_expr always_deref r P
    | Pack (_, s)
    | PackRegion s ->
        let _, a, _ = (class_of_region s).c_invariant in
        if a.node = PTrue then
          const_void
        else
          expr_assert
            (apply_invariant_predicate
               (class_of_region s)
               (read_region in_term always_deref s V)
               s)
    | Unpack _
    | UnpackRegion _ ->
        const_void
    | Adopt (a, r, s, ak) ->
        seq2
          (expr a)
          (expr { x with node = AdoptRegion (s, r, ak) })
    | Focus (a, s, r) ->
        (* TODO: use function copy_region_trees *)
        let r_tree = list_of_region_tree r in
        let r_tree =
          List.filter (function RSub _ -> true | RVar _ -> false) r_tree in
        let s_tree = List.map (reroot_region r s) r_tree in
        let id = Ident.create "p" in
        expr_let
          id
          (expr a)
          (seq [
             assign (region_var s P) (var id);
             assign
               (region_var s V)
               (expr_get (read_region in_expr always_deref r G) (var id));
             seq
               (List.map2
                  (fun s r ->
                     assign
                       (region_var s G)
                       (read_region in_expr always_deref r G))
                  s_tree
                  r_tree);
             var id;
           ])
    | Unfocus (a, r, s) ->
        seq2
          (expr a)
          (expr { x with node = UnfocusRegion (s, r, UKClosed) })
    | Region (rid, a, _) ->
        List.fold_left
          (fun acc (id, ty) -> expr_let id (black_box ty) acc)
          (expr a)
          (region_references_of_region_tree rid)
    | Print (_, a) ->
        expr a
    | AdoptRegion (s, r, AKSingletonToEmpty) ->
        seq [
          assign
            (region_var r P)
            (read_region in_expr always_deref s P);
          assign
            (region_var r V)
            (read_region in_expr always_deref s V);
          copy_region_trees s r;
        ]
    | AdoptRegion (s, r, AKSingletonToGroup) ->
        let lbl = Ident.create "before_adopt" in
        let new_g_value =
          a_eq
            (read_region in_term always_deref r G)
            (set
               (read_region ~at: lbl in_term always_deref r G)
               (read_region in_term always_deref s P)
               (read_region in_term always_deref s V))
        in
        let new_g_extends =
          (* La version commentée semble aider les prouveurs, mais la version
             utilisée donne plus d'information (on pourrait mettre les deux
             mais ça ferait beaucoup d'hypothèses) *)
(*          extends
            (read_region in_term always_deref r G)
            (read_region ~at: lbl in_term always_deref r G)*)
          a_not
            (in_region
               (read_region in_term always_deref s P)
               (read_region ~at: lbl in_term always_deref r G))
        in
        let p = Ident.create "p" in
        let pointer_is_different =
          a_forall p pointer
            (a_implies
               (in_region (term_var p) (read_region ~at: lbl in_term always_deref r G))
               (a_neq (term_var p) (read_region in_term always_deref s P)))
        in
        seq [
(*          assign
            (region_var r G)
            (expr_set
               (read_region in_expr always_deref r G)
               (read_region in_expr always_deref s P)
               (read_region in_expr always_deref s V)); *)
          label lbl
            (black_box
               (type_annot
                  ~writes: [ region_var r G ]
                  ~post: (a_and (a_and new_g_value new_g_extends) pointer_is_different)
                  (type_base unit)));
          (* TODO: faire les extends aussi pour les sous-regions et dans les
             autres cas d'adoption (mais pas tous !) *)
          append_region_trees s r;
        ]
    | AdoptRegion (s, r, AKGroupToGroup) ->
        seq [
          assign
            (region_var r G)
            (expr_disjoint_union
               (read_region in_expr always_deref r G)
               (read_region in_expr always_deref s G));
          append_region_trees s r;
        ]
    | AdoptRegion (s, r, AKSingletonToFocus t) ->
        seq [
          assign
            (region_var r G)
            (expr_set
               (read_region in_expr always_deref r G)
               (read_region in_expr always_deref s P)
               (read_region in_expr always_deref s V));
          append_region_trees s t;
        ]
    | AdoptRegion (s, r, AKGroupToFocus t) ->
        (* cannot factorize with AKGroupToGroup because the append is into t *)
        seq [
          assign
            (region_var r G)
            (expr_disjoint_union
               (read_region in_expr always_deref r G)
               (read_region in_expr always_deref s G));
          append_region_trees s t;
        ]
    | UnfocusRegion (s, r, _) ->
        (* cannot factorize with AKSingletonToFocus because the append is
           into t *)
        (* TODO: cas d'un unfocus de groupe ? *)
        seq [
          (* TODO: commenter ce assume dans ma thèse *)
          assume
            (in_region
               (read_region in_term always_deref s P)
               (read_region in_term always_deref r G));
          assign
            (region_var r G)
            (expr_set
               (read_region in_expr always_deref r G)
               (read_region in_expr always_deref s P)
               (read_region in_expr always_deref s V));
          copy_region_trees s r;
        ]
    | BlackBox _ ->
        const_void
    | WeakenRegion (_, WKEmptyToGroup) ->
        const_void
    | WeakenRegion (r, WKClosedToGroup) ->
        assign
          (region_var r G)
          (expr_set
             (read_region in_expr always_deref r G)
             (read_region in_expr always_deref r P)
             (read_region in_expr always_deref r V))
    | Call (id, args, consumes, produces, subst, perms) ->
        let regs =
          call_regions (find_value id)
(*          |> (fun r -> log "@[<hv 2>CALL REGIONS FOR %a ARE %a@]@." Pp.identifier id (Pp.list ~sep: Pp.comma_space Pp.region) r; r)*)
          |> List.map (subst_regreg_list subst)
(*          |> (fun r -> log "@[<hv 2>CALL REGIONS FOR %a AFTER SUBST ARE %a@]@." Pp.identifier id (Pp.list ~sep: Pp.comma_space Pp.region) r; r)*)
          |> List.map (region_arguments_of_region_app ~perms ~consumes)
          |> List.flatten
(*          |> (fun r -> log "@[<hv 2>CALL REGIONS FOR %a AFTER PLOP ARE %a@]@." Pp.identifier id (Pp.list ~sep: Pp.comma_space Pp.identifier) r; r)*)
        in
        let args = List.map expr args in
        let x = Ident.create "p" in
        expr_let
          x
          (app_list (var id) (regs @ args))
          (seq [
            assume_permissions produces;
            var x;
           ])
    | Typed (e, _) ->
        expr e
    | Assert (a, perms) ->
        expr_assert (assertion perms (Some at_init) None a)
    | Old _ ->
        Loc.locate_error x.loc
          "``old'' makes no sense in expressions, it can only be used in terms"

and term perms old current t =
(*
  let is_singleton = is_singleton_in perms in
*)
  let old_term t = term perms old old t in
  let term t = term perms old current t in
  let logic_app_region_arguments = logic_app_region_arguments ?at:current in
  let read_region = read_region ?at: current in
  match t.node with
    | Const CUnit ->
        term_void
    | Const (CInt i) ->
        term_int i
    | Const (CBool b) ->
        term_bool b
    | Tuple l ->
        term_tuple (List.map term l)
    | LogicApp (x, l, rs, _) ->
        log "LogicApp(%a): old = %a@." Pp.identifier x 
          (Pp.option Pp.identifier) old;
        let regions =
          logic_app_region_arguments in_term perms rs in
        term_app x (regions @ List.map term l)
    | Proj (e, i, n) ->
        term_proj n (i - 1) (term e)
    | Left a ->
        left (term a)
    | Right a ->
        right (term a)
    | Var v ->
        term_var v
    | Let (v, a, b) ->
        term_let v (term a) (term b)
    | If (a, b, c) ->
        term_if (term a) (term b) (term c)
(*

    | Deref (a, r, _, _) ->
        (* The third argument (deref kind) is not computed by typing but by
           inference, which is not applied to terms, so we compute it
           ourself with is_singleton. *)
        if is_singleton r then
          read_region in_term (deref_of_perms perms) r V
        else
          get (read_region in_term (deref_of_perms perms) r G) (term a)
*)
    | Deref (a, r, _, _) ->
        Format.eprintf "Deref in logic@.";
        begin match Infer.find_deref_kind r perms with
          | DKSingleton ->
              read_region in_term (deref_of_perms perms) r V
          | DKUnavailable | DKGroup ->
              get (read_region in_term (deref_of_perms perms) r G) (term a)
          | DKFocus s ->
              term_if
                (term_eq (term a) 
                   (read_region in_term (deref_of_perms perms) s P))
                (read_region in_term (deref_of_perms perms) s V)
                (get (read_region in_term (deref_of_perms perms) r G) (term a))
        end
    | Unop (`bnot, a) ->
        term_not (term a)
    | Unop (`neg, a) ->
        term_neg (term a)
    | Binop (`band, a, b) ->
        term_and (term a) (term b)
    | Binop (`bor, a, b) ->
        term_or (term a) (term b)
    | Binop (`bxor, a, b) ->
        term_xor (term a) (term b)
    | Binop (`add, a, b) ->
        term_add (term a) (term b)
    | Binop (`sub, a, b) ->
        term_sub (term a) (term b)
    | Binop (`mul, a, b) ->
        term_mul (term a) (term b)
    | Binop (`div, a, b) ->
        term_div (term a) (term b)
    | Binop (`imod, a, b) ->
        term_mod (term a) (term b)
    | Binop (`lt, a, b) ->
        term_lt (term a) (term b)
    | Binop (`gt, a, b) ->
        term_gt (term a) (term b)
    | Binop (`le, a, b) ->
        term_le (term a) (term b)
    | Binop (`ge, a, b) ->
        term_ge (term a) (term b)
    | Binop (`eq, a, b) ->
        term_eq (term a) (term b)
    | Typed (e, _) ->
        term e
    | Old a ->
        old_term a

    | Call _ ->
        assert false (* impossible (use LogicApp instead) *)

    | Seq _ | While _ | Assign _ | New _ | Pack _ | Unpack _ | Adopt _ | Focus _
    | Unfocus _ | Region _ | Print _ | AdoptRegion _ | UnfocusRegion _
    | PackRegion _ | UnpackRegion _ | BlackBox _ | WeakenRegion _
    | Assert _  ->
        log "term = %a@." Pp.expr t;
        assert false (* impossible *)

and assertion perms old current a =
  let old_assertion a = assertion perms old old a in
  let assertion a = assertion perms old current a in
  let term t = term perms old current t in
  let logic_app_region_arguments = logic_app_region_arguments ?at:current in
  match a.node with
    | PTrue ->
        a_true
    | PFalse ->
        a_false
    | PIff (a, b) ->
        a_iff (assertion a) (assertion b)
    | PImplies (a, b) ->
        a_implies (assertion a) (assertion b)
    | PAnd (a, b) ->
        a_and (assertion a) (assertion b)
    | POr (a, b) ->
        a_or (assertion a) (assertion b)
    | PNot a ->
        a_not (assertion a)
    | PEqual (a, b) ->
        a_eq (term a) (term b)
    | PDiff (a, b) ->
        a_neq (term a) (term b)
    | PGt (a, b) ->
        a_gt (term a) (term b)
    | PLt (a, b) ->
        a_lt (term a) (term b)
    | PGe (a, b) ->
        a_ge (term a) (term b)
    | PLe (a, b) ->
        a_le (term a) (term b)
    | PTerm a ->
        a_eq (term a) (term_bool true)
    | PForall (id, t, a) ->
        a_forall id (translate_type t) (assertion a)
    | PExists (id, t, a) ->
        a_exists id (translate_type t) (assertion a)
    | PApp (id, tl, rs) ->
        let regions = logic_app_region_arguments in_term perms rs in
        a_app id (regions @ List.map term tl)
    | POld a ->
        old_assertion a

let writes_of_region r =
  [ region_var r V;
    region_var r G ]

let reads_of_region perms r =
  if is_region_root_positive_in perms r then
    [ region_var r G;
      region_var r P;
      region_var r V ]
  else
    [ ]

let writes_of_value x =
  Effects.writes_of_value x
  |> List.map writes_of_region
  |> List.flatten

let reads_of_value x =
  x.v_regions
  |> List.map (fun id -> RVar id)
  |> List.map list_of_region_tree
  |> List.flatten
  |> List.map (reads_of_region x.v_consumes)
  |> List.flatten

let parameter_of_value pre post args x =
  let writes = writes_of_value x in
  let reads = reads_of_value x in
(*  log "WRITES OF %a: %a@." Pp.identifier x.v_name
    (Pp.list ~sep: Pp.comma_space Pp.identifier) writes;*)
  let typ =
    type_annot
      ~pre
      ~post
      ~writes
      ~reads
      (type_base (translate_type x.v_return_type))
  in
  let typ =
    List.fold_left
      (fun typ (id, t) -> type_fun id t typ)
      typ
      (List.rev args)
  in
  decl_parameter
    x.v_name
    typ

let let_of_value pre post args x body =
  decl_let
    (Ident.create (Ident.name x.v_name ^ "_safety"))
    (expr_fun
       ~pre
       ~post
       ~args
       (label at_init (seq [
          assume_singletons x.v_params x.v_consumes;
          assume_permissions x.v_consumes;
          expr body;
        ])))

(* if the return type is a pointer type of a singleton region, we add the
   following precondition: result = R_p *)
(* REMARQUE : la c'est utilise aussi bien pour le parameter que pour le let
   mais on pourrait ne le mettre que dans le parameter, ca ferait une obligation
   de preuve triviale de moins ; en parler a Claude *)
let add_pointer_equality x post =
  match x.v_return_type.node with
    | TEPointer (_, r) when is_singleton_in x.v_produces r ->
        let equality =
          a_eq term_result (read_region in_term always_deref r P) in
        a_and post equality
    | _ ->
        post

let value_def acc x =
  Effects.fixpoint [x];
  let pre = assertion x.v_consumes (Some at_old) None x.v_pre in
  let post = assertion x.v_produces (Some at_old) None x.v_post in
  let post = add_pointer_equality x post in
  let args = arguments_of_value x in
  let acc = parameter_of_value pre post args x :: acc in
  match x.v_body with
    | None ->
        acc
    | Some body ->
        let_of_value pre post args x body :: acc

let invariant_predicate_declaration_of c =
  let x, a, regs = c.c_invariant in
  let regs =
    List.map
      (fun r ->
         region_var r G,
         region_group_logic_type r)
      regs
  in
  decl_predicate
    (invariant_id c)
    (regs @ [ x, translate_type c.c_type ])
    (Some (assertion [] None None a))
    (* no permission for invariants, as permissions are used to test whether
       a region is singleton, but the only regions which can be read in
       invariants are the owned regions, which are group *)

let class_def acc x =
  invariant_predicate_declaration_of x :: acc

let logic_type_def acc x =
  let lt = decl_logic_type ~args: x.lt_type_params x.lt_name in
  lt :: acc

(* name: name of the logic function or predicate (as an Ident.t)
   eq: equality (for logic functions) or equivalence (for predicates)
   app: application (a_app for predicates, term_app for logic functions)
   regions: names of the region parameters (as Ident.t), and their types
     (already translated)
   param_types: types of the normal parameters (already translated) *)
let auto_footprint_axiom eq app name regions params =
  let axiom_id = Ident.create (Ident.name name ^ "_footprint") in
  let param_names = List.map fst params in
  let region_names = List.map fst regions in
  let extended_regions =
    List.map
      (fun (id, ty) -> Ident.create (Ident.name id ^ "_x"), ty)
      regions
  in
  let extended_region_names = List.map fst extended_regions in
  let body =
    eq
      (app name (List.map term_var (extended_region_names @ param_names)))
      (app name (List.map term_var (region_names @ param_names)))
  in
  let body =
    List.fold_left2
      (fun acc r' r -> a_implies (extends (term_var r') (term_var r)) acc)
      body
      extended_region_names
      region_names
  in
  let body =
    List.fold_left
      (fun acc (id, ty) -> a_forall id ty acc)
      body
      (extended_regions @ regions @ params)
  in
  decl_axiom
    ~is_lemma: false
    axiom_id
    body

let logic_function_def acc x =
  let regions = region_arguments_of_logic x in
  let region_types = List.map snd regions in
  let param_types = List.map translate_type x.lf_params in
  let args = region_types @ param_types in
  let params =
    List.map
      (fun ty -> Ident.create "x", ty)
      param_types
  in
  let lf =
    decl_logic
      x.lf_name
      ~args
      (translate_type x.lf_return_type)
  in
  let acc = lf :: acc in
  if regions <> [] then
    auto_footprint_axiom a_eq term_app x.lf_name regions params :: acc
  else
    acc

let axiom_def acc x =
  let body =
    List.fold_left
      (fun acc (x, t) -> a_forall x t acc)
      (assertion [] None None x.ax_assertion)
      (* no permission, as all regions are assumed group *)
      (region_arguments_of_axiom x)
  in
  let ax = decl_axiom ~is_lemma:x.ax_is_lemma x.ax_name body 
  in
  ax :: acc

let predicate_def acc x =
  let regions = region_arguments_of_predicate x in
  let params = List.map (fun (x, t) -> x, translate_type t) x.p_params in
  let p =
    decl_predicate
      x.p_name
      (regions @ params)
      (Opt.map (assertion [] None None) x.p_body)
  in
  (* no permission, as all regions are assumed group *)
  let acc = p :: acc in
  if regions <> [] then
    auto_footprint_axiom a_iff a_app x.p_name regions params :: acc
  else
    acc

let def acc = function
  | Class x -> class_def acc x
  | Value x -> value_def acc x
  | LogicType x -> logic_type_def acc x
  | LogicFunction x -> logic_function_def acc x
  | Axiom x -> axiom_def acc x
  | Predicate x -> predicate_def acc x

let file x =
  Why.file (List.rev (List.fold_left def [] x))
