{
(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

  open Lexing
  open Parser
  open Misc

  let identifier = function
    | "as" -> AS
    | "class" -> CLASS
    | "own" -> OWN
    | "end" -> END
    | "let" -> LET
    | "in" -> IN
    | "if" -> IF
    | "then" -> THEN
    | "else" -> ELSE
    | "while" -> WHILE
    | "do" -> DO
    | "new" -> NEW
    | "pack" -> PACK
    | "unpack" -> UNPACK
    | "adopt" -> ADOPT
    | "focus" -> FOCUS
    | "unfocus" -> UNFOCUS
    | "packregion" -> PACKREGION
    | "unpackregion" -> UNPACKREGION
    | "adoptregion" -> ADOPTREGION
    | "unfocusregion" -> UNFOCUSREGION
    | "region" -> REGION
    | "val" -> VAL
    | "consumes" -> CONSUMES
    | "produces" -> PRODUCES
    | "true" -> TRUE
    | "false" -> FALSE
    | "or" -> OR
    | "and" -> AND
    | "xor" -> XOR
    | "not" -> NOT
    | "mod" -> MOD
    | "print" -> PRINT
    | "Left" -> LEFT
    | "Right" -> RIGHT
    | "blackbox" -> BLACKBOX
    | "invariant" -> INVARIANT
    | "requires" -> REQUIRES
    | "ensures" -> ENSURES
    | "forall" -> FORALL
    | "exists" -> EXISTS
    | "logic" -> LOGIC
    | "type" -> TYPE
    | "function" -> FUNCTION
    | "axiom" -> AXIOM
    | "assert" -> ASSERT
    | "selector" -> SELECTOR
    | "predicate" -> PREDICATE
    | "old" -> OLD
    | "lemma" -> LEMMA
    | s -> IDENT s

  exception Lex_error of Loc.t * string

  let lex_error lexbuf s =
    Printf.ksprintf
      (fun s ->
         let loc = lexbuf_location lexbuf in
         raise (Lex_error (loc, s)))
      s

  let buf = Buffer.create 128
}

rule token = parse
  | [' ' '\t' '\r']* { token lexbuf }
  | '\n' { new_line lexbuf; token lexbuf }
  | "(*" { comment lexbuf; token lexbuf }
  | ['a'-'z' 'A'-'Z' '_'] ['a'-'z' 'A'-'Z' '_' '0'-'9']* as x { identifier x }
  | ['0'-'9']+ as x { INT (int_of_string x) }
  | '(' { LPAR }
  | ')' { RPAR }
  | '{' { LBRACE }
  | '}' { RBRACE }
  | "<>" { LTGT }
  | '<' { INF }
  | '>' { SUP }
  | "<=" { INFEQ }
  | ">=" { SUPEQ }
  | ';' { SEMI }
  | '.' { DOT }
  | '=' { EQUAL }
  | '!' { BANG }
  | ":=" { COLONEQUAL }
  | "==>" { IMPLIES }
  | "<=>" { IFF }
(*  | "~" { ANOT }
  | "==" { AEQUAL }
  | "<<" { AINF }
  | ">>" { ASUP }
  | "/\\" { AAND }
  | "\\/" { AOR }*)
  | ':' { COLON }
  | '[' { LBRACKET }
  | ']' { RBRACKET }
  | ',' { COMMA }
  | '*' { STAR }
  | '|' { VERT }
  | '!' { BANG }
  | '+' { PLUS }
  | '-' { MINUS }
  | '/' { DIV }
  | "^e" { PERMEMPTY }
  | "^o" { PERMOPEN }
  | "^c" { PERMCLOSED }
  | "^g" { PERMGROUP }
  | "->" { ARROW }
  | '"' { Buffer.clear buf; string lexbuf }
  | eof { EOF }
  | _ { lex_error lexbuf "invalid character" }

and comment = parse
  | "*)" { }
  | "(*" { comment lexbuf; comment lexbuf }
  | '\n' { new_line lexbuf; comment lexbuf }
  | eof { lex_error lexbuf "unterminated comment" }
  | _ { comment lexbuf }

and string = parse
  | '"' { STRING (Buffer.contents buf) }
  | '\n' { Buffer.add_char buf '\n'; new_line lexbuf; string lexbuf }
  | [^'\n' '"']+ as x { Buffer.add_string buf x; string lexbuf }
  | eof { lex_error lexbuf "unterminated string" }
