(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Lexing
open Lexer
open Misc

(* If I could rewrite Capucine, I would change the following.
   - Typing would use a union-find data structure, and each type would
     actually be a key of this union-find. Keys would thus be abstract
     and we would never forget an expand_head.
   - The distinction between terms and predicates would not be done
     by the parser, but by a later pass.
   - The AST for terms would be different. Never be afraid to create more
     ASTs.
   - More thought for interp.ml, especially for how to read
     region references and term/expr contexts. *)

let parse file =
  let ch = try
    open_in file
  with
    | Sys_error sys ->
        error "Cannot open file (%s)" sys
  in
  let lexbuf = Lexing.from_channel ch in
  try
    lexbuf.lex_curr_p <- { lexbuf.lex_curr_p with pos_fname = file };
    let ast = Parser.file Lexer.token lexbuf in
    close_in ch;
    ast
  with
    | Lex_error (loc, err)
    | Misc.Parse_error (loc, err) ->
        Loc.locate_error loc "%s" err
    | Parsing.Parse_error ->
        Loc.locate_error (lexbuf_location lexbuf) "Parse error"

let file file =
  log "========= File %s ==========\n%!" file;
  log "--------- Parsing ----------\n%!";
  let ast = parse file in
  log "--------- Binding ----------\n%!";
  let _, ast = Bind.file ast in
  Pp.file (Format.formatter_of_out_channel stdout) ast;
  log "--------- Typing ----------\n%!";
  let env, ast = Typing.file ast in
  Pp.file (Format.formatter_of_out_channel stdout) ast;
  log "--------- Inference ----------\n%!";
  Infer.classes_environment := env.Typing_common.classes;
  let ast = Infer.file ast in
  Pp.file (Format.formatter_of_out_channel stdout) ast;
  log "--------- Why Interpretation ----------\n%!";
  Interp.classes_environment := env.Typing_common.classes;
  Interp.values_environment := env.Typing_common.values;
  let why = Interp.file ast in
  Pp.print_unique_identifiers := true;
  Why.to_channel stdout why;
  Opt.iter (fun s -> Why.to_file s why) !Options.out

let () =
  List.iter file !Options.files
