(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Ast
open Misc
open Typing

exception Permission_not_found of permission

(* empty and closed are sub regions of group *)
let is_sub_permission a b =
  (* maybe expanding a and b is not useful *)
  let a = Unify.expand_permission a in
  let b = Unify.expand_permission b in
  if a <> b then
    match a, b with
      | (PEmpty s | PClosed s), PGroup r -> s = r
      | _ -> false
  else
    true

let region_eq a b =
  let a = Unify.expand_region a in
  let b = Unify.expand_region b in
  a = b

let permission_eq a b =
  match a, b with
    | PEmpty a, PEmpty b
    | POpen a, POpen b
    | PClosed a, PClosed b
    | PGroup a, PGroup b ->
        region_eq a b
    | PArrow (a1, a2), PArrow (b1, b2) ->
        region_eq a1 b1 && region_eq a2 b2
    | _ ->
        false

(* is p in set, or a super-permission of a permission of set? *)
let check_membership set p =
  if not (List.exists (fun a -> is_sub_permission a p) set) then
    raise (Permission_not_found p)

(* are all permissions of a super-permissions of permissions of set? *)
let check_inclusion a b =
  List.iter (check_membership b) a

let region_of = function
  | TPointer (_, _, _, r) -> Unify.expand_region r
  | _ -> raise (Invalid_argument "region_of")

let class_of = function
  | TPointer (_, _, c, _) -> c
  | _ -> raise (Invalid_argument "class_of")

(* consume p, or a sub-permission of p, in caps *)
let consume loc caps p =
  try
    list_removef (fun a -> is_sub_permission a p) caps
  with Not_found ->
    Loc.locate_error loc
      "Permission %a is required here.\nAvailable permissions are: %a"
      Pp.permission p Pp.permissions caps

let owned_regions env r c =
  let c = Env.get_class env c in
  List.map (fun o -> RSub (r, o)) c.c_owned_regions

(* for arrows, r must be at the right (i.e. positive position) *)
let rec permission_on r = function
  | [] -> None
  | (PEmpty s
    | PClosed s
    | POpen s
    | PGroup s
    | PArrow (_, s) as x) :: _ when region_eq s r -> Some x
  | _ :: rem -> permission_on r rem

let instanciate_permission regions = function
  | PEmpty r ->
      PEmpty (instanciate_region regions r)
  | PClosed r ->
      PClosed (instanciate_region regions r)
  | POpen r ->
      POpen (instanciate_region regions r)
  | PGroup r ->
      PGroup (instanciate_region regions r)
  | PArrow (s, r) ->
      PArrow (instanciate_region regions s, instanciate_region regions r)

let rec expr env caps x =
  let consume = consume x.loc in
  match x.node with
    | Const _
    | Var _ ->
        caps
    | Unop (_, e)
    | Proj (e, _)
    | Deref e
    | Left e
    | Right e ->
        expr env caps e
    | Binop (_, e1, e2)
    | Let (_, e1, e2)
    | Seq (e1, e2) ->
        expr env (expr env caps e1) e2
    | Tuple el ->
        List.fold_left (expr env) caps el
    | If (e1, e2, e3) ->
        expr env (expr env (expr env caps e1) e2) e3
    | Call (f, el, rs) ->
        let f = Env.get_value env f in
        let instanciate_permissions = List.map (instanciate_permission !rs) in
        let caps = List.fold_left (expr env) caps el in
        let consumes = instanciate_permissions f.v_consumes in
        let produces = instanciate_permissions f.v_produces in
        let caps = List.fold_left consume caps consumes in
        caps @ produces
    | While (e1, e2) ->
        let caps1 = expr env caps e1 in
        let caps2 = expr env caps1 e2 in
        check_inclusion caps1 caps2; (* maybe not the best expressiveness *)
        caps2
    | Assign (e1, e2) ->
        let caps1 = expr env caps e1 in
        let caps2 = expr env caps1 e2 in
        let rc = POpen (region_of e1.typ) in
        begin try
          check_membership caps2 rc
        with Permission_not_found _ ->
          Loc.locate_error x.loc "Permission %a is required for this assignment"
            Pp.permission rc
        end;
        caps2
    | New ((_, _, c), r) ->
        let caps = consume caps (PEmpty r) in
        let owned = owned_regions env r c in
        let owned = List.map (fun r -> PEmpty r) owned in
        (POpen r) :: caps @ owned
    | Pack e ->
        let r = region_of e.typ in
        let caps = expr env caps e in
        let caps = consume caps (POpen r) in
        let owned = owned_regions env r (class_of e.typ) in
        let owned = List.map (fun r -> PGroup r) owned in
        let caps = List.fold_left consume caps owned in
        (PClosed r) :: caps
    | Unpack e ->
        let r = region_of e.typ in
        let caps = expr env caps e in
        let caps = consume caps (PClosed r) in
        let owned = owned_regions env r (class_of e.typ) in
        let owned = List.map (fun r -> PGroup r) owned in
        (POpen r) :: caps @ owned
    | Adopt (e, r) ->
        let s = region_of e.typ in
        let caps = expr env caps e in
        let caps = consume caps (PClosed s) in
        let caps = consume caps (PGroup r) in
        (PGroup r) :: caps
    | Focus (e, s) ->
        let r = region_of e.typ in
        let caps = expr env caps e in
        let caps = consume caps (PEmpty s) in
        let caps = consume caps (PGroup r) in
        (PClosed s) :: (PArrow (s, r)) :: caps
    | FocusBind (e, vid, rid, body) ->
        let r = region_of e.typ in
        let s = RVar rid in
        let caps = expr env caps e in
        let caps = consume caps (PGroup r) in
        let caps = (PClosed s) :: (PArrow (s, r)) :: caps in
        expr env caps body
    | Unfocus (e, r) ->
        let s = region_of e.typ in
        let caps = expr env caps e in
        let caps = consume caps (PClosed s) in
        let caps = consume caps (PArrow (s, r)) in
        (PGroup r) :: caps
    | Region (r, e) ->
        let caps = (PEmpty (RVar r)) :: caps in
        expr env caps e
    | Print (s, e) ->
        let caps' = expr env caps e in
        log "%s consumes: %a\n" s Pp.permissions caps;
        log "%s produces: %a\n%!" s Pp.permissions caps';
        caps'
    | UnpackRegion _
    | PackRegion _ ->
        assert false (* TODO *)
    | UnfocusRegion (s, r) ->
        let caps = consume caps (PClosed s) in
        let caps = consume caps (PArrow (s, r)) in
        (PGroup r) :: caps
    | AdoptRegion (s, r) ->
        begin match permission_on s caps, permission_on r caps with
          | None, None ->
              Loc.locate_error x.loc
                "No permission is available for both %a and %a"
                Pp.region s Pp.region r
          | None, Some _ ->
              Loc.locate_error x.loc
                "No permission is available for %a"
                Pp.region s
          | Some _, None ->
              Loc.locate_error x.loc
                "No permission is available for %a"
                Pp.region r
          | Some a, Some b ->
              let caps = consume caps a in
              let caps = consume caps b in
              begin match a, b with
                | PEmpty _, _ ->
                    Loc.locate_warning x.loc "Warning: region %a is empty. \
There is usually no point in adopting an empty region." Pp.region s;
                    b :: caps
                | _, POpen _ ->
                    Loc.locate_error x.loc "Invalid adoption: region %a is \
open but region %a is not empty." Pp.region r Pp.region s
                | POpen _, PEmpty _ ->
                    (POpen r) :: caps
                | POpen _, _ ->
                    Loc.locate_error x.loc "Invalid adoption: region %a is \
open but region %a is not empty." Pp.region s Pp.region r;
                | PClosed _, PEmpty _ ->
                    (PClosed r) :: caps
                | PClosed _, (PClosed _ | PGroup _) ->
                    (PGroup r) :: caps
                | PClosed _, PArrow _ ->
                    b :: caps
                | PGroup _, (PEmpty _ | PClosed _ | PGroup _) ->
                    (PGroup r) :: caps
                | PGroup _, PArrow _ ->
                    b :: caps
                | PArrow _, _ ->
                    Loc.locate_error x.loc "Invalid adoption: region %a is \
being focused." Pp.region s
              end
        end
    | BlackBox (c, p) ->
        let caps = List.fold_left consume caps c in
        p @ caps

let class_def env x =
  () (* nothing to do here *)

let value_def env x =
  begin match x.v_body with
    | None ->
        ()
    | Some body ->
        try
          check_inclusion x.v_produces (expr env x.v_consumes body)
        with Permission_not_found p ->
          Loc.locate_error body.loc
            "The implementation of %s does not produce permission %a"
            x.v_name Pp.permission p
  end;
  log "%s: caping ok\n" x.v_name

let def env = function
  | Class x -> class_def env x
  | Value x -> value_def env x

let file env x =
  List.iter (def env) x
