(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Ast
open Misc
open Env

let const = function
  | CUnit -> TBase TUnit
  | CInt _ -> TBase TInt
  | CBool _ -> TBase TBool

let operator: [< op] -> typ = function
  | #int_op -> TBase TInt
  | #bool_op -> TBase TBool

type environment = unit

let empty_env = ()

type expr_result = {
  er_expr: expr;
  er_consumes: permission list;
  er_produces: permission list;
}

let rec consume acc perm = Caping.consume Loc.dummy_location acc perm
let permission_eq = Caping.permission_eq
let region_eq = Caping.region_eq
let region_of = Caping.region_of
let union a b = a @ b

type conflict =
  | NoConflict
  | ParentOccurence of region
  | SonOccurence of region
  | DirectConflict of permission

let region_of_permission = function
  | PEmpty r
  | POpen r
  | PClosed r
  | PGroup r
  | PArrow (_, r) ->
      r

let is_closed = function
  | PClosed _ -> true
  | _ -> false

let is_open = function
  | POpen _ -> true
  | _ -> false

let rec is_parent_region a b =
  region_eq a b ||
    match b with
      | RSub (r, _) -> is_parent_region a r
      | _ -> false

(** Renvoit dans l'ordre de priorité : DirectConflict, ParentOccurence,
    SonOccurence, NoConflict. *)
let find_conflict r l =
  try
    DirectConflict
      (List.find (fun p -> region_eq (region_of_permission p) r) l)
  with Not_found ->
    try
      ParentOccurence
        (region_of_permission
           (List.find (fun p -> is_parent_region (region_of_permission p) r) l))
    with Not_found ->
      try
        SonOccurence
          (region_of_permission
             (List.find
                (fun p -> is_parent_region r (region_of_permission p))
                l))
      with Not_found ->
        NoConflict

let expr_of_node ?(loc = Loc.dummy_location) n =
  { loc = loc;
    node = n;
    typ = TBase TUnit }

let owned_regions_of r =
  [] (* TODO *)

let unpack r =
  {
    er_expr = expr_of_node (UnpackRegion r);
    er_consumes = [ PClosed r ];
    er_produces = POpen r :: owned_regions_of r;
  }

let pack r =
  {
    er_expr = expr_of_node (PackRegion r);
    er_consumes = POpen r :: owned_regions_of r;
    er_produces = [ PClosed r ];
  }

let unfocus s r =
  {
    er_expr = expr_of_node (UnfocusRegion (s, r));
    er_consumes = [ PClosed s; PArrow (s, r) ];
    er_produces = [ PGroup r ];
  }

let black_box a b =
  {
    er_expr = expr_of_node (Const CUnit);
    er_consumes = a;
    er_produces = b;
  }

let seq_expr ?loc a b =
  match a.node, b.node with
    | Const CUnit, _ -> b
    | _, Const CUnit -> a
    | _, _ -> expr_of_node ?loc (Seq (a, b))

let let_expr ?loc v a b =
  match b.node with
    | Var s when s = v -> a
    | _ -> expr_of_node ?loc (Let (v, a, b))

let weaken r = black_box [ PClosed r ] [ PGroup r ]

(** Modifie [e1] et [e2] de telle façon que les permissions consommées
    par [e2] soient produites par [e1].

    Pour chaque capacité [c] consommée par [b] :
    - si [c] est produite par [a], rien à faire ;
    - si [a] produit une capacité sur la région de [c] ou une de ses
    parentes, choisir l'opération à insérer et la joindre
    récursivement ;
    - si [a] ne produit pas telle capacité, appliquer CWeak4.

    Ensuite on vérifie que les capacités consommées par [b] sont
    produites par [a]. Si non, la contrainte est insoluble. Si oui, on
    les consomme, et on rajoute le reste aux capacités produites par
    [b]. *)
let rec join (a: expr_result) (b: expr_result): expr_result * expr_result =
  let a = List.fold_left join_one a b.er_consumes in
  let rem = List.fold_left consume a.er_produces b.er_consumes in
  let b = { b with er_produces = union b.er_produces rem } in
  a, b

(** Insère des opérations après [a] pour assurer que [perm] soit produit.
    Attention, si vous modifiez cette fonction en insérant des opérations
    avant [a], le cas [Tuple] de [expr] ne marchera plus. *)
and join_one a perm =
  let r = region_of_permission perm in
  match find_conflict r a.er_produces with
    | NoConflict | SonOccurence _ ->
        (* SonOccurence sera résolu par la présence de la capacité ouverte *)
        { a with
            er_consumes = perm :: a.er_consumes;
            er_produces = perm :: a.er_produces }
    | ParentOccurence r ->
        join_one (seqf a (unpack r)) perm
    | DirectConflict c ->
        if permission_eq c perm then a
        else match c, perm with
          | PEmpty r, PGroup _ -> seqf a (weaken r)
          | POpen r, PClosed _ -> seqf a (pack r)
          | POpen r, PGroup _ -> seqf a (seq (pack r) (weaken r))
          | PClosed r, POpen _ -> seqf a (unpack r)
          | PClosed r, PGroup _ -> seqf a (weaken r)
          | PArrow (s, r), PGroup _ -> seqf a (unfocus s r)
          | _ -> failwith "infer.ml: join_one"

and seq a b =
  let a, b = join a b in
  {
    er_expr = seq_expr ~loc: a.er_expr.loc a.er_expr b.er_expr;
    er_consumes = a.er_consumes;
    er_produces = b.er_produces;
  }

and seqf = let c = ref 0 in fun a b ->
  let v = "_seqf_" ^ string_of_int !c in
  incr c;
  let a, b = join a b in
  let seq = seq_expr ~loc: b.er_expr.loc b.er_expr (expr_of_node (Var v)) in
  {
    er_expr = let_expr ~loc: a.er_expr.loc v a.er_expr seq;
    er_consumes = a.er_consumes;
    er_produces = b.er_produces;
  }

(** Modifie [a] de telle façon qu'il produise toutes les permissions
    consommées par [b] et [c]. Pour cela, on appelle [join_one] sur
    toutes ces permissions. Les permissions restantes sont rajoutées
    aux branches [b] et [c]. C'est donc un peu comme [join] mais pour
    deux branches. *)
let branch_begin a b c =
  let a = List.fold_left join_one a b.er_consumes in
  let a = List.fold_left join_one a c.er_consumes in
  let rem_b = List.fold_left consume a.er_produces b.er_consumes in
  let b = { b with er_produces = union b.er_produces rem_b } in
  let rem_c = List.fold_left consume a.er_produces c.er_consumes in
  let c = { c with er_produces = union c.er_produces rem_c } in
  a, b, c

(** Modifie [b] et [c] de telle façon que les deux expressions produisent
    les mêmes permissions.

    Comment ça marche : on calcule une union maximale des permissions
    produites.  Cette union maximale est construite de la façon
    suivante : pour chaque région, soit la région n'apparaît dans
    aucune permission produite et on ne fait rien ; soit la région
    apparaît dans les permissions produites des deux branches, on
    choisit la permission la plus générale (voir ci-dessous) et on
    appelle [join_one] sur les deux branches sur cette permission ;
    soit la région apparaît dans l'une des deux branches uniquement.

    Ce dernier cas a lui-même plusieurs sous-cas. On a une branche
    (disons A) qui produit P sur R dont l'autre branche B ne parle pas
    dans ses productions. Si A consomme une permission sur R, alors on
    a de toute façon besoin que ce qu'il y a avant la branche produise
    quelque chose sur R, donc on rajoute P aux permissions consommées
    et produites par B. Si A ne consomme pas une telle permission,
    alors de toute façon il la produit de "nulle part". Si on
    rajoutait P dans les consommations de B, la branche A aurait au
    final aussi P en entrée et on finirait par avoir un doublon, ce
    qui est forcément une erreur. Donc on oublie simplement P dans A.

    La permission la plus générale doit être choisie de telle façon
    que l'on aie le moins de [pack] possible à faire. Autrement dit,
    si on a le choix entre packer une branche ou unpacker l'autre, on
    préfère unpacker. *)
let rec branch_end ?(redo = true) a b =
  let a, bp =
    List.fold_left
      (fun (a, bp) p ->
         let rp = region_of_permission p in
         match find_conflict rp a.er_produces with
           | SonOccurence _ ->
               (* sera traité par le cas symmétrique avec ParentOccurence *)
               a, p :: bp
           | ParentOccurence _ ->
               join_one a p, p :: bp
           | DirectConflict ap ->
               if is_closed p && is_open ap then
                 (* sera traité par le cas symmétrique *)
                 a, p :: bp
               else
                 join_one a p, p :: bp
           | NoConflict ->
               match find_conflict rp b.er_consumes with
                 | NoConflict ->
                     (* oublions juste la capacité *)
                     a, bp
                 | DirectConflict _
                 | ParentOccurence _
                 | SonOccurence _ ->
                     join_one a p, p :: bp)
      (a, b.er_produces)
      b.er_produces
  in
  let b = { b with er_produces = bp } in
  if redo then branch_end ~redo: false b a else b, a

(** Infère le type, les permissions consommées et les permissions
    produites minimales nécessaires pour que l'expression soit typable. *)
let rec expr env (e: expr): expr_result =
  let expr = expr env in
  let make ?(e = e) ?(n = e.node) c p =
    {
      er_expr = { e with node = n };
      er_consumes = c;
      er_produces = p;
    }
  in
  match e.node with
    | Const _
    | Var _ ->
        make [] []
    | Unop (op, a) ->
        let a = expr a in
        make ~n: (Unop (op, a.er_expr)) a.er_consumes a.er_produces
    | Binop (op, a, b) ->
        let a, b = join (expr a) (expr b) in
        make ~n: (Binop (op, a.er_expr, b.er_expr)) a.er_consumes b.er_produces
    | Tuple l ->
        let l = List.map expr l in
        let c, l, p = expr_list l in
        let l = List.map (fun x -> x.er_expr) l in
        make ~n: (Tuple l) c p
    | Proj (a, i) ->
        let a = expr a in
        make ~n: (Proj (a.er_expr, i)) a.er_consumes a.er_produces
    | Left a ->
        let a = expr a in
        make ~n: (Left a.er_expr) a.er_consumes a.er_produces
    | Right a ->
        let a = expr a in
        make ~n: (Right a.er_expr) a.er_consumes a.er_produces
    | Let (v, a, b) ->
        let a, b = join (expr a) (expr b) in
        make ~n: (Let (v, a.er_expr, b.er_expr)) a.er_consumes b.er_produces
    | Seq (a, b) ->
        (* Ne pas utiliser seq pour laisser les (). *)
        let a, b = join (expr a) (expr b) in
        make ~n: (Seq (a.er_expr, b.er_expr)) a.er_consumes b.er_produces
    | Call (f, l, s) ->
        let fv = Env.get_value env f in
        let l = List.map expr l in
        begin match List.rev l with
          | [] ->
              make fv.v_consumes fv.v_produces
          | last :: tl ->
              let last = seq last (black_box fv.v_consumes fv.v_produces) in
              let c, l, p = expr_list (List.rev (last :: tl)) in
              let l = List.map (fun x -> x.er_expr) l in
              make ~n: (Call (f, l, s)) c p
        end
    | If (a, b, c) ->
        let a = expr a in
        let b = expr b in
        let c = expr c in
        let b, c = branch_end b c in
        (* On place le plus possible d'annotations dans les branches
           elles-mêmes, et ensuite on finit avec branch_begin. *)
        let bb = black_box a.er_produces a.er_produces in
        let b = seq bb b in
        let c = seq bb c in
        let a, b, c = branch_begin a b c in
        make
          ~n: (If (a.er_expr, b.er_expr, c.er_expr))
          a.er_consumes
          b.er_produces
    | While (a, b) ->
        let b = expr b in
        let b = seq b (black_box b.er_consumes b.er_consumes) in
        let a, b = join (expr a) b in
        make ~n: (While (a.er_expr, b.er_expr)) a.er_consumes b.er_produces
    | Assign (a, b) ->
        let p = [ POpen (region_of a.typ) ] in
        let b = seqf (expr b) (black_box p p) in
        let a, b = join (expr a) b in
        make ~n: (Assign (a.er_expr, b.er_expr)) a.er_consumes b.er_produces
    | Deref a ->
        let a = expr a in
        make ~n: (Deref a.er_expr) a.er_consumes a.er_produces
    | New ((rl, tl, c), r) ->
        make [ PEmpty r ] [ POpen r ]
    | Pack a ->
        let ra = region_of a.typ in
        let a =
          seq
            (expr a)
            (black_box (POpen ra :: owned_regions_of ra) [ PClosed ra ])
        in
        make ~n: (Pack a.er_expr) a.er_consumes a.er_produces
    | Unpack a ->
        let ra = region_of a.typ in
        let a =
          seq
            (expr a)
            (black_box [ PClosed ra ] (POpen ra :: owned_regions_of ra))
        in
        make ~n: (Unpack a.er_expr) a.er_consumes a.er_produces
    | Adopt (a, r) ->
        (* TODO: other kinds of adoption *)
        let ra = region_of a.typ in
        let a = expr a in
        let a = seq a (black_box [ PClosed ra; PGroup r ] [ PGroup r ]) in
        make ~n: (Adopt (a.er_expr, r)) a.er_consumes a.er_produces
    | Focus (a, r) ->
        let ra = region_of a.typ in
        let a =
          seq
            (expr a)
            (black_box [ PGroup ra; PEmpty r ] [ PArrow (r, ra); PClosed r ])
        in
        make ~n: (Focus (a.er_expr, r)) a.er_consumes a.er_produces
    | FocusBind _ ->
        assert false (* TODO *)
    | Unfocus (a, r) ->
        let ra = region_of a.typ in
        let a = expr a in
        let a = seq a (black_box [ PClosed ra; PArrow (ra, r) ] [ PGroup r ]) in
        make ~n: (Unfocus (a.er_expr, r)) a.er_consumes a.er_produces
    | Region (s, a) ->
        let a = seq (black_box [] [ PEmpty (RVar s) ]) (expr a) in
        (* TODO: consume permissions on s after a is finished *)
        make ~n: (Region (s, a.er_expr)) a.er_consumes a.er_produces
    | Print (s, a) ->
        let a = expr a in
        make ~n: (Print (s, a.er_expr)) a.er_consumes a.er_produces
    | AdoptRegion (s, r) ->
        (* TODO: other kinds of adoption *)
        make [ PClosed s; PGroup r ] [ PGroup r ]
    | UnfocusRegion (s, r) ->
        make [ PClosed s; PArrow (s, r) ] [ PGroup r ]
    | PackRegion s ->
        make (POpen s :: owned_regions_of s) [ PGroup s ]
    | UnpackRegion s ->
        make [ PGroup s ] (POpen s :: owned_regions_of s)
    | BlackBox (c, p) ->
        make c p

and expr_list (l: expr_result list) =
  let bb, l = list_fold_map
    (fun bb a ->
       let bb, a = join bb a in
       { bb with er_produces = a.er_produces }, a)
    (black_box [] [])
    l
  in
  bb.er_consumes, l, bb.er_produces

let value_def env x =
  match x.v_body with
    | None ->
        env, Value x
    | Some body ->
        let er = expr env body in
        let er = seq (black_box [] x.v_consumes) er in
        let er = seqf er (black_box x.v_produces []) in
        (* TODO: check consumes *)
        let x = { x with v_body = Some er.er_expr } in
        let env = add_value env x.v_name x in
        env, Value x

(* On n'infère rien au niveau des classes. *)
let class_def env x =
  env, Class x

let def env = function
  | Class x -> class_def env x
  | Value x -> value_def env x

let file env x =
  list_fold_map def env x
