%{
(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

  open Parsing
  open Ast

  let loc () =
    symbol_start_pos (), symbol_end_pos ()

  let node x = {
    loc = loc ();
    node = x;
    typ = TBase TUnit; (* dummy type *)
  }

  let parser_error s =
    Printf.ksprintf (fun s -> raise (Misc.Parse_error (loc (), s))) s

  let region_binder x =
    match x with
      | RVar s -> s
      | _ -> parser_error "only region binders are allowed here"

  let type_binder x =
    match x.node with
      | TEIdent s -> s
      | _ -> parser_error "only type binders are allowed here"

  let class_expr_binder (rp, tp, n) =
    List.map region_binder rp, List.map type_binder tp, n
%}

%token <string> IDENT
%token <string> STRING
%token <int> INT
%token CLASS OWN END VAL LET IN IF THEN ELSE WHILE DO NEW PACK UNPACK ADOPT AS
%token FOCUS UNFOCUS REGION LEFT RIGHT
%token CONSUMES PRODUCES
%token LPAR RPAR SEMI DOT EQUAL BANG COLONEQUAL LBRACKET RBRACKET COMMA EOF
%token INF SUP COLON STAR LBRACE RBRACE VERT
%token PERMEMPTY PERMOPEN PERMCLOSED PERMGROUP ARROW BANG
%token TRUE FALSE
%token OR XOR AND NOT PLUS MINUS DIV MOD
%token PRINT BLACKBOX

%nonassoc IN
%left SEMI
%nonassoc THEN ELSE DO PACK UNPACK COLONEQUAL
%left AND
%left OR XOR
%left PLUS MINUS
%left STAR VERT DIV MOD
%left DOT
%nonassoc PRINT
%nonassoc NOT LEFT RIGHT
%nonassoc BANG

%type <Ast.file> file
%start file
%%

file:
| class_def file
    { Class $1 :: $2 }
| value_def file
    { Value $1 :: $2 }
| EOF
    { [] }
;

class_def:
| CLASS class_expr self_region EQUAL own_clause type_expr END
    { let rp, tp, n = class_expr_binder $2 in
      {
        c_region_params = rp;
        c_type_params = tp;
        c_name = n;
        c_owned_regions = $5;
        c_type = $6;
        c_self_region = $3;
      } }
;

self_region:
| LBRACKET IDENT RBRACKET
    { Some $2 }
|
    { None }
;

own_clause:
| OWN ident_comma_list SEMI
    { $2 }
|
    { [] }
;

ident_comma_list:
| IDENT comma_ident_star
    { $1 :: $2 }
;

comma_ident_star:
| COMMA IDENT comma_ident_star
    { $2 :: $3 }
|
    { [] }
;

value_def:
| VAL IDENT LPAR argument_list RPAR COLON type_expr consumes_clause
  produces_clause optional_body
    { {
        v_name = $2;
        v_params = $4;
        v_return_type = $7;
        v_consumes = $8;
        v_produces = $9;
        v_body = $10;
      } }
;

argument_list:
| argument comma_argument_star
    { $1 :: $2 }
|
    { [] }
;

argument:
| IDENT COLON type_expr
    { $1, $3 }
;

comma_argument_star:
| COMMA argument comma_argument_star
    { $2 :: $3 }
|
    { [] }
;

consumes_clause:
| CONSUMES permission_comma_list
    { $2 }
|
    { [] }
;

produces_clause:
| PRODUCES permission_comma_list
    { $2 }
|
    { [] }
;

permission_comma_list:
| permission comma_permission_star
    { $1 :: $2 }
;

comma_permission_star:
| COMMA permission comma_permission_star
    { $2 :: $3 }
|
    { [] }
;

permission:
| region PERMEMPTY
    { PEmpty $1 }
| region PERMOPEN
    { POpen $1 }
| region PERMCLOSED
    { PClosed $1 }
| region PERMGROUP
    { PGroup $1 }
| region ARROW region
    { PArrow ($1, $3) }
;

region:
| IDENT
    { RVar $1 }
| region DOT IDENT
    { RSub ($1, $3) }
;

optional_body:
| EQUAL expr
    { Some $2 }
|
    { None }
;

type_expr:
| IDENT
    { node (TEIdent $1) }
| type_expr VERT type_expr
    { node (TESum ($1, $3)) }
| LPAR type_expr star_type_star RPAR
    { match $3 with
        | [] -> $2
        | x -> node (TETuple ($2 :: x)) }
| class_expr LBRACKET region RBRACKET
    { node (TEPointer ($1, $3)) }
;

class_expr:
| IDENT
    { [], [], $1 }
| IDENT region_parameters
    { $2, [], $1 }
| IDENT type_parameters
    { [], $2, $1 }
| IDENT region_parameters type_parameters
    { $2, $3, $1 }
| IDENT type_parameters region_parameters
    { $3, $2, $1 }
;

region_parameters:
| LBRACE region comma_region_star RBRACE
    { $2 :: $3 }
;

comma_region_star:
| COMMA region comma_region_star
    { $2 :: $3 }
|
    { [] }
;

type_parameters:
| LPAR type_expr comma_type_star RPAR
    { $2 :: $3 }
;

star_type_star:
| STAR type_expr star_type_star
    { $2 :: $3 }
|
    { [] }
;

comma_type_star:
| COMMA type_expr comma_type_star
    { $2 :: $3 }
|
    { [] }
;

comma_expr_star:
| COMMA expr comma_expr_star
    { $2 :: $3 }
|
    { [] }
;

expr:
| LPAR RPAR
    { node (Const CUnit) }
| INT
    { node (Const (CInt $1)) }
| TRUE
    { node (Const (CBool true)) }
| FALSE
    { node (Const (CBool false)) }
| LPAR expr comma_expr_star RPAR
    { match $3 with
        | [] -> $2
        | x -> node (Tuple ($2 :: x)) }
| expr DOT INT
    { node (Proj ($1, $3)) }
| IDENT
    { node (Var $1) }
| LET IDENT EQUAL expr IN expr
    { node (Let ($2, $4, $6)) }
| expr SEMI expr
    { node (Seq ($1, $3)) }
| IDENT LPAR RPAR
    { node (Call ($1, [], ref [])) }
| IDENT LPAR expr comma_expr_star RPAR
    { node (Call ($1, $3 :: $4, ref [])) }
| IF expr THEN expr
    { node (If ($2, $4, node (Const CUnit))) }
| IF expr THEN expr ELSE expr
    { node (If ($2, $4, $6)) }
| WHILE expr DO expr
    { node (While ($2, $4)) }
| expr COLONEQUAL expr
    { node (Assign ($1, $3)) }
| BANG expr
    { node (Deref $2) }
| NEW class_expr LBRACKET region RBRACKET
    { node (New ($2, $4)) }
| PACK expr
    { node (Pack $2) }
| UNPACK expr
    { node (Unpack $2) }
| ADOPT expr IN region
    { node (Adopt ($2, $4)) }
| FOCUS expr IN region
    { node (Focus ($2, $4)) }
| FOCUS expr AS IDENT REGION IDENT IN expr
    { node (FocusBind ($2, $4, $6, $8)) }
| UNFOCUS expr IN region
    { node (Unfocus ($2, $4)) }
| REGION IDENT IN expr
    { node (Region ($2, $4)) }
| expr PLUS expr
    { node (Binop (`add, $1, $3)) }
| expr MINUS expr
    { node (Binop (`sub, $1, $3)) }
| expr STAR expr
    { node (Binop (`mul, $1, $3)) }
| expr DIV expr
    { node (Binop (`div, $1, $3)) }
| expr MOD expr
    { node (Binop (`imod, $1, $3)) }
| expr AND expr
    { node (Binop (`band, $1, $3)) }
| expr OR expr
    { node (Binop (`bor, $1, $3)) }
| expr XOR expr
    { node (Binop (`bxor, $1, $3)) }
| NOT expr
    { node (Unop (`bnot, $2)) }
| MINUS expr
    { node (Unop (`neg, $2)) }
| PRINT expr
    { node (Print ("user print", $2)) }
| PRINT STRING expr %prec PRINT
    { node (Print ($2, $3)) }
| LEFT expr
    { node (Left $2) }
| RIGHT expr
    { node (Right $2) }

| ADOPT REGION region IN region
    { node (AdoptRegion ($3, $5)) }
| UNFOCUS REGION region IN region
    { node (UnfocusRegion ($3, $5)) }
| PACK REGION region
    { node (PackRegion $3) }
| UNPACK REGION region
    { node (UnpackRegion $3) }

| BLACKBOX LPAR consumes_clause produces_clause RPAR
    { node (BlackBox ($3, $4)) }
;
