(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Ast
open Misc

exception Unification_error

let unification_error () = raise Unification_error

module type UNIFIABLE = sig
  type t
  val unify: bool -> Env.t -> t -> t -> unit
  val subst: int -> t -> t -> t
  val occur_check: int -> t -> unit
end

module type SUBST = sig
  type v
  type t
  val create: unit -> t
  val add: Env.t -> t -> int -> v -> unit
  val apply: t -> v -> v
end

module Subst(X: UNIFIABLE): SUBST with type v = X.t = struct
  type v = X.t
  type t = (int * X.t) list ref

  let create () =
    ref []

  let rec sep i = function
    | [] ->
        raise Not_found
    | (j, v) :: r when i = j ->
        v, r
    | x :: r ->
        let v, r' = sep i r in
        v, x :: r'

  (* assumes i not in s *)
  let subst_in_subst s i v =
    List.map (fun (j, w) -> j, X.subst i v w) s

  let add env s i v =
    X.occur_check i v;
    try
      let v', s' = sep i !s in
      s := (i, v) :: subst_in_subst s' i v;
      X.unify false env v v' (* no loop as i is not in v (occur check above) *)
    with Not_found ->
      s := (i, v) :: subst_in_subst !s i v

  let apply s v =
    List.fold_left (fun v (i, w) -> X.subst i w v) v (List.rev !s)
end

module type UNIFY = sig
  type t
  val unify: bool -> Env.t -> t -> t -> unit
  val subst: int -> t -> t -> t
  val occur_check: int -> t -> unit
  val expand: t -> t
end

module rec RSubst: SUBST with type v = Ast.region = Subst(Region)

and Region: UNIFY with type t = Ast.region = struct
  type t = Ast.region

  let subst = RSubst.create ()

  let rec unify sub env a b =
    log "unify %b (%a) (%a)@." sub Pp.region a Pp.region b;
    let () = match a, b with
(*      | RVar a, _ when sub && Env.get_region_parent env a = Some b -> ()*)
      | RVar a, RVar b when a = b -> ()
      | RUVar a, RUVar b when a = b -> ()
      | RUVar a, b | b, RUVar a -> RSubst.add env subst a b
      | RSub (a1, a2), RSub (b1, b2) when a2 = b2 -> unify sub env a1 b1
      | RVar _, RVar _
      | RSub (_, _), RSub (_, _)
      | RVar _, RSub _ | RSub _, RVar _ -> unification_error ()
    in
    log "ok@."

  let rec occur_check i = function
    | RUVar j when i = j -> unification_error ()
    | RVar _ | RUVar _ -> ()
    | RSub (r, _) -> occur_check i r

  let expand v =
    RSubst.apply subst v

  let rec subst i v = function
    | RUVar j when i = j -> v
    | RVar _ | RUVar _ as x -> x
    | RSub (r, x) -> RSub (subst i v r, x)
end

module rec TSubst: SUBST with type v = Ast.typ = Subst(Type)

and Type: UNIFY with type t = Ast.typ = struct
  type t = Ast.typ

  let subst = TSubst.create ()

  let rec unify sub env a b =
    let unify = unify sub env in
    let unify_regions = Region.unify sub env in
    match a, b with
      | TVar a, TVar b when a = b -> ()
      | TUVar a, TUVar b when a = b -> ()
      | TUVar a, b | b, TUVar a -> TSubst.add env subst a b
      | TTuple a, TTuple b -> List.iter2 unify a b
      | TBase a, TBase b when a = b -> ()
      | TPointer (rla, tla, ca, ra), TPointer (rlb, tlb, cb, rb) ->
          if ca <> cb then unification_error ();
          List.iter2 unify_regions rla rlb;
          List.iter2 unify tla tlb;
          unify_regions ra rb
      | TSum (a1, b1), TSum (a2, b2) ->
          unify a1 a2;
          unify b1 b2
      | _ -> unification_error ()

  let rec occur_check i = function
    | TUVar j when i = j -> unification_error ()
    | TVar _ | TUVar _ | TBase _ -> ()
    | TTuple l
    | TPointer (_, l, _, _) -> List.iter (occur_check i) l
    | TSum (a, b) ->
        occur_check i a;
        occur_check i b

  let expand v =
    TSubst.apply subst v

  let rec subst i v = function
    | TUVar j when i = j -> v
    | TVar _ | TUVar _ | TBase _ as x -> x
    | TTuple l -> TTuple (List.map (subst i v) l)
    | TSum (a, b) -> TSum (subst i v a, subst i v b)
    | TPointer (rl, tl, c, r) -> TPointer (rl, List.map (subst i v) tl, c, r)
end

let unify_regions loc sub env a b =
  try
    Region.unify sub env a b
  with Unification_error ->
    Loc.locate_error loc "Unification error (sub: %b): regions %a and %a"
      sub Pp.region a Pp.region b

let unify_types loc sub env a b =
  try
    Type.unify sub env a b
  with Unification_error ->
    Loc.locate_error loc "Unification error (sub: %b): types %a and %a"
      sub Pp.typ a Pp.typ b

(* strict *)
let rec is_sub_region env a b =
  match a, b with
    | RVar a, _ when Env.get_region_parent env a = Some b ->
        true
    | RSub (a1, a2), RSub (b1, b2) when a2 = b2 ->
        is_sub_region env a1 b1
    | _ ->
        false

let expand_region = Region.expand

let rec expand_regions_in_type = function
  | TUVar _ | TVar _ | TBase _ as x -> x
  | TTuple l -> TTuple (List.map expand_regions_in_type l)
  | TPointer (rl, tl, c, r) ->
      TPointer (
        List.map expand_region rl,
        List.map expand_regions_in_type tl,
        c,
        expand_region r
      )
  | TSum (a, b) -> TSum (expand_regions_in_type a, expand_regions_in_type b)

let expand_type t = expand_regions_in_type (Type.expand t)

let expand_permission = function
  | PEmpty r -> PEmpty (expand_region r)
  | POpen r -> POpen (expand_region r)
  | PClosed r -> PClosed (expand_region r)
  | PGroup r -> PGroup (expand_region r)
  | PArrow (s, r) -> PArrow (expand_region s, expand_region r)
