%{
(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

  open Parsing
  open Ast
  open Past

  let loc () =
    symbol_start_pos (), symbol_end_pos ()

  let node x = {
    loc = loc ();
    node = x;
  }

  let parser_error s =
    Printf.ksprintf (fun s -> raise (Misc.Parse_error (loc (), s))) s

  let region_binder x =
    match x.node with
      | RVar s -> s
      | _ -> parser_error "only region binders are allowed here"

  let type_binder x =
    match x.node with
      | TEPointer (([], [], s), None) -> s
      | _ -> parser_error "only type binders are allowed here"

  let class_expr_binder (rp, tp, n) =
    List.map region_binder rp, List.map type_binder tp, n

  let unit_if_empty = function
    | [] ->
        [ "_", node (TEPointer (([], [], "unit"), None)) ]
    | l ->
        l
%}

%token <string> IDENT
%token <string> STRING
%token <int> INT
%token CLASS OWN END VAL LET IN IF THEN ELSE WHILE DO NEW PACK UNPACK ADOPT AS
%token FOCUS UNFOCUS REGION LEFT RIGHT
%token CONSUMES PRODUCES
%token LPAR RPAR SEMI DOT EQUAL BANG COLONEQUAL LBRACKET RBRACKET COMMA EOF
%token INF SUP INFEQ SUPEQ COLON STAR LBRACE RBRACE VERT LTGT
%token PERMEMPTY PERMOPEN PERMCLOSED PERMGROUP ARROW BANG
%token TRUE FALSE
%token OR XOR AND NOT PLUS MINUS DIV MOD
%token PRINT BLACKBOX
%token PACKREGION UNPACKREGION ADOPTREGION UNFOCUSREGION
%token INVARIANT REQUIRES ENSURES
%token IMPLIES IFF FORALL EXISTS
%token LOGIC TYPE FUNCTION AXIOM ASSERT SELECTOR PREDICATE OLD LEMMA

%nonassoc quantifier_prec
%nonassoc solo_term
%right IMPLIES
%nonassoc IFF
%nonassoc IN
%nonassoc AS
%left SEMI
%nonassoc THEN DO PACK UNPACK COLONEQUAL
%nonassoc ELSE
%left AND
%left OR XOR
%left EQUAL
%left PLUS MINUS
%nonassoc INF SUP INFEQ SUPEQ
%left STAR VERT DIV MOD
%nonassoc PACKREGION UNPACKREGION ADOPTREGION UNFOCUSREGION
%left DOT
%nonassoc PRINT
%nonassoc NOT LEFT RIGHT
%nonassoc BANG

%type <Past.file> file
%type <Past.class_expr> class_expr
%start file
%%

file:
| class_def file
    { Class $1 :: $2 }
| value_def file
    { Value $1 :: $2 }
| logic_type_def file
    { LogicType $1 :: $2 }
| logic_function_def file
    { LogicFunction $1 :: $2 }
| axiom_def file
    { Axiom $1 :: $2 }
| selector_def file
    { Selector $1 :: $2 }
| predicate_def file
    { Predicate $1 :: $2 }
| EOF
    { [] }
;

selector_def:
| SELECTOR LPAR ident_comma_list RPAR
    { $3 }
| SELECTOR ident_comma_list
    { $2 }
;

class_def:
| CLASS class_expr EQUAL own_clause type_expr invariant END
    { let rp, tp, n = class_expr_binder $2 in
      {
        c_region_params = rp;
        c_type_params = tp;
        c_name = n;
        c_owned_regions = $4;
        c_type = $5;
        c_invariant = $6;
      } }
;

logic_type_def:
| LOGIC TYPE IDENT type_parameters
    { {
        lt_name = $3;
        lt_type_params = List.map type_binder $4;
      } }
;

logic_function_def:
| LOGIC FUNCTION IDENT type_parameters COLON type_expr
    { {
        lf_name = $3;
        lf_params = $4;
        lf_return_type = $6;
      } }
;

axiom_def:
| AXIOM IDENT COLON assertion
    { {
        ax_name = $2;
	ax_is_lemma = false;
        ax_assertion = $4;
      } }
| LEMMA IDENT COLON assertion
    { {
        ax_name = $2;
	ax_is_lemma = true;
        ax_assertion = $4;
      } }
;

predicate_def:
| PREDICATE IDENT LPAR argument_list RPAR EQUAL assertion
    { {
        p_name = $2;
        p_params = $4;
        p_body = Some $7;
      } }
| PREDICATE IDENT LPAR argument_list RPAR
    { {
        p_name = $2;
        p_params = $4;
        p_body = None;
      } }
;

invariant:
| INVARIANT LPAR IDENT RPAR EQUAL assertion
    { $3, $6 }
|
    { "_", node PTrue }
;

term:
| LBRACKET expr RBRACKET
    { $2 }
| IDENT
    { node (Var $1) }
| INT
    { node (Const (CInt $1)) }
| LPAR RPAR
    { node (Const CUnit) }
;

comma_term_star:
| COMMA term comma_term_star
    { $2 :: $3 }
|
    { [] }
;

term_comma_list:
| term comma_term_star
    { $1 :: $2 }
|
    { [] }
;

assertion:
| term %prec solo_term
    { node (PTerm $1) }
| assertion IMPLIES assertion
    { node (PImplies ($1, $3)) }
| assertion IFF assertion
    { node (PIff ($1, $3)) }
| assertion AND assertion
    { node (PAnd ($1, $3)) }
| assertion OR assertion
    { node (POr ($1, $3)) }
| NOT assertion
    { node (PNot $2) }
| term EQUAL term
    { node (PEqual ($1, $3)) }
| term LTGT term
    { node (PDiff ($1, $3)) }
| term INF term
    { node (PLt ($1, $3)) }
| term SUP term
    { node (PGt ($1, $3)) }
| term INFEQ term
    { node (PLe ($1, $3)) }
| term SUPEQ term
    { node (PGe ($1, $3)) }
| FORALL IDENT COLON type_expr DOT assertion %prec quantifier_prec
    { node (PForall ($2, $4, $6)) }
| EXISTS IDENT COLON type_expr DOT assertion %prec quantifier_prec
    { node (PExists ($2, $4, $6)) }
| TRUE
    { node PTrue }
| FALSE
    { node PFalse }
| LPAR assertion RPAR
    { $2 }
| IDENT LPAR term_comma_list RPAR
    { node (PApp ($1, $3)) }
| OLD LPAR assertion RPAR
    { node (POld $3) }
;

own_clause:
| OWN own_region_list SEMI
    { $2 }
|
    { [] }
;

own_region_binder:
| IDENT
    { $1, None }
| IDENT COLON class_expr
    { $1, Some $3 }
;

own_region_list:
| own_region_binder own_region_list_rem
    { $1 :: $2 }
;

own_region_list_rem:
| COMMA own_region_binder own_region_list_rem
    { $2 :: $3 }
|
    { [] }
;

ident_comma_list:
| IDENT comma_ident_star
    { $1 :: $2 }
;

comma_ident_star:
| COMMA IDENT comma_ident_star
    { $2 :: $3 }
|
    { [] }
;

value_def:
| VAL IDENT LPAR argument_list RPAR COLON type_expr
  consumes_clause produces_clause pre_clause post_clause optional_body
    { {
        v_name = $2;
        v_params = unit_if_empty $4;
        v_return_type = $7;
        v_consumes = $8;
        v_produces = $9;
        v_pre = $10;
        v_post = $11;
        v_body = $12;
      } }
;

argument_list:
| argument comma_argument_star
    { $1 :: $2 }
|
    { [] }
;

argument:
| IDENT COLON type_expr
    { $1, $3 }
;

comma_argument_star:
| COMMA argument comma_argument_star
    { $2 :: $3 }
|
    { [] }
;

consumes_clause:
| CONSUMES permission_comma_list
    { $2 }
|
    { [] }
;

produces_clause:
| PRODUCES permission_comma_list
    { $2 }
|
    { [] }
;

pre_clause:
| REQUIRES assertion
    { $2 }
|
    { node PTrue }
;

post_clause:
| ENSURES assertion
    { $2 }
|
    { node PTrue }
;

permission_comma_list:
| permission comma_permission_star
    { $1 :: $2 }
;

comma_permission_star:
| COMMA permission comma_permission_star
    { $2 :: $3 }
|
    { [] }
;

permission:
| region PERMEMPTY
    { PEmpty $1 }
| region PERMOPEN
    { POpen $1 }
| region PERMCLOSED
    { PClosed $1 }
| region PERMGROUP
    { PGroup $1 }
| region ARROW region
    { PArrow ($1, $3) }
| region INF region
    { PSub ($1, $3) }
;

region:
| IDENT
    { node (RVar $1) }
| region DOT IDENT
    { node (RSub ($1, $3)) }
;

optional_body:
| EQUAL expr
    { Some $2 }
|
    { None }
;

type_expr:
| type_expr VERT type_expr
    { node (TESum ($1, $3)) }
| LPAR type_expr star_type_star RPAR
    { match $3 with
        | [] -> $2
        | x -> node (TETuple ($2 :: x)) }
| class_expr LBRACKET region RBRACKET
    { node (TEPointer ($1, Some $3)) }
| class_expr
    { node (TEPointer ($1, None)) }
;

class_expr:
| IDENT
    { [], [], $1 }
| IDENT region_parameters
    { $2, [], $1 }
| IDENT type_parameters
    { [], $2, $1 }
| IDENT region_parameters type_parameters
    { $2, $3, $1 }
| IDENT type_parameters region_parameters
    { $3, $2, $1 }
;

region_parameters:
| LBRACE region comma_region_star RBRACE
    { $2 :: $3 }
;

comma_region_star:
| COMMA region comma_region_star
    { $2 :: $3 }
|
    { [] }
;

type_parameters:
| LPAR type_expr comma_type_star RPAR
    { $2 :: $3 }
| LPAR RPAR
    { [] }
;

star_type_star:
| STAR type_expr star_type_star
    { $2 :: $3 }
|
    { [] }
;

comma_type_star:
| COMMA type_expr comma_type_star
    { $2 :: $3 }
|
    { [] }
;

comma_expr_star:
| COMMA expr comma_expr_star
    { $2 :: $3 }
|
    { [] }
;

expr:
| LPAR RPAR
    { node (Const CUnit) }
| INT
    { node (Const (CInt $1)) }
| TRUE
    { node (Const (CBool true)) }
| FALSE
    { node (Const (CBool false)) }
| LPAR expr comma_expr_star RPAR
    { match $3 with
        | [] -> $2
        | x -> node (Tuple ($2 :: x)) }
| expr DOT INT
    { node (Proj ($1, SInt $3)) }
| expr DOT IDENT
    { node (Proj ($1, SString $3)) }
| IDENT
    { node (Var $1) }
| LET IDENT EQUAL expr IN expr
    { node (Let ($2, $4, $6)) }
| expr SEMI expr
    { node (Seq ($1, $3)) }
| IDENT LPAR RPAR
    { node (Call ($1, [])) }
| IDENT LPAR expr comma_expr_star RPAR
    { node (Call ($1, $3 :: $4)) }
| IF expr THEN expr
    { node (If ($2, $4, node (Const CUnit))) }
| IF expr THEN expr ELSE expr
    { node (If ($2, $4, $6)) }
| WHILE expr DO expr
    { node (While ($2, $4, node PTrue)) }
| WHILE expr INVARIANT assertion DO expr
    { node (While ($2, $6, $4)) }
| expr COLONEQUAL expr
    { node (Assign ($1, $3)) }
| BANG expr
    { node (Deref $2) }
| NEW class_expr LBRACKET region RBRACKET
    { node (New ($2, Some $4)) }
| NEW class_expr
    { node (New ($2, None)) }
| PACK expr
    { node (Pack $2) }
| UNPACK expr
    { node (Unpack $2) }
| ADOPT expr AS region
    { node (Adopt ($2, $4)) }
| FOCUS expr AS region
    { node (Focus ($2, Some $4)) }
| FOCUS expr %prec AS
    { node (Focus ($2, None)) }
| UNFOCUS expr AS region
    { node (Unfocus ($2, $4)) }
| REGION IDENT IN expr
    { node (Region ($2, $4, None)) }
| REGION IDENT COLON class_expr IN expr
    { node (Region ($2, $6, Some $4)) }
| expr PLUS expr
    { node (Binop (`add, $1, $3)) }
| expr MINUS expr
    { node (Binop (`sub, $1, $3)) }
| expr STAR expr
    { node (Binop (`mul, $1, $3)) }
| expr DIV expr
    { node (Binop (`div, $1, $3)) }
| expr MOD expr
    { node (Binop (`imod, $1, $3)) }
| expr AND expr
    { node (Binop (`band, $1, $3)) }
| expr OR expr
    { node (Binop (`bor, $1, $3)) }
| expr XOR expr
    { node (Binop (`bxor, $1, $3)) }
| expr INF expr
    { node (Binop (`lt, $1, $3)) }
| expr SUP expr
    { node (Binop (`gt, $1, $3)) }
| expr INFEQ expr
    { node (Binop (`le, $1, $3)) }
| expr SUPEQ expr
    { node (Binop (`ge, $1, $3)) }
| expr EQUAL expr
    { node (Binop (`eq, $1, $3)) }
| NOT expr
    { node (Unop (`bnot, $2)) }
| MINUS expr
    { node (Unop (`neg, $2)) }
| PRINT expr
    { node (Print ("user print", $2)) }
| PRINT STRING expr %prec PRINT
    { node (Print ($2, $3)) }
| LEFT expr
    { node (Left $2) }
| RIGHT expr
    { node (Right $2) }

| ADOPTREGION region AS region
    { node (AdoptRegion ($2, $4)) }
| UNFOCUSREGION region AS region
    { node (UnfocusRegion ($2, $4)) }
| PACKREGION region
    { node (PackRegion $2) }
| UNPACKREGION region
    { node (UnpackRegion $2) }

| BLACKBOX LPAR consumes_clause produces_clause RPAR
    { node (BlackBox ($3, $4)) }

| LPAR expr COLON type_expr RPAR
    { node (Typed ($2, $4)) }

| ASSERT assertion %prec PRINT
    { node (Assert $2) }

| OLD LPAR expr RPAR
    { node (Old $3) }
;
