(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

(** Parsed ASTs. *)

open Ast

type region = region_node node

and region_node =
  | RVar of string
  | RSub of region * string

type type_expr = type_node node

and type_node =
  | TETuple of type_expr list
  | TEPointer of class_expr * region option
  | TESum of type_expr * type_expr

and class_expr = region list * type_expr list * string

type permission = region Ast.permission

type selector =
  | SInt of int
  | SString of string

type expr = expr_node node

and expr_node =
  | Const of const
  | Unop of un_op * expr
  | Binop of bin_op * expr * expr
  | Tuple of expr list
  | Proj of expr * selector
  | Left of expr
  | Right of expr
  | Var of string
  | Let of string * expr * expr
  | Seq of expr * expr
  | Call of string * expr list
  | If of expr * expr * expr
  | While of expr * expr * assertion
  | Assign of expr * expr
  | Deref of expr
  | New of class_expr * region option
  | Pack of expr
  | Unpack of expr
  | Adopt of expr * region
  | Focus of expr * region option
  | Unfocus of expr * region
  | Region of string * expr * class_expr option
  | Print of string * expr

  | AdoptRegion of region * region
  | UnfocusRegion of region * region
  | PackRegion of region
  | UnpackRegion of region

  | BlackBox of permission list * permission list
  | Typed of expr * type_expr
  | Assert of assertion
  | Old of expr

(** We only use parts of the AST *)
and term = expr

and assertion = assertion_node node

and assertion_node =
  | PTrue
  | PFalse
  | PIff of assertion * assertion
  | PImplies of assertion * assertion
  | PAnd of assertion * assertion
  | POr of assertion * assertion
  | PNot of assertion
  | PEqual of term * term
  | PDiff of term * term
  | PLt of term * term
  | PGt of term * term
  | PLe of term * term
  | PGe of term * term
  | PTerm of term
  | PForall of string * type_expr * assertion
  | PExists of string * type_expr * assertion
  | PApp of string * term list
  | POld of assertion

type class_def = {
  c_region_params: string list;
  c_type_params: string list;
  c_name: string;
  c_owned_regions: (string * class_expr option) list;
  c_type: type_expr;
  c_invariant: string * assertion;
}

type value_def = {
  v_name: string;
  v_params: (string * type_expr) list;
  v_return_type: type_expr;
  v_consumes: permission list;
  v_produces: permission list;
  v_pre: assertion;
  v_post: assertion;
  v_body: expr option;
}

type logic_type_def = {
  lt_name: string;
  lt_type_params: string list;
}

type logic_function_def = {
  lf_name: string;
  lf_params: type_expr list;
  lf_return_type: type_expr;
}

type axiom_def = {
  ax_name: string;
  ax_is_lemma: bool;
  ax_assertion: assertion;
}

type predicate_def = {
  p_name: string;
  p_params: (string * type_expr) list;
  p_body: assertion option;
}

type def =
  | Class of class_def
  | Value of value_def
  | LogicType of logic_type_def
  | LogicFunction of logic_function_def
  | Axiom of axiom_def
  | Selector of string list
  | Predicate of predicate_def

type file = def list
