(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Ast
open Bast
open Format

let print_unique_identifiers = ref true

let expand_region_ref =
  ref (fun _ -> failwith "pp.ml: expand_region must be initialized by Unify")
let expand_region r = !expand_region_ref r

let nothing fmt () = ()
let char c fmt () = fprintf fmt "%c" c
let string s fmt () = fprintf fmt "%s" s
let space fmt () = fprintf fmt "@ "
let comma_space fmt () = fprintf fmt ",@ "
let todo s fmt () = fprintf fmt "(TODO: Pp.%s)" s

let list ?(sep = nothing) ?(left = nothing) ?(right = nothing) f fmt = function
  | [] -> ()
  | [x] ->
      left fmt ();
      f fmt x;
      right fmt ();
  | x :: rem ->
      left fmt ();
      f fmt x;
      List.iter
        (fun x ->
           sep fmt ();
           f fmt x)
        rem;
      right fmt ()

let option ?(left = nothing) ?(right = nothing) f fmt = function
  | None -> ()
  | Some x ->
      left fmt ();
      f fmt x;
      right fmt ()

let identifier fmt i =
  if !print_unique_identifiers then
    fprintf fmt "%s" (Ident.uname i)
  else
    fprintf fmt "%s" (Ident.name i)

let capitalized_identifier fmt i =
  if !print_unique_identifiers then
    fprintf fmt "%s" (String.capitalize (Ident.uname i))
  else
    fprintf fmt "%s" (String.capitalize (Ident.name i))

let rec region fmt r =
  let r = expand_region r in
  match r with
    | RVar id -> fprintf fmt "%a" identifier id
    | RSub (r, id) -> fprintf fmt "%a.%s" region r id

let permission fmt = function
  | PEmpty r ->
      fprintf fmt "%a^e" region r
  | POpen r ->
      fprintf fmt "%a^o" region r
  | PClosed r ->
      fprintf fmt "%a^c" region r
  | PGroup r ->
      fprintf fmt "%a^g" region r
  | PArrow (s, r) ->
      fprintf fmt "%a -> %a" region s region r
  | PSub (s, r) ->
      fprintf fmt "%a < %a" region s region r

let region_list_of_permission = function
  | PEmpty r
  | POpen r
  | PClosed r
  | PGroup r ->
      [r]
  | PArrow (s, r)
  | PSub (s, r) ->
      [r; s]

(* sort by region first *)
let compare_permissions a b =
  let c =
    compare
      (region_list_of_permission a)
      (region_list_of_permission b)
  in
  if c = 0 then
    compare a b
  else
    c

let permissions fmt caps =
  let caps = List.sort compare_permissions caps in
  list ~sep: (string ", ") permission fmt caps

let const fmt = function
  | CUnit ->
      fprintf fmt "()"
  | CInt i ->
      fprintf fmt "%d" i
  | CBool b ->
      fprintf fmt "%B" b

let op fmt o =
  let s = match o with
    | `add -> "+"
    | `sub -> "-"
    | `mul -> "*"
    | `div -> "/"
    | `imod -> "mod"
    | `neg -> "-"
    | `bor -> "or"
    | `band -> "and"
    | `bxor -> "xor"
    | `bnot -> "not"
    | `gt -> ">"
    | `lt -> "<"
    | `ge -> ">="
    | `le -> "<="
    | `eq -> "="
  in
  fprintf fmt "%s" s

let rec type_expr fmt e =
  let f x = fprintf fmt x in
  match e.node with
    | TETuple l ->
        f "@[<hv 2>(%a)@]" (list ~sep: comma_space type_expr) l
    | TEPointer (c, r) ->
        f "%a[%a]"
          class_expr c
          region r
    | TEIdent s ->
        f "%a" identifier s
    | TESum (a, b) ->
        f "@[<hv>(%a@ | %a)@]" type_expr a type_expr b
    | TEBase TInt ->
        f "int"
    | TEBase TBool ->
        f "bool"
    | TEBase TUnit ->
        f "unit"
    | TEBase TString ->
        f "string"
    | TELogic (tl, id) ->
        f "%a%a"
          identifier id
          (list ~left: (char '(') ~right: (char ')') ~sep: comma_space
             type_expr) tl

and class_expr fmt (rl, tl, s) =
  fprintf fmt "%a%a%a"
    identifier s
    (list ~left: (char '(') ~right: (char ')') ~sep: (string ", ")
       type_expr)
    tl
    (list ~left: (char '{') ~right: (char '}') ~sep: (string ", ")
       region)
    rl

let rec expr fmt e =
  let f x = fprintf fmt x in
  match e.node with
    | Const c ->
        const fmt c
    | Unop (o, a) ->
        f "(%a %a)" op o expr a
    | Binop (o, a, b) ->
        f "(@[<hv 2>%a@ %a@ %a@])" expr a op o expr b
    | Tuple l ->
        f "(@[<hv 2>%a@])" (list ~sep: comma_space expr) l
    | Proj (a, i, _) ->
        f "%a.%d" expr a i
    | Left a ->
        f "(@[<hv 2>Left@ %a@])" expr a
    | Right a ->
        f "(@[<hv 2>Right@ %a@])" expr a
    | Var s
    | LogicApp (s, [], _, _) ->
        f "%a" identifier s
    | Let (s, a, b) ->
        f "(@[<hv>@[<hv>@[<hv 2>let %a =@ %a@]@ in@]@ %a@])"
          identifier s expr a expr b
    | Seq (a, b) ->
        f "(@[<hv>%a;@ %a@])" expr a expr b
    | Call (s, l, _, _, _, _)
    | LogicApp (s, l, _, _) ->
        f "(@[<hv 2>%a(%a)@])" identifier s (list ~sep: comma_space expr) l
    | If (a, b, c) ->
        f "(@[<hv>if %a@ then %a@ else %a@])" expr a expr b expr c
    | While (a, b, c, _) ->
        f "(@[<hv 2>while %a@ invariant@ %a@ do@ %a@])"
          expr a assertion c expr b
    | Assign (a, b, _) ->
        f "(@[<hv 2>%a :=@ %a@])" expr a expr b
    | Deref (a, _, _, _) ->
        f "(!%a)" expr a
    | New ((rl, tl, s), r) ->
        f "(new %a%a%a[%a])" identifier s
          (list ~sep: comma_space ~left: (char '{') ~right: (char '}') region)
          rl
          (list ~sep: comma_space ~left: (char '(') ~right: (char ')')
             type_expr)
          tl
          region r
    | Pack (a, _) ->
        f "(@[<hv 2>pack@ %a@])" expr a
    | Unpack (a, _) ->
        f "(@[<hv 2>unpack@ %a@])" expr a
    | Adopt (a, r, _, _) ->
        f "(@[<hv 2>adopt@ %a@ as %a@])" expr a region r
    | Focus (a, r, _) ->
        f "(@[<hv 2>focus@ %a@ as %a@])" expr a region r
    | Unfocus (a, r, _) ->
        f "(@[<hv 2>unfocus@ %a@ as %a@])" expr a region r
    | Region (s, e, None) ->
        f "(@[<hv>region %a in@ %a@])" identifier s expr e
    | Region (s, e, Some c) ->
        f "(@[<hv>region %a:@ %a in@ %a@])" identifier s class_expr c expr e
    | Print (s, e) ->
        f "(@[<hv>print \"%s\" %a@])" (String.escaped s) expr e
    | AdoptRegion (a, b, _) ->
        f "(@[adoptregion %a as %a@])" region a region b
    | UnfocusRegion (a, b, _) ->
        f "(@[unfocusregion %a as %a@])" region a region b
    | PackRegion r ->
        f "(@[packregion %a@])" region r
    | UnpackRegion r ->
        f "(@[unpackregion %a@])" region r
    | BlackBox (c, p) ->
        f "(@[<hv 2>blackbox@ consumes %a@ produces %a@])"
          (list ~sep: comma_space permission) c
          (list ~sep: comma_space permission) p
    | WeakenRegion (r, WKEmptyToGroup) ->
        f "((* weaken %a (empty to group) *))" region r
    | WeakenRegion (r, WKClosedToGroup) ->
        f "((* weaken %a (closed to group) *))" region r
    | Typed (e, t) ->
        f "(@[<hv 2>%a:@ %a@])" expr e type_expr t
    | Assert (a, _) ->
        f "(@[<hv 2>assert@ %a@])" assertion a
    | Old t ->
        f "@[<hv 2>old(@,%a)@]" expr t

and term x = expr x

and assertion fmt a =
  let f x = fprintf fmt x in
  match a.node with
    | PTrue ->
        f "true"
    | PFalse ->
        f "false"
    | PIff (a, b) ->
        f "@[(%a <=>@ %a)@]" assertion a assertion b
    | PImplies (a, b) ->
        f "@[(%a ==>@ %a)@]" assertion a assertion b
    | PAnd (a, b) ->
        f "@[(%a and@ %a)@]" assertion a assertion b
    | POr (a, b) ->
        f "@[(%a or@ %a)@]" assertion a assertion b
    | PNot a ->
        f "~%a" assertion a
    | PEqual (a, b) ->
        f "@[(%a =@ %a)@]" term a term b
    | PDiff (a, b) ->
        f "@[(%a <>@ %a)@]" term a term b
    | PGt (a, b) ->
        f "@[(%a >@ %a)@]" term a term b
    | PLt (a, b) ->
        f "@[(%a <@ %a)@]" term a term b
    | PGe (a, b) ->
        f "@[(%a >=@ %a)@]" term a term b
    | PLe (a, b) ->
        f "@[(%a <=@ %a)@]" term a term b
    | PTerm a ->
        f "[%a]" term a
    | PForall (id, t, a) ->
        f "@[(forall %a: %a.@ %a)@]"
          identifier id
          type_expr t
          assertion a
    | PExists (id, t, a) ->
        f "@[(exists %a: %a.@ %a)@]"
          identifier id
          type_expr t
          assertion a
    | PApp (id, tl, _) ->
        f "@[<hv 2>%a(%a)@]"
          identifier id
          (list ~sep: comma_space term) tl
    | POld a ->
        f "@[<hv 2>old(@,%a)@]" assertion a

let class_def fmt x =
  let f x = fprintf fmt x in
  let iid, ia, _ = x.c_invariant in
  f "@[<hv 2>class %a%a%a =@ %a@ invariant(%a) =@ %a@ end@]@.@."
    identifier x.c_name
    (list ~left: (char '(') ~sep: comma_space ~right: (char ')') identifier)
    x.c_type_params
    (list ~left: (char '{') ~sep: comma_space ~right: (char '}') identifier)
    (List.map fst x.c_owned_regions)
    type_expr x.c_type
    identifier iid
    assertion ia

let value_def fmt x =
  let f x = fprintf fmt x in
  f "@[<hv 2>val @[<hv 2>%a(%a)@]: %a"
    identifier x.v_name
    (list ~sep: comma_space
       (fun _ (n, t) -> f "%a: %a" identifier n type_expr t))
    x.v_params
    type_expr x.v_return_type;
  begin match x.v_consumes with
    | [] -> ()
    | l -> f "@ consumes %a" (list ~sep: (string ", ") permission) l
  end;
  begin match x.v_produces with
    | [] -> ()
    | l -> f "@ produces %a" (list ~sep: (string ", ") permission) l
  end;
  begin match x.v_body with
    | None -> ()
    | Some b -> f " =@ %a" expr b
  end;
  f "@]@.@."

let logic_type_def fmt x =
  let f x = fprintf fmt x in
  f "@[<hv 2>logic type %a@ (@[<hov 2>%a@])@]@.@."
    identifier x.lt_name
    (list ~sep: comma_space identifier) x.lt_type_params

let logic_function_def fmt x =
  let f x = fprintf fmt x in
  f "@[<hv 2>logic function %a@ (@[<hov 2>%a@]):@ %a@]@.@."
    identifier x.lf_name
    (list ~sep: comma_space type_expr) x.lf_params
    type_expr x.lf_return_type

let axiom_def fmt x =
  let f x = fprintf fmt x in
  f "@[<hv 2>axiom %a:@ %a@]@.@."
    identifier x.ax_name
    assertion x.ax_assertion

let predicate_def fmt x =
  let f x = fprintf fmt x in
  f "@[<hv 2>predicate %a(%a)%a@]@.@."
    identifier x.p_name
    (list ~sep: comma_space
       (fun _ (n, t) -> f "%a: %a" identifier n type_expr t))
    x.p_params
    (option ~left: (fun fmt () -> fprintf fmt ":@ ") assertion) x.p_body

let def fmt = function
  | Class x -> class_def fmt x
  | Value x -> value_def fmt x
  | LogicType x -> logic_type_def fmt x
  | LogicFunction x -> logic_function_def fmt x
  | Axiom x -> axiom_def fmt x
  | Predicate x -> predicate_def fmt x

let file fmt f =
  List.iter (def fmt) f
