(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

(** Type a bound AST:
    - compute its type,
    - modify it to insert some subregion constraints. *)

open Misc
open Ast
open Bast
open Asttools
open Typing_common
open Effects

(** Check that pointer and logic types are applied to the right number
    of arguments.
    Also, call [Regiontypes.add] on pointers which are in region variables. *)
let rec check_type env t =
  match t.node with
    | TEIdent _
    | TEBase _ ->
        ()
    | TESum (a, b) ->
        check_type env a;
        check_type env b
    | TETuple l ->
        List.iter (check_type env) l
    | TEPointer ((rl, tl, c), r) ->
        let cd = find t.loc env.classes c in
        let len = List.length rl in
        let elen = List.length cd.c_region_params in
        if len <> elen then
          Loc.locate_error t.loc "type %a expects %d region argument(s), \
but is here applied to %d" Pp.identifier cd.c_name elen len;
        let len = List.length tl in
        let elen = List.length cd.c_type_params in
        if len <> elen then
          Loc.locate_error t.loc "type %a expects %d type argument(s), \
but is here applied to %d" Pp.identifier cd.c_name elen len;
        List.iter (check_type env) tl;
        Regiontypes.add_if_var r (rl, tl, c)
    | TELogic (tl, id) ->
        let lt = find t.loc env.logic_types id in
        let len = List.length tl in
        let elen = List.length lt.lt_type_params in
        if len <> elen then
          Loc.locate_error t.loc "logic type %a expects %d argument(s), \
but is here applied to %d" Pp.identifier lt.lt_name elen len;
        List.iter (check_type env) tl

(** Insert a black box at the end of the expression, requiring that
    region [s] is a subregion of [r]. *)
(* One could think that the adoption should be done before. But that is not
   the case. We type an expression, it is well-typed, and now we want to
   use the expression with another type, so we need to do some adoptions. *)
let apply_constraint x (s, r) =
  let p = [ PSub (s, r) ] in
  let bb = { x with node = BlackBox (p, p) } in
  { x with node = seqf x bb }

(** Return a fresh unifiable type variable. *)
let fresh_utv r =
  make_node (TEIdent (Ident.create ~unifiable: true r))

(** Return a fresh unifiable region variable. *)
let fresh_urv r =
  RVar (Ident.create ~unifiable: true r)

let const = function
  | CUnit -> tbase TUnit
  | CInt _ -> tbase TInt
  | CBool _ -> tbase TBool

let op = function
  | #int_op -> tbase TInt
  | #bool_op -> tbase TBool

let check_pointer loc t =
  match t.node with
    | TEPointer _ ->
        ()
    | _ ->
        Loc.locate_error loc "this should be a pointer"

let class_expr loc t =
  match t.node with
    | TEPointer (ce, _) ->
        ce
    | _ ->
        Loc.locate_error loc "this should be a pointer"

let rec instanciate_region rs = function
  | RVar id -> List.assoc id rs
  | RSub (r, x) -> RSub (instanciate_region rs r, x)

let instanciate_permission rs = function
  | PEmpty r -> PEmpty (instanciate_region rs r)
  | POpen r -> POpen (instanciate_region rs r)
  | PClosed r -> PClosed (instanciate_region rs r)
  | PGroup r -> PGroup (instanciate_region rs r)
  | PArrow (s, r) -> PArrow (instanciate_region rs s, instanciate_region rs r)
  | PSub (s, r) -> PSub (instanciate_region rs s, instanciate_region rs r)

let rec instanciate_type rs ts t =
  let node = match t.node with
    | TEIdent id as x ->
        begin try (List.assoc id ts).node with Not_found -> x end
    | TEBase _ as x ->
        x
    | TETuple x ->
        TETuple (List.map (instanciate_type rs ts) x)
    | TESum (a, b) ->
        TESum (instanciate_type rs ts a, instanciate_type rs ts b)
    | TEPointer ((rl, tl, c), r) ->
        TEPointer (
          (List.map (instanciate_region rs) rl,
           List.map (instanciate_type rs ts) tl,
           c),
          instanciate_region rs r
        )
    | TELogic (tl, id) ->
        TELogic (List.map (instanciate_type rs ts) tl, id)
  in
  { t with node = node }

let instanciate_type_variables =
  List.map (fun t -> t, fresh_utv ("_itv_" ^ Ident.name t))

let instanciate_region_variables =
  List.map (fun r -> r, fresh_urv ("_irv_" ^ Ident.name r))

let instanciate_value v =
  let rs =
    List.map (fun r -> r, fresh_urv ("_ivr_" ^ Ident.name r)) v.v_regions in
  let ts = instanciate_type_variables v.v_type_variables in
  let instanciate_type = instanciate_type rs ts in
  let instanciate_permission = instanciate_permission rs in
  List.map (fun (_, t) -> instanciate_type t) v.v_params,
  instanciate_type v.v_return_type,
  List.map instanciate_permission v.v_consumes,
  List.map instanciate_permission v.v_produces,
  rs

(* Return (param types, region params, return type) *)
let instanciate_logic_function lf =
  let rs = instanciate_region_variables lf.lf_region_variables in
  let ts = instanciate_type_variables lf.lf_type_variables in
  let instanciate_type = instanciate_type rs ts in
  List.map instanciate_type lf.lf_params,
  List.map (fun x -> List.assoc x rs) lf.lf_region_variables,
  instanciate_type lf.lf_return_type

(* Return (param types, region params) *)
let instanciate_predicate p =
  let rs = instanciate_region_variables p.p_region_variables in
  let ts = instanciate_type_variables p.p_type_variables in
  let instanciate_type = instanciate_type rs ts in
  List.map instanciate_type (List.map snd p.p_params),
  List.map (fun x -> List.assoc x rs) p.p_region_variables

let region_of loc t =
  match t.node with
    | TEPointer (_, r) ->
        r
    | _ ->
        Loc.locate_error loc "this should be a pointer"

let add_list = Bind.add_list
let add_list2 env l = List.fold_left (fun a (b, c) -> Env.add a b c) env l

(** Return [(t, e)] where [t] is the type and [e] the modified expression. *)
let rec expr actually_a_term env x =
  let expr x = expr actually_a_term x in
  let expr_as x = expr_as actually_a_term x in
  let return typ node = typ, { node = node; loc = x.loc } in
  match x.node with
    | Const c ->
        return (const c) (Const c)
    | Unop (o, a) ->
        let t = op o in
        return t (Unop (o, expr_as env a t))
    | Binop (#cmp_op as o, a, b) ->
        let t = fresh_utv "cmp_type" in
        let a = expr_as env a t in
        let b = expr_as env b t in
        return (tbase TBool) (Binop (o, a, b))
    | Binop (#int_bin_op | #bool_bin_op as o, a, b) ->
        let t = op o in
        return t (Binop (o, expr_as env a t, expr_as env b t))
    | Tuple l ->
        let tl, el = List.split (List.map (expr env) l) in
        return (ttuple tl) (Tuple el)
    | Proj (a, i, _) ->
        if i > 0 then
          begin match expr env a with
            | { node = TETuple tl }, a ->
                let len = List.length tl in
                if len >= i then
                  return (List.nth tl (i - 1)) (Proj (a, i, len))
                else
                  Loc.locate_error a.loc
                    "this tuple have only %d components" (List.length tl)
            | _ ->
                Loc.locate_error a.loc "this should be a tuple"
          end
        else
          Loc.locate_error x.loc "projection indices start from 1"
    | Left a ->
        let t, a = expr env a in
        return (tsum t (fresh_utv "_left")) (Left a)
    | Right a ->
        let t, a = expr env a in
        return (tsum (fresh_utv "_right") t) (Right a)
    | Var v ->
        return (find x.loc env.variables v) (Var v)
    | Let (v, a, b) ->
        let ta, a = expr env a in
        let env = { env with variables = Env.add env.variables v ta } in
        let tb, b = expr env b in
        return tb (Let (v, a, b))
    | Seq (a, b) ->
        let a = expr_as env a (tbase TUnit) in
        let tb, b = expr env b in
        return tb (Seq (a, b))
    | Call (id, l, _, _, _, _) ->
        let v = find x.loc env.values id in
        let (ptl, rt, c, p, subst) = instanciate_value v in
        let elen = List.length ptl in
        let len = List.length l in
        if elen <> len then
          Loc.locate_error x.loc "function %a expects %d arguments, \
but is here applied to %d" Pp.identifier v.v_name elen len;
        let pl = List.map2 (expr_as env) l ptl in
        return rt (Call (id, pl, c, p, subst, []))
    | LogicApp (id, l, _, _) ->
        let lf = find x.loc env.logic_functions id in
        let ptl, region_subst, rt = instanciate_logic_function lf in
        let elen = List.length ptl in
        let len = List.length l in
        if elen <> len then
          Loc.locate_error x.loc "logic function %a expects %d arguments, \
but is here applied to %d" Pp.identifier lf.lf_name elen len;
        let pl = List.map2 (expr_as env) l ptl in
        return rt (LogicApp (id, pl, region_subst, []))
    | If (a, b, c) ->
        let a = expr_as env a (tbase TBool) in
        let tb, b = expr env b in
        let c = expr_as env c tb in
        return tb (If (a, b, c))
    | While (a, b, i, _) ->
        let a = expr_as env a (tbase TBool) in
        let b = expr_as env b (tbase TUnit) in
        let i = assertion env i in
        return (tbase TUnit) (While (a, b, i, []))
    | Assign (a, b, _) ->
        let ta, a = expr env a in
        let b = expr_as env b (pointed_type x.loc env ta) in
        return (tbase TUnit) (Assign (a, b, region_of a.loc ta))
    | Deref (a, _, _, _) ->
        let ta, a = expr env a in
        return
          (pointed_type x.loc env ta)
          (Deref (a, region_of a.loc ta, DKUnavailable, []))
    | New ((rl, tl, v), r) ->
        let t = tpointer rl tl v r in
        check_type env t;
        return t (New ((rl, tl, v), r))
    | Pack (a, _) ->
        let ta, a = expr env a in
        check_pointer a.loc ta;
        return (tbase TUnit) (Pack (a, region_of a.loc ta))
    | Unpack (a, _) ->
        let ta, a = expr env a in
        check_pointer a.loc ta;
        return (tbase TUnit) (Unpack (a, region_of a.loc ta))
    | Adopt (a, r, _, ak) ->
        let ta, a = expr env a in
        let rl, tl, c = class_expr a.loc ta in
        return (tpointer rl tl c r) (Adopt (a, r, region_of a.loc ta, ak))
    | Focus (a, r, _) ->
        let ta, a = expr env a in
        let rl, tl, c = class_expr a.loc ta in
        Regiontypes.add_if_var r (rl, tl, c);
        return (tpointer rl tl c r) (Focus (a, r, region_of a.loc ta))
    | Unfocus (a, r, _) ->
        let ta, a = expr env a in
        check_pointer a.loc ta;
        return (tbase TUnit) (Unfocus (a, r, region_of a.loc ta))
    | Region (v, a, ceo) ->
        let env = { env with regions = Env.add env.regions v () } in
        let ta, a = expr env a in
        return ta (Region (v, a, ceo))
    | Print (s, a) ->
        let ta, a = expr env a in
        return ta (Print (s, a))
    | AdoptRegion (s, r, ak) ->
        return (tbase TUnit) (AdoptRegion (s, r, ak))
    | UnfocusRegion (s, r, kind) ->
        return (tbase TUnit) (UnfocusRegion (s, r, kind))
    | PackRegion r ->
        return (tbase TUnit) (PackRegion r)
    | UnpackRegion r ->
        return (tbase TUnit) (UnpackRegion r)
    | BlackBox (c, p) ->
        return (tbase TUnit) (BlackBox (c, p))
    | WeakenRegion _ ->
        assert false (* the user cannot write a weakening annotation *)
    | Typed (e, t) ->
        let te = expr_as env e t in
        return t (Typed (te, t))
    | Assert (a, _) ->
        return (tbase TUnit) (Assert (assertion env a, []))
    | Old a ->
        let ta, a = expr env a in
        return ta (Old a)        

and expr_as actually_a_term env x et =
  match x.node with
    | Region (r, a, t) ->
        { x with node = Region (r, expr_as actually_a_term env a et, t) }
    | _ ->
        let t, x = expr actually_a_term env x in
        let constraints = Unify.unify_types x.loc true t et in
        if actually_a_term then
          x
        else
          List.fold_left apply_constraint x constraints

and term_as env t ty =
  expr_as true env t ty

and term env t =
  let ty = fresh_utv "term_type" in
  term_as env t ty, ty

and assertion env a =
  let node = match a.node with
    | PTrue
    | PFalse as x ->
        x
    | PIff (a, b) ->
        PIff (assertion env a, assertion env b)
    | PImplies (a, b) ->
        PImplies (assertion env a, assertion env b)
    | PAnd (a, b) ->
        PAnd (assertion env a, assertion env b)
    | POr (a, b) ->
        POr (assertion env a, assertion env b)
    | PNot a ->
        PNot (assertion env a)
    | PEqual (a, b) ->
        let t, ty = term env a in
        PEqual (t, term_as env b ty)
    | PDiff (a, b) ->
        let t, ty = term env a in
        PDiff (t, term_as env b ty)
    | PLt (a, b) ->
        let t, ty = term env a in
        PLt (t, term_as env b ty)
    | PGt (a, b) ->
        let t, ty = term env a in
        PGt (t, term_as env b ty)
    | PLe (a, b) ->
        let t, ty = term env a in
        PLe (t, term_as env b ty)
    | PGe (a, b) ->
        let t, ty = term env a in
        PGe (t, term_as env b ty)
    | PTerm a ->
        PTerm (term_as env a (make_node (TEBase TBool)))
    | PForall (id, t, a) ->
        check_type env t;
        let env = { env with variables = Env.add env.variables id t } in
        PForall (id, t, assertion env a)
    | PExists (id, t, a) ->
        check_type env t;
        let env = { env with variables = Env.add env.variables id t } in
        PExists (id, t, assertion env a)
    | PApp (id, l, _) ->
        let p = find a.loc env.predicates id in
        let ptl, region_subst = instanciate_predicate p in
        let elen = List.length ptl in
        let len = List.length l in
        if elen <> len then
          Loc.locate_error a.loc "predicate %a expects %d arguments, \
but is here applied to %d" Pp.identifier p.p_name elen len;
        let pl = List.map2 (term_as env) l ptl in
        PApp (id, pl, region_subst)
    | POld a ->
        POld (assertion env a)
  in
  { a with node = node }

let expr_as = expr_as false
let expr = expr false

let invariant env (x, a, _) t =
  let env = { env with variables = Env.add env.variables x t } in
  let a = assertion env a in
  let reads = RegionSet.elements (assertion_reads a) in
  x, a, reads

let value_def env x =
  let env = { env with values = Env.add env.values x.v_name x } in
  let ienv =
    { env with
        type_variables = add_list env.type_variables x.v_type_variables;
        regions = add_list env.regions x.v_regions;
        variables = add_list2 env.variables x.v_params }
  in
  List.iter (fun (_, t) -> check_type ienv t) x.v_params;
  check_type ienv x.v_return_type;
  let body =
    match x.v_body with
      | None -> None
      | Some body -> Some (expr_as ienv body x.v_return_type)
  in
  let ienv_post =
    { ienv with variables = Env.add ienv.variables x.v_result x.v_return_type }
  in
  env,
  { x with
      v_body = body;
      v_pre = assertion ienv x.v_pre;
      v_post = assertion ienv_post x.v_post }

let class_def env x =
  let env = { env with classes = Env.add env.classes x.c_name x } in
  check_type env x.c_type;
  let x = { x with c_invariant = invariant env x.c_invariant x.c_type } in
  let env = { env with classes = Env.replace env.classes x.c_name x } in
  env, x

let logic_type_def env x =
  let env = { env with logic_types = Env.add env.logic_types x.lt_name x } in
  env, x

let logic_function_def env x =
  let ienv =
    { env with
        type_variables = add_list env.type_variables x.lf_type_variables;
        regions = add_list env.regions x.lf_region_variables }
  in
  List.iter (check_type ienv) x.lf_params;
  check_type ienv x.lf_return_type;
  let env =
    { env with logic_functions = Env.add env.logic_functions x.lf_name x } in
  env, x

let axiom_def env x =
  let ienv =
    { env with
        type_variables = add_list env.type_variables x.ax_type_variables;
        regions = add_list env.regions x.ax_region_variables }
  in
  env,
  { x with ax_assertion = assertion ienv x.ax_assertion }

let predicate_def env x =
  log "Typing predicate %a@." Pp.identifier x.p_name;
  let ienv =
    { env with
        type_variables = add_list env.type_variables x.p_type_variables;
        regions = add_list env.regions x.p_region_variables }
  in
  List.iter (check_type ienv) (List.map snd x.p_params);
  let ienv =
    { ienv with variables = add_list2 env.variables x.p_params } in
  { env with predicates = Env.add env.predicates x.p_name x },
  { x with p_body = Opt.map (assertion ienv) x.p_body }

let def env = function
  | Class x -> let env, x = class_def env x in env, Class x
  | Value x -> let env, x = value_def env x in env, Value x
  | LogicType x -> let env, x = logic_type_def env x in env, LogicType x
  | LogicFunction x ->
      let env, x = logic_function_def env x in env, LogicFunction x
  | Axiom x -> let env, x = axiom_def env x in env, Axiom x
  | Predicate x -> let env, x = predicate_def env x in env, Predicate x

let file ?(env = empty_env) x =
  list_fold_map def env x
