(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

val use_boron: bool -> unit

type logic_type

val logic_type:
  ?args: logic_type list ->
  Ident.t ->
  logic_type
val type_variable: Ident.t -> logic_type
val unit: logic_type
val int: logic_type
val bool: logic_type
val string: logic_type
val tuple: logic_type list -> logic_type

type term

val at_old: Ident.t
val at_init: Ident.t

val term_void: term
val term_ints: string -> term
val term_int: int -> term
val term_reals: string -> term
val term_real: float -> term
val term_bool: bool -> term
val term_app: Ident.t -> term list -> term
val term_var: Ident.t -> term
val term_deref: ?at: Ident.t -> Ident.t -> term
val term_if: term -> term -> term -> term
val term_result: term
val term_tuple: term list -> term
val term_proj: int -> int -> term -> term
val term_let: Ident.t -> term -> term -> term

val term_and: term -> term -> term
val term_or: term -> term -> term
val term_xor: term -> term -> term
val term_add: term -> term -> term
val term_sub: term -> term -> term
val term_mul: term -> term -> term
val term_div: term -> term -> term
val term_mod: term -> term -> term
val term_not: term -> term
val term_neg: term -> term
val term_gt: term -> term -> term
val term_lt: term -> term -> term
val term_ge: term -> term -> term
val term_le: term -> term -> term
val term_eq: term -> term -> term

type assertion

val a_bool: bool -> assertion
val a_true: assertion
val a_false: assertion
val a_app: Ident.t -> term list -> assertion
val a_and: assertion -> assertion -> assertion
val a_or: assertion -> assertion -> assertion
val a_iff: assertion -> assertion -> assertion
val a_implies: assertion -> assertion -> assertion
val a_not: assertion -> assertion
val a_if: term -> assertion -> assertion -> assertion
val a_let: Ident.t -> term -> assertion -> assertion
val a_forall: Ident.t -> logic_type -> assertion -> assertion
val a_exists: Ident.t -> logic_type -> assertion -> assertion
val a_eq: term -> term -> assertion
val a_neq: term -> term -> assertion
val a_gt: term -> term -> assertion
val a_lt: term -> term -> assertion
val a_ge: term -> term -> assertion
val a_le: term -> term -> assertion

type typ

val type_base: logic_type -> typ
val type_ref: typ -> typ
val type_fun:
  arg_name: Ident.t ->
  arg_type: typ ->
  typ ->
  typ
val type_annot:
  ?pre: assertion ->
  ?reads: Ident.t list ->
  ?writes: Ident.t list ->
  ?post: assertion ->
  ?exceptions: (Ident.t * assertion) list ->
  typ ->
  typ

type expr

(*
(** result is [false] implies the expression have no side effect *)
val may_have_side_effects: expr -> bool
val expr_of_term: term -> expr
*)

val const_void: expr
val const_ints: string -> expr
val const_int: int -> expr
val const_reals: string -> expr
val const_real: float -> expr
val const_bool: bool -> expr
val var: Ident.t -> expr

val expr_and: expr -> expr -> expr
val expr_or: expr -> expr -> expr
val expr_xor: expr -> expr -> expr
val expr_add: expr -> expr -> expr
val expr_sub: expr -> expr -> expr
val expr_mul: expr -> expr -> expr
val expr_div: expr -> expr -> expr
val expr_mod: expr -> expr -> expr
val expr_not: expr -> expr
val expr_neg: expr -> expr
val expr_gt: expr -> expr -> expr
val expr_lt: expr -> expr -> expr
val expr_ge: expr -> expr -> expr
val expr_le: expr -> expr -> expr
val expr_eq: expr -> expr -> expr

val deref: Ident.t -> expr
val expr_if: expr -> expr -> expr -> expr
val loop: expr -> ?invariant: assertion -> ?variant: (term * Ident.t option) ->
  expr -> expr
val seq: expr list -> expr
val seq2: expr -> expr -> expr
val app: expr -> expr -> expr
val app_list: expr -> expr list -> expr
val expr_raise: ?arg: expr -> Ident.t -> expr
val expr_try: expr -> Ident.t -> ?arg: Ident.t -> expr -> expr
val expr_fun: ?args: (Ident.t * typ) list -> ?pre: assertion ->
  ?post: assertion -> ?exceptions: (Ident.t * assertion) list -> expr -> expr
val expr_assert: assertion -> expr
val label: Ident.t -> expr -> expr
val black_box: typ -> expr
val absurd: expr
val expr_let: Ident.t -> expr -> expr -> expr
val assign: Ident.t -> expr -> expr
val expr_ignore: expr -> expr -> expr
val expr_tuple: expr list -> expr
val expr_proj: int -> int -> expr -> expr (** [expr_proj count index expr] *)

val assume: assertion -> expr

type decl

val decl_logic_type: ?args: Ident.t list -> Ident.t -> decl
val decl_logic: Ident.t -> ?args: logic_type list -> logic_type -> decl
val decl_parameter: Ident.t -> typ -> decl
val decl_let: Ident.t -> expr -> decl
val decl_predicate: Ident.t -> (Ident.t * logic_type) list ->
  assertion option -> decl
val decl_axiom: is_lemma:bool -> Ident.t -> assertion -> decl

type file

val file: decl list -> file

val to_channel: out_channel -> file -> unit
val to_file: string -> file -> unit
