(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Tast
open Misc
open Types
open Pp
open Why

(* interp common *)

module IMap = Map.Make(Ident)
module ISet = Set.Make(Ident)

let dloc = Lang.dummy_location

let find_class_decl env id =
  try
    Env.find env.classes id
  with Not_found ->
    error "Interp.find_class_decl %a raised Not_found" Ident.pp id

let find_fun_decl env id =
  try
    Env.find env.functions id
  with Not_found ->
    error "Interp.find_fun_decl %a raised Not_found" Ident.pp id

(******************************************************************************)

let region_get_name =
  imap_memo (fun field_name -> Ident.create (Ident.name field_name))

let region_set_name =
  imap_memo
    (fun field_name -> Ident.create ("set_" ^ Ident.name field_name))

(* proj -> tuple region -> pointer -> proj result *)
let expr_get f r p =
  Why.app (Why.var (region_get_name f)) (Whylib.expr_get r p)

let term_get f r p =
  Why.term_app (region_get_name f) [ Whylib.term_get r p ]

let expr_set f r p v =
  let v =
    Why.app (Why.app (Why.var (region_set_name f)) (Whylib.expr_get r p)) v
  in
  Whylib.expr_set r p v

(*
let term_set f r p v =
  let v = term_app (region_set_name f) [ Whylib.term_get r p; v ] in
  Whylib.term_set r p v
*)

(******************************************************************************)

let get_path expr_get deref var (root, path) =
  List.fold_left
    (fun acc (v, perm, own) ->
      match perm with
        | PPNone ->
            type_error dloc "expr_get_path: no permission (var %a, own %a)"
              pp_ident v pp_ident own
        | PPFocus ->
            type_error dloc
              "expr_get_path: focus permission (cannot read in focus yet)"
        | PPEmpty
        | PPGroup
        | PPSingle ->
            expr_get own acc (var v))
    (deref root)
    (List.rev path)

let get_path_full expr_get deref var path x f =
  let path_to_x = get_path expr_get deref var path in
  expr_get f path_to_x x

let expr_get_path = get_path expr_get deref var

let rec expr_set_path root path k =
  match path with
    | [] ->
        k root
    | (x, (PPSingle | PPGroup | PPFocus | PPEmpty), r) :: rem ->
        let new_root = expr_get r root (var x) in
        expr_set r root (var x)
          (expr_set_path (new_root) rem k)
    | (_, PPNone, _) :: _ ->
        type_error dloc "expr_set_path: no permission"

let expr_set_path_full root path v field value =
  expr_set_path root path
    (fun root -> expr_set field root v value)
