(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Pp
open Lang
open Misc
open Lang_ast
open Tast
open Types

module P = Lang_ast
module T = Tast

(*
let debug s =
  T.IPack (Ident.create s)
*)

let check_argument_count loc a b =
  if List.length a <> List.length b then
    type_error loc "wrong argument count"

let result = Ident.create "result"

(******************************************************************************)

let string_of_pident i = match i.node with Ident s -> s

let lookup text env id =
  let sid = string_of_pident id in
  try
    Env.lookup_and_find env sid
  with Not_found ->
    type_error id.loc "unbound %s: %s" text sid

let lookup_poly_var env id =
  lookup "polymorphic type variable" env.poly_type_variables id

let lookup_var env id =
  lookup "variable" env.variables id

let lookup_region env id =
  lookup "region" env.regions id

let lookup_function env id =
  lookup "function" env.functions id

let lookup_type env id =
  lookup "type" env.types id

let lookup_logic_function env id =
  lookup "logic function" env.logic_functions id

let lookup_predicate env id =
  lookup "predicate" env.predicates id

let lookup_owned decl name =
  let sname = string_of_pident name in
  try
    List.find (fun (id, _) -> Ident.name id = sname) decl.c_singles
  with Not_found ->
    try
      List.find (fun (id, _) -> Ident.name id = sname) decl.c_groups
    with Not_found ->
      type_error name.loc "unbound owned region: %s in class %a"
        sname Ident.pp decl.c_name

(******************************************************************************)

let make_subst loc a b =
  try
    List.fold_left2
      (fun acc a b -> IMap.add a b acc)
      IMap.empty
      a
      b
  with _ ->
    type_error loc "invalid number of arguments"

let subst_region loc sreg svar r =
  match URegion.find r with
    | RVar ->
        type_error loc "cannot substitute in a non-unified region (%a)"
          pp_region r
    | RRoot name ->
        begin
          try
            IMap.find name sreg
          with Not_found ->
            r
        end
    | ROwn (v, o) ->
        begin
          try
            match IMap.find v svar with
              | None ->
                  type_error loc "cannot substitute %a because you did not use \
a variable in the call" pp_region r
              | Some v' ->
                  URegion.create (ROwn (v', o))
          with Not_found ->
            r
        end

let subst_permission loc sreg svar p =
  let subst = subst_region loc sreg svar in
  match p with
    | PEmpty r -> PEmpty (subst r)
    | POpen r -> POpen (subst r)
    | PClosed r -> PClosed (subst r)
    | PGroup r -> PGroup (subst r)
    | PFocus (s, r) -> PFocus (subst s, subst r)

(* svar is only used with function calls *)
let rec subst_type loc sreg styp svar t =
  match UType.find t with
    | TVar ->
        type_error loc "cannot substitute in a non-unified type (%a)"
          pp_type t
    | TPointer r ->
        UType.create (TPointer (subst_region loc sreg svar r))
    | TLogicPointer ->
        UType.create TLogicPointer
    | TBase _ ->
        t
    | TPoly name ->
        begin
          try
            IMap.find name styp
          with Not_found ->
            t
        end
    | TLogic (n, r, l) ->
        let r = List.map (subst_region loc sreg svar) r in
        let l = List.map (subst_type loc sreg styp svar) l in
        UType.create (TLogic (n, r, l))

let subst_class_expr loc sreg styp svar (name, regions, types) =
  let regions = List.map (subst_region loc sreg svar) regions in
  let types = List.map (subst_type loc sreg styp svar) types in
  name, regions, types

let decl_of_class_expr loc env (name, _, _) =
  try
    Env.find env.classes name
  with Not_found ->
    type_error loc "unbound class: %a"
      Ident.pp name

(* owner: variable identifier *)
let subst_of_class_expr loc env ((_, regions, types) as ce: class_expr) owner =
  let decl = decl_of_class_expr loc env ce in
  let subst_regions =
    make_subst
      loc
      (List.map fst decl.c_region_parameters)
      regions
  in
  (* substitute owned regions *)
  let subst_regions =
    List.fold_left
      (fun acc (id, _) ->
        let region = URegion.create (ROwn (owner, id)) in
        IMap.add id region acc)
      subst_regions
      (decl.c_singles @ decl.c_groups)
  in
  let subst_types =
    make_subst
      loc
      decl.c_type_parameters
      types
  in
  subst_regions, subst_types

let field_of_class_expr env ((name, _, _) as ce) field owner =
  let loc = field.loc in
  let field = string_of_pident field in
  let cd = decl_of_class_expr loc env ce in
  let sreg, styp = subst_of_class_expr loc env ce owner in
  let svar = IMap.empty in
  let field, field_type_before_subst =
    try
      List.find (fun (id, _) -> Ident.name id = field) cd.c_fields
    with Not_found ->
      type_error loc "unbound field: %s in class %a" field Ident.pp name
  in
  let field_type = subst_type loc sreg styp svar field_type_before_subst in
  field, field_type

let find_owned loc env class_expr owned f =
  let name, _, _ = class_expr in
  let decl = decl_of_class_expr loc env class_expr in
  try
    f decl.c_singles
  with Not_found ->
    try
      f decl.c_groups
    with Not_found ->
      type_error loc "unbound owned region: %s in class %a"
        owned Ident.pp name

let class_expr_of_owned_before_subst loc env class_expr owned =
  find_owned loc env class_expr (Ident.name owned) (List.assoc owned)

let class_expr_of_owned loc env class_expr owned owner =
  let before_subst =
    class_expr_of_owned_before_subst loc env class_expr owned
  in
  let subst_regions, subst_types =
    subst_of_class_expr loc env class_expr owner
  in
  let subst_vars = IMap.empty in
  subst_class_expr
    loc
    subst_regions
    subst_types
    subst_vars
    before_subst

let rec class_expr_of_region loc env region =
  match URegion.find region with
    | RRoot name ->
        let class_expr =
          try
            Env.find env.regions name
          with Not_found ->
            type_error loc "unbound region: %a"
              Ident.pp name
        in
        class_expr
    | ROwn (var, name) ->
        let var_class_expr = class_expr_of_var loc env var in
        class_expr_of_owned loc env var_class_expr name var
    | RVar ->
        type_error loc "region %a has not been unified yet"
          pp_region region

and region_of_var loc env var =
  let var_type =
    try
      Env.find env.variables var
    with Not_found ->
      type_error loc "unbound variable: %a" Ident.pp var
  in
  match UType.find var_type with
    | TPointer region ->
        region
    | TLogicPointer ->
        type_error loc "%a has type %a, this logic pointer has no region"
          Ident.pp var pp_type var_type
    | TBase _ | TLogic _ ->
        type_error loc "%a has type %a, it is not a pointer"
          Ident.pp var pp_type var_type
    | TPoly _ ->
        type_error loc "%a has a polymorph type, it may not be a pointer"
          Ident.pp var
    | TVar ->
        type_error loc "type of %a has not been unified yet"
          Ident.pp var

and class_expr_of_var loc env var =
  class_expr_of_region loc env (region_of_var loc env var)

let field_of_var loc env var ?(region = region_of_var loc env var) field =
  let ce = class_expr_of_region loc env region in
  field_of_class_expr env ce field var

let class_decl_of_region loc env region =
  decl_of_class_expr loc env (class_expr_of_region loc env region)

let class_decl_of_var loc env var =
  class_decl_of_region loc env (region_of_var loc env var)

let is_owned_single loc env var owned =
  let decl = class_decl_of_var loc env var in
  List.exists
    (fun (id, _) -> Ident.compare id owned = 0)
    decl.c_singles

let is_owned_group loc env var owned =
  let decl = class_decl_of_var loc env var in
  List.exists
    (fun (id, _) -> Ident.compare id owned = 0)
    decl.c_groups

let region env region =
  match region.node with
    | P.RRoot name ->
        let name, _ = lookup_region env name in
        URegion.create (RRoot name)
    | P.ROwn (var, name) ->
        let var, _ = lookup_var env var in
        let (class_name, _, _) as ce = class_expr_of_var region.loc env var in
        let decl = decl_of_class_expr region.loc env ce in
        let name, _ = lookup_owned decl name in
        URegion.create (ROwn (var, name))

let rec type_expr env t =
  match t.node with
    | P.TInt ->
        UType.create (TBase TInt)
    | P.TUnit ->
        UType.create (TBase TUnit)
    | P.TBool ->
        UType.create (TBase TBool)
    | P.TPointer r ->
        UType.create (TPointer (region env r))
    | P.TLogicPointer ->
        UType.create TLogicPointer
    | P.TPolyVar s ->
        let id, () = lookup_poly_var env s in
        UType.create (TPoly id)
    | P.TLogic (t, r, a) ->
        (* TODO: check number of arguments *)
        let t, _ = lookup_type env t in
        let r = List.map (region env) (olf r) in
        let a = List.map (type_expr env) (olf a) in
        UType.create (TLogic (t, r, a))

let unify_class_expr loc ((n1, r1, t1): class_expr) ((n2, r2, t2): class_expr) =
  if Ident.compare n1 n2 <> 0 then
    type_error loc "classes are not equal: %a and %a" Ident.pp n1 Ident.pp n2;
  if List.length r1 <> List.length r2 then
    type_error loc "wrong argument count for region parameters of class expression";
  if List.length t1 <> List.length t2 then
    type_error loc "wrong argument count for type parameters of class expression";
  List.iter2 URegion.unify r1 r2;
  List.iter2 UType.unify t1 t2

let class_expr env (name, regions, types) =
  let regions = olf regions in
  let types = olf types in
  let name, decl =
    try
      Env.lookup_and_find env.classes name
    with Not_found ->
      type_error dummy_location "unbound class: %s" name
  in
  let regions = List.map (region env) regions in
  (* TODO: check that region types are compatible *)
  let types = List.map (type_expr env) types in
  name, regions, types

let named_region env (name, ce) =
  let name = Ident.create name in
  let ce = class_expr env ce in
  name, ce

let named_type_expr env (name, t) =
  let name = Ident.create name in
  let t = type_expr env t in
  name, t

(******************************************************************************)

(* Types are not substituted *)
let rec class_expr_of_region_term loc env r =
  match r with
    | LRoot id ->
        begin
          try
            Env.find env.regions id
          with Not_found ->
            type_error loc "unknown region: %a" Ident.pp id
        end
    | LOwn (s, _, f) ->
        let sce = class_expr_of_region_term loc env s in
        class_expr_of_owned_before_subst loc env sce f

let rec logicify_pointers loc var owns ty =
  match UType.find ty with
    | TBase _
    | TLogicPointer
    | TPoly _ ->
        ty
    | TVar ->
        type_error loc "type not unified yet: %a" pp_type ty
    | TPointer r ->
        begin
          match URegion.find r with
            | RRoot id ->
                if ISet.mem id owns then
                  match var with
                    | Some var ->
                        UType.create
                          (TPointer (URegion.create (ROwn (var, id))))
                    | None ->
                        UType.create TLogicPointer
                else
                  UType.create TLogicPointer
            | _ ->
                UType.create TLogicPointer
        end
    | TLogic (id, r, l) ->
        UType.create
          (TLogic (id, r, List.map (logicify_pointers loc var owns) l))

(* Type of pointers is replaced by [?] if needed *)
let region_term_field loc env r var f =
  let (name, regions, types) = class_expr_of_region_term loc env r in
  let decl =
    try
      Env.find env.classes name
    with Not_found ->
      type_error loc "unknown class: %a" Ident.pp name
  in
  let sreg =
    List.fold_left2
      (fun acc (id, _) r -> IMap.add id r acc)
      IMap.empty
      decl.c_region_parameters
      regions
  in
  let styp =
    List.fold_left2
      (fun acc id t -> IMap.add id t acc)
      IMap.empty
      decl.c_type_parameters
      types
  in
  let f', ty =
    try
      List.find (fun (id, ty) -> Ident.name id = f) decl.c_fields
    with Not_found ->
      type_error loc "unknown field: %s in class %a" f Ident.pp name
  in
  let ty = subst_type loc sreg styp IMap.empty ty in
  let owns = List.map fst (decl.c_singles @ decl.c_groups) in
  let owns =
    List.fold_left
      (fun acc id -> ISet.add id acc)
      ISet.empty
      owns
  in
  let ty' = logicify_pointers loc var owns ty in
  log "LOGICIFY %a INTO %a@." pp_type ty pp_type ty';
  f', ty'

let region_term_own loc env r f =
  find_owned loc env (class_expr_of_region_term loc env r)
    f
    (list_findf
       (fun (id, _) ->
         if Ident.name id = f then Some id else None))

(******************************************************************************)

let permission env p =
  match p.node with
    | P.POpen r -> POpen (region env r)
    | P.PGroup r -> PGroup (region env r)
    | P.PFocus (s, r) -> PFocus (region env s, region env r)
    | P.PEmpty r -> PEmpty (region env r)
    | P.PClosed r -> PClosed (region env r)

(* assumes a region which was not unified is different *)
let eq_region r s =
  match URegion.find r, URegion.find s with
    | RRoot r, RRoot s ->
        Ident.compare r s = 0
    | ROwn (rvar, rown), ROwn (svar, sown) ->
        Ident.compare rvar svar = 0
        && Ident.compare rown sown = 0
    | RVar, RVar ->
        URegion.compare r s = 0
    | (RRoot _ | ROwn _ | RVar _), _ ->
        false

let eq_permission p q =
  match p, q with
    | POpen r, POpen s
    | PGroup r, PGroup s
    | PEmpty r, PEmpty s
    | PClosed r, PClosed s ->
        eq_region r s
    | PFocus (s1, r1), PFocus (s2, r2) ->
        eq_region s1 s2 && eq_region r1 r2
    | (POpen _ | PGroup _ | PEmpty _ | PClosed _ | PFocus _), _ ->
        false

let exist_permission p perms =
  List.exists (eq_permission p) perms

let positive_occurence r p =
  match p with
    | POpen s
    | PGroup s
    | PFocus (_, s)
    | PEmpty s
    | PClosed s ->
        eq_region r s

let positive_occurence_list r perms =
  List.exists (positive_occurence r) perms

(* non-unified regions do not count *)
let rec parent_positive_occurence loc env r perms =
  positive_occurence_list r perms ||
    match URegion.find r with
      | RRoot _ ->
          false
      | ROwn (var, _) ->
          parent_positive_occurence loc env (region_of_var loc env var) perms
      | RVar ->
          false

(******************************************************************************)

let const c =
  match c with
    | P.CUnit -> CUnit, tunit
    | P.CTrue -> CTrue, tbool
    | P.CFalse -> CFalse, tbool
    | P.CInt i -> CInt i, tint

let label env id =
  fst (lookup "label" env.labels id)

(******************************************************************************)

let compare_op o =
  match o with
    | P.LNeq -> LNeq
    | P.LLt -> LLt
    | P.LLe -> LLe
    | P.LGt -> LGt
    | P.LGe -> LGe
    | P.LEq -> LEq
    | _ -> assert false

let is_compare_op o =
  try
    ignore (compare_op o);
    true
  with _ ->
    false

let logic_op o =
  match o with
    | P.LOr -> LOr
    | P.LImpl -> LImpl
    | P.LIff -> LIff
    | P.LAnd -> LAnd
    | _ -> assert false

let is_logic_op o =
  try
    ignore (logic_op o);
    true
  with _ ->
    false

let term_op o =
  match o with
    | P.LSub -> LSub
    | P.LMul -> LMul
    | P.LDiv -> LDiv
    | P.LAdd -> LAdd
    | _ -> assert false

let is_term_op o =
  try
    ignore (term_op o);
    true
  with _ ->
    false

let logic_app_arguments type_region type_argument_as loc r a
    type_parameters region_parameters formal_parameters =
  let styp =
    List.fold_left
      (fun acc tid ->
        IMap.add tid (UType.create TVar) acc)
      IMap.empty
      type_parameters
  in
  let sreg =
    List.fold_left
      (fun acc (rid, _) ->
        IMap.add rid (URegion.create RVar) acc)
      IMap.empty
      region_parameters
  in
  let r = List.map type_region r in
  let a =
    List.map2
      (fun (_, formal) instance ->
        let expected = subst_type loc sreg styp IMap.empty formal in
        type_argument_as instance expected)
      formal_parameters
      a
  in
  styp, sreg, r, a

let predicate_app type_region type_argument_as make loc env n r a =
  let r = olf r in
  let id, p = lookup_predicate env n in
  check_argument_count loc r p.p_regions;
  check_argument_count loc a p.p_parameters;
  let styp, sreg, r, a =
    logic_app_arguments type_region type_argument_as loc r a
      p.p_type_parameters p.p_regions p.p_parameters
  in
  make id r a

let logic_app type_region type_argument_as make loc env n r a =
  let r = olf r in
  let id, lfd = lookup_logic_function env n in
  check_argument_count loc r lfd.lf_regions;
  check_argument_count loc a lfd.lf_parameters;
  let styp, sreg, r, a =
    logic_app_arguments type_region type_argument_as loc r a
      lfd.lf_type_parameters lfd.lf_regions lfd.lf_parameters
  in
  let ret = subst_type loc sreg styp IMap.empty lfd.lf_return_type in
  make id r a, ret

let pointers_always loc env perms r s: [`Different | `Equal | `Unknown] =
  if
    not (eq_region r s)
    && not (exist_permission (PFocus (r, s)) perms
            || exist_permission (PFocus (s, r)) perms)
    && parent_positive_occurence loc env r perms
    && parent_positive_occurence loc env s perms
  then
    `Different
  else if
    eq_region r s
    && (exist_permission (POpen r) perms
        || exist_permission (PClosed r) perms)
  then
    `Equal
  else
    `Unknown

let rec predicate env perms p: ([`True | `False | `Unknown] * _) =
  match p.node with
    | P.LConst { node = P.CTrue } ->
        `True, PTrue
    | P.LConst { node = P.CFalse } ->
        `False, PFalse

    | P.LBinOp (a, { node = op }, b) when is_compare_op op ->
        let op = compare_op op in
        begin
          match op with
            | LEq | LNeq ->
                log "LNEQ@.";
                let a, aty = term env a in
                let b, bty = term env b in
                (* pre-unify variables (i.e. unify without errors) *)
                begin
                  match UType.find aty, UType.find bty with
                    | _, TVar
                    | TVar, _ ->
                        UType.unify aty bty
                    | _ ->
                        ()
                end;
                (* special case for pointers *)
                let always =
                  match UType.find aty, UType.find bty with
                    | TPointer ra, TPointer rb ->
                        log "pointer %a, pointer %a@."
                          pp_region ra pp_region rb;
                        let always =
                          pointers_always p.loc env perms ra rb
                        in
                        begin
                          match always, op with
                            | `Equal, LEq -> `True
                            | `Different, LEq -> `False
                            | `Unknown, LEq -> `Unknown
                            | `Equal, LNeq -> `False
                            | `Different, LNeq -> `True
                            | `Unknown, LNeq -> `Unknown
                            | _ -> assert false (* only eq or leq here *)
                        end
                    | TPointer _, TLogicPointer
                    | TLogicPointer, TPointer _
                    | TLogicPointer, TLogicPointer ->
                        `Unknown
                    | _ ->
                        UType.unify aty bty;
                        `Unknown
                in
                always, PCompare (a, op, b)
            | LLt | LLe | LGt | LGe ->
                let a = term_as env a tint in
                let b = term_as env b tint in
                (* TODO: sometimes we know *)
                `Unknown, PCompare (a, op, b)
        end

    | P.LBinOp (a, { node = op }, b) when is_logic_op op ->
        let always_a, a = predicate env perms a in
        let always_b, b = predicate env perms b in
        let op = logic_op op in
        let always =
          match always_a, always_b, op with
            | `True, _, LOr
            | _, `True, LOr
            | `True, `True, LAnd
            | `False, _, LImpl
            | _, `True, LImpl
            | `False, `False, LIff
            | `True, `True, LIff
              -> `True
            | `False, `False, LOr
            | `False, _, LAnd
            | _, `False, LAnd
            | `True, `False, LImpl
            | `False, `True, LIff
            | `True, `False, LIff
              -> `False
            | _
              -> `Unknown
        in
        always, PLogicOp (a, op, b)

    | P.LExists (id, ty, body)
    | P.LForall (id, ty, body) ->
        let id = Ident.create id in
        let ty' = type_expr env ty in
        begin
          match UType.find ty' with
            | TPointer r ->
                type_error ty.loc "@[<hov>Quantification@ on@ pointers@ \
should@ be@ done@ using@ the@ [?]@ type.@ Then,@ use@ the@ \
``%a@ in@ %a''@ predicate."
                  Ident.pp id pp_region r
            | _ -> ()
        end;
        let ienv = { env with variables = Env.add env.variables id ty' } in
        let always, body = predicate ienv perms body in
        let q =
          match p.node with
            | P.LExists _ -> Exists
            | P.LForall _ -> Forall
            | _ -> assert false (* impossible *)
        in
        (* We assume all types are inhabited. *)
        always, PQuantify (q, id, ty', body)

    | P.LExistsRegion (id, ce, body)
    | P.LForallRegion (id, ce, body) ->
        let id = Ident.create id in
        let ce = class_expr env ce in
        let ienv = { env with regions = Env.add env.regions id ce } in
        let always, body = predicate ienv perms body in
        let q =
          match p.node with
            | P.LExistsRegion _ -> Exists
            | P.LForallRegion _ -> Forall
            | _ -> assert false (* impossible *)
        in
        always, PQuantifyRegion (q, id, ce, body)

    | P.LIn (p, r) ->
        let always = always_in_region env p r in
        let p' = pointer_term env p in
        let r' = region_term env r in
        always, PIn (p', r')

    | P.LAt (p, l) ->
        let always, p = predicate env perms p in
        let l = label env l in
        always, PAt (p, l)

    | P.LNot p ->
        let always, p = predicate env perms p in
        let not_always =
          match always with
            | `False -> `True
            | `True -> `False
            | `Unknown -> `Unknown
        in
        not_always, PNot p

    | P.LAtPre p ->
        let always, p = predicate env perms p in
        always, PAt (p, Why.at_old)

    | P.LApp (n, r, a) ->
        `Unknown,
        predicate_app
          (region_term env)
          (term_as env)
          (fun id r a -> PApp (id, r, a))
          p.loc
          env
          n r a

    | P.LBinOp _
    | P.LGet _
    | P.LConst { node = P.CInt _ }
    | P.LConst { node = P.CUnit }
    | P.LVar _ ->
        type_error p.loc "This is not a predicate, only a term."

(* this is dirty because we mix parsetree with typedtree *)
and always_in_region env p r =
  let _, pty = term env p in
  let r' = region_term env r in
  match UType.find pty with
    | TPointer pr ->
        begin
          match URegion.find pr with
            | RRoot id ->
                begin
                  match r' with
                    | LRoot id2 when Ident.compare id id2 = 0 ->
                        `True
                    | _ ->
                        `Unknown
                end
            | ROwn (var, own) ->
                begin
                  match r' with
                    | LOwn (reg2, LVar var2, own2)
                        when Ident.compare var var2 = 0
                        && Ident.compare own own2 = 0 ->
                        (* we might as well check that var2 is in reg2;
                           not necessarily inconsistent but most likely
                           an error *)
                        let pvar =
                          { node = Ident (Ident.name var2);
                            loc = p.loc }
                        in
                        let preg =
                          match r.node with
                            | P.LOwn (preg, _, _) -> preg
                            | _ -> assert false (* dirty *)
                        in
                        always_in_region env
                          { p with node = P.LVar pvar }
                          preg
                    | _ ->
                        `Unknown
                end
            | _ ->
                `Unknown
        end
    | _ ->
        `Unknown

and term env t =
  match t.node with
    | P.LConst { node = c } ->
        let c, ty = const c in
        LConst c, ty

    | P.LBinOp (a, { node = op }, b) when is_term_op op ->
        let a = term_as env a tint in
        let b = term_as env b tint in
        let op = term_op op in
        LBinOp (a, op, b), tint

    | P.LVar name ->
        let name, ty = lookup_var env name in
        LVar name, ty

    | P.LGet (r, p, f) ->
        let r = region_term env r in
        let p' = pointer_term env p in
        let var =
          match p' with
            | LVar name -> Some name
            | _ -> None
        in
        let f, ty = region_term_field t.loc env r var f in
        LGet (r, p', f), ty

    | P.LAt (t, l) ->
        let t, ty = term env t in
        let l = label env l in
        LAt (t, l), ty

    | P.LAtPre t ->
        let t, ty = term env t in
        LAt (t, Why.at_old), ty

    | P.LApp (n, r, a) ->
        logic_app
          (region_term env)
          (term_as env)
          (fun id r a -> LApp (id, r, a))
          t.loc
          env
          n r a

    | P.LExists _
    | P.LForall _
    | P.LExistsRegion _
    | P.LForallRegion _
    | P.LIn _
    | P.LNot _
    | P.LBinOp _ ->
        type_error t.loc "This is a predicate, but a term is expected here."

and term_as env t ty =
  let t', tty = term env t in
  try
    UType.unify ty tty;
    t'
  with _ ->
    type_error t.loc "This term has type %a but it should have type %a."
      pp_type tty pp_type ty

and pointer_term env t =
  let t', pty = term env t in
  begin
    match UType.find pty with
      | TPointer _
      | TLogicPointer ->
          t'
      | TVar ->
          type_error t.loc "Term type not unified yet."
      | TBase _ | TLogic _ ->
          type_error t.loc "Term is not a pointer."
      | TPoly _ ->
          type_error t.loc "Term is polymorph, it may not be a pointer."
  end

and region_term env t =
  match t.node with
    | P.LRoot id ->
        let id, _ = lookup_region env id in
        LRoot id
    | P.LOwn (t, p, f) ->
        let t' = region_term env t in
        let p' = pointer_term env p in
        let f' = region_term_own t.loc env t' f in
        LOwn (t', p', f')

(******************************************************************************)

let pointer_comparison loc env perms r s =
  match pointers_always loc env perms r s with
    | `Different ->
        warning
          "%aWarning: regions %a and %a are disjoint, \
their pointers are never equal"
          print_location loc pp_region r pp_region s
    | `Equal ->
        warning
          "%aWarning: regions %a is singleton, its pointer is unique"
          print_location loc pp_region r
    | `Unknown ->
        ()

let rec path_permission_of_region loc env r perms =
  try
    match List.find (positive_occurence r) perms with
      | PEmpty _ ->
          PPEmpty
      | POpen _ | PClosed _ ->
          PPSingle
      | PGroup _ ->
          PPGroup
      | PFocus _ ->
          PPFocus
  with Not_found ->
    match URegion.find r with
      | ROwn (var, _) ->
          let var_region = region_of_var loc env var in
          path_permission_of_region loc env var_region perms
      | RRoot _ | RVar ->
          PPNone

(* return [(root, path)] where [path] is a list of [(var, perm, own)],
   starting with var of region root;
   perm is the permission of the var *)
let rec path ?(acc = []) loc env perms x =
  let ty = (* a supprimer *)
    try
      Env.find env.variables x
    with Not_found ->
      type_error loc "(path) variable not found: %a"
        pp_ident x
  in
  match UType.find ty with
    | TPointer r ->
        region_path ~acc loc env perms r
    | _ ->
        assert false

and region_path ?(acc = []) loc env perms r =
  match URegion.find r with
    | RRoot id ->
        id, acc
    | ROwn (x, o) ->
        let p = path_permission_of_region loc env r perms in
        path ~acc: ((x, p, o) :: acc) loc env perms x
    | RVar ->
        type_error loc "region variable not unified: %a" pp_region r

let region_with_path loc env perms r =
  let r = region env r in
  r, region_path loc env perms r

let rec expr env perms e =
  let expr e = expr env perms e in
  let expr_as e t =
    let typed_e, type_of_e = expr e in
    try
      UType.unify type_of_e t;
      typed_e
    with Type_error _ ->
      type_error e.loc
        "This expression has type %a but it should have type %a."
        pp_type type_of_e pp_type t
  in
  let find_var name = lookup_var env name in
  match e.node with
    | P.EConst c ->
        let c, ty = const c.node in
        EConst c, ty
    | P.EVar name ->
        let name, ty = find_var name in
        EVar name, ty
    | P.EBinOp (a, { node = And | Or as op }, b) ->
        let a = expr_as a tbool in
        let b = expr_as b tbool in
        EBinOp (a, op, b), tbool
    | P.EBinOp (a, { node = Sub | Mul | Div | Add as op }, b) ->
        let a = expr_as a tint in
        let b = expr_as b tint in
        EBinOp (a, op, b), tint
    | P.EBinOp (a, { node = Lt | Le | Gt | Ge as op }, b) ->
        let a = expr_as a tint in
        let b = expr_as b tint in
        EBinOp (a, op, b), tbool
    | P.EBinOp (a, { node = Neq | Eq as op }, b) ->
        let a, ta = expr a in
        let b =
          match UType.find ta with
            | TPointer region1 ->
                let b, tb = expr b in
                begin
                  match UType.find tb with
                    | TPointer region2 ->
                        pointer_comparison e.loc env perms region1 region2;
                        b
                    | _ ->
                        type_error e.loc "cannot compare %a and %a"
                          pp_type ta pp_type tb
                end
            | _ ->
                expr_as b ta
        in
        EBinOp (a, op, b), tbool
    | P.ESelect (var, field) ->
        let var, var_ty = find_var var in
        let region = region_of_var e.loc env var in
        let ce = class_expr_of_region e.loc env region in
        if not (parent_positive_occurence e.loc env region perms) then
          type_error e.loc "@.%a@.No permission allows to read %a: [%a] here."
            pp_available perms
            Ident.pp var
            pp_region region;
        let field, field_ty = field_of_class_expr env ce field var in
        let path = path e.loc env perms var in
        ESelect (var, field, path), field_ty

    | P.EApp (n, r, a) ->
        logic_app
          (region_with_path e.loc env perms)
          expr_as
          (fun id r a -> EApp (id, r, a))
          e.loc
          env
          n r a

(******************************************************************************)

let rec simple_consume ?(acc = []) perm perms =
  match perms with
    | [] ->
        raise Not_found
    | p :: rem ->
        if eq_permission p perm then
          List.rev_append acc rem
        else
          simple_consume ~acc: (p :: acc) perm rem

let simple_consume_list to_consume perms =
  List.fold_left
    (fun perms to_consume -> simple_consume to_consume perms)
    perms
    to_consume

exception CannotConsume of permission * permission list

let owned_regions_of_decl var decl =
  let make_owned (region, _) = URegion.create (ROwn (var, region)) in
  List.map make_owned decl.c_singles,
  List.map make_owned decl.c_groups

let owned_permissions_of_decl var decl =
  let singles, groups = owned_regions_of_decl var decl in
  List.map (fun r -> PClosed r) singles
  @ List.map (fun r -> PGroup r) groups

(* Return a variable of type [reg].
   May raise [Not_found].
   Does not check whether [reg] is singleton. *)
let var_of_region env reg =
  Env.such_as env.variables
    (fun t ->
      match UType.find t with
        | TPointer r -> eq_region r reg
        | _ -> false)

let owned_regions_of_var loc env var =
  let cd = class_decl_of_var loc env var in
  owned_regions_of_decl var cd

let owned_permissions_of_var loc env var =
  let cd = class_decl_of_var loc env var in
  owned_permissions_of_decl var cd

let make_focus loc env perms var region =
  IFocus (var, path loc env perms var, region, region_path loc env perms region)

let make_unfocus loc env perms focus_region region =
  let focus_region_path = region_path loc env perms focus_region in
  let region_path = region_path loc env perms region in
  let var =
    try
      var_of_region env focus_region
    with Not_found ->
      type_error loc "Cannot find a variable of region %a, needed to \
interpret unfocus." pp_region focus_region
  in
  IUnfocus (focus_region, focus_region_path, region, region_path, var)

let make_new loc env perms var reg =
  let region_path = region_path loc env perms reg in
  let ce = class_expr_of_region loc env reg in
  INew (var, reg, region_path, ce)

let make_use_invariant loc env perms var =
  let var_path = path loc env perms var in
  let reg = region_of_var loc env var in
  let ce = class_expr_of_region loc env reg in
  IUseInvariant (var, var_path, ce)

let make_adopt_aux loc env perms singleton_region region =
  let singleton_region_path = region_path loc env perms singleton_region in
  let region_path = region_path loc env perms region in
  let var =
    try
      var_of_region env singleton_region
    with Not_found ->
      type_error loc "Cannot find a variable of region %a, needed to \
interpret adoption." pp_region singleton_region
  in
  IAdopt (singleton_region, singleton_region_path, region, region_path, var)

let make_pack loc env perms var =
  let ce = class_expr_of_var loc env var in
  IPack (var, ce, path loc env perms var)

let operations_producing env perms perm =
  let loc = dummy_location in
  (* find a variable of type [region] in environment *)
  let find_var region f =
    try
      let var =
        Env.such_as env.variables
          (fun t ->
            match UType.find t with
              | TPointer r -> eq_region r region
              | _ -> false)
      in
      f var
    with Not_found ->
      []
  in
  (* construct an unpack operation *)
  let unpack var =
    let region =
      try
        match UType.find (Env.find env.variables var) with
          | TPointer region -> region
          | _ -> assert false (* impossible (invalid argument) *)
      with Not_found ->
        assert false (* impossible (invalid argument) *)
    in
    let cd = class_decl_of_region loc env region in
    let owned = owned_permissions_of_decl var cd in
    [ [ PClosed region ], IUnpack var, POpen region :: owned ]
  in
  match perm with
    | POpen region ->
        (* To obtain R^o, we can unpack any variable of type [R]. *)
        find_var region unpack
    | PClosed region ->
        (* To obtain R^c, we can pack any variable of type [R]. *)
        let cd = class_decl_of_region loc env region in
        let do_pack =
          find_var region
            (fun x ->
              let owned = owned_permissions_of_decl x cd in
              let pack = make_pack loc env perms x in
              [ POpen region :: owned, pack, [ PClosed region ] ])
        in
        (* Or, if R = x.r and r is single, we can unpack x. *)
        let do_unpack =
          match URegion.find region with
            | ROwn (var, owned) when is_owned_single loc env var owned ->
                unpack var
            | _ ->
                []
        in
        do_pack @ do_unpack
    | PGroup region ->
        (* To obtain R^g, we can unfocus S if S -° R is available. *)
        let unfocusable_regions =
          list_filterf
            (function
              | PFocus (s, _) -> Some s
              | _ -> None)
            perms
        in
        let do_unfocus =
          List.map
            (fun focus_region ->
              [ PClosed focus_region; PFocus (focus_region, region) ],
              make_unfocus loc env perms focus_region region,
(*              IUnfocus (focus_region, region),*)
              [ PGroup region ])
            unfocusable_regions
        in
        (* Or, we can weaken R. *)
        let do_weaken_single =
          [ PClosed region ], IWeakenSingle region, [ PGroup region ]
        in
        let do_weaken_empty =
          [ PEmpty region ], IWeakenEmpty region, [ PGroup region ]
        in
        (* Or, if R = x.r and r is group, we can unpack x. *)
        let do_unpack =
          match URegion.find region with
            | ROwn (var, owned) when is_owned_group loc env var owned ->
                unpack var
            | _ ->
                []
        in
        do_weaken_single :: do_weaken_empty :: do_unpack @ do_unfocus
    | _ ->
        []

(* apply effect of operation on environment *)
let apply_operation env operation =
  match operation with
    | ILetRegion (region, ce) ->
        { env with regions = Env.add env.regions region ce }
    | IFocus (var, _, region, _) ->
        let new_type = UType.create (TPointer region) in
        let variables = Env.replace env.variables var new_type in
        log "replaced var %a in environment (now of region %a)\n%!"
          Ident.pp var pp_region region;
        { env with variables = variables }
    | IUnfocus (s, _, r, _, _) | IAdopt (s, _, r, _, _) ->
        let new_type = UType.create (TPointer r) in
        let vars =
          Env.filter env.variables
            (fun t ->
              match UType.find t with
                | TPointer vr when eq_region vr s -> true
                | _ -> false)
        in
        let variables =
          List.fold_left
            (fun env var -> Env.replace env var new_type)
            env.variables
            vars
        in
        { env with variables = variables }
    | _ ->
        env

(* consume [perm] from [perms], returning a list of prerequisite instructions,
   and may consume more than [perm] because of them *)
let rec consume ?(forbidden = []) env perms perm =
(*  log "(available %a)" pp_permissions perms;
  log "(consume %a)" pp_permission perm;*)
  try
    let perms = simple_consume perm perms in
(*    log "(%a already available)" pp_permission perm;*)
    env, perms, []
  with Not_found ->
    if List.exists (eq_permission perm) forbidden then
      raise (CannotConsume (perm, perms))
    else
      let forbidden = perm :: forbidden in
      let operations = operations_producing env perms perm in
      let result = ref None in
      List.iter
        (fun ((consumes, operation, produces) as op) ->
          try
            log "@[<hv 2>apply %a for %a...@ "
              pp_operation_producing op pp_permission perm;
            let env, perms, prereq =
              consume_list_pp ~forbidden env perms consumes
            in
            let env = apply_operation env operation in
            log "ok@]@ ";
            let perms = perms @ produces in
            let perms = simple_consume perm perms in
            result := Some (env, perms, prereq @ [ operation ])
          with CannotConsume _ ->
            log "failed@]@ ")
        operations;
      match !result with
        | None ->
            raise (CannotConsume (perm, perms))
        | Some result ->
            result

and consume_list ?forbidden env perms consumes =
  match consumes with
    | [] ->
        env, perms, []
    | perm :: rem ->
        let env, perms, prereq = consume ?forbidden env perms perm in
        let env, perms, prereq2 =
          consume_list ?forbidden env perms rem
        in
        env, perms, prereq @ prereq2

and consume_list_pp ?forbidden env perms consumes =
  log "@[<hv 2>";
  try
    let r = consume_list ?forbidden env perms consumes in
    log "@]@ ";
    r
  with exn ->
    log "@]@ ";
    raise exn

(* same as consume, except that as we know a variable of the region, we can
   also try to do a focus *)
let consume_single f loc env perms var =
  let region = region_of_var loc env var in
  let ce = class_expr_of_region loc env region in
  log "consume_single (var %a, region %a, class %a)\n%!"
    Ident.pp var pp_region region pp_class_expr ce;
  try
    log "- try to consume %a...\n%!" pp_permission (f region);
    let env, perms, prereq = consume env perms (f region) in
    log "- consumed %a directly\n%!" pp_permission (f region);
    env, perms, prereq, region
  with CannotConsume _ ->
    log "- failed to consume %a directly\n%!" pp_permission (f region);
    try
      (* try to consume group *)
      let env, perms, prereq = consume env perms (PGroup region) in
      (* good, we can focus *)
      let focus_region_ident = Ident.create "F" in
      let focus_region = URegion.create (RRoot focus_region_ident) in
      let let_region = ILetRegion (focus_region_ident, ce) in
      let focus = make_focus loc env perms var focus_region in
      (* combine let region and focus permissions *)
      let perms =
        PClosed focus_region
        :: PFocus (focus_region, region)
        :: perms
      in
      (* environment is modified by let region and focus *)
      let env = apply_operation env let_region in
      let env = apply_operation env focus in
      (* now we have to (un)pack *)
      let env, perms, prereq2 =
        consume env perms (f focus_region)
      in
      let full_prereq = prereq @ [ let_region; focus ] @ prereq2 in
      env, perms, full_prereq, focus_region
    with CannotConsume _ ->
      type_error loc "Permission %a is unavailable here.@.%a"
        pp_permission (f region)
        pp_available perms

let consume_open = consume_single (fun x -> POpen x)
let consume_closed = consume_single (fun x -> PClosed x)

let last_consume loc env perms perm =
  try
    consume env perms perm
  with CannotConsume _ ->
    type_error loc "Permission %a is unavailable here.@.%a"
      pp_permission perm
      pp_available perms

let rec last_consume_list ?(where = "here") loc env perms list =
  try
    consume_list env perms list
  with CannotConsume (perm, perms) ->
    type_error loc "Permission %a is unavailable %s.@.%a"
      pp_permission perm
      where
      pp_available perms

(******************************************************************************)

let make_adopt loc env perms s r =
  (* TODO: check class exprs are compatible *)
  let env, perms, prereq1 = last_consume loc env perms (PClosed s) in
  let env, perms, prereq2 = last_consume loc env perms (PGroup r) in
  let perms = PGroup r :: perms in
  let adopt = make_adopt_aux loc env perms s r in
  let env = apply_operation env adopt in
  env,
  perms,
  prereq1 @ prereq2 @ [ adopt ]

let unify_with_adoptions loc env perms t1 t2 =
  match UType.find t1, UType.find t2 with
    | TPointer r1, TPointer r2 ->
        if eq_region r1 r2 then
          env, perms, []
        else
          begin
            try
              (* TODO: unify_(regions,class_exprs)_with_adoptions (recursive) *)
              URegion.unify r1 r2;
              env, perms, []
            with Type_error _ ->
              make_adopt loc env perms r1 r2
          end
    | (TLogicPointer | TPointer _ | TBase _ | TPoly _ | TVar _ | TLogic _), _ ->
        begin
          try
            UType.unify t1 t2
          with Type_error _ ->
            type_error loc
              "incompatible types: %a and %a"
              pp_type t1 pp_type t2
        end;
        env, perms, []
    (* more to do with logic types *)

let expr_as_no_adopt env perms e ty =
  let e2, e2_type = expr env perms e in
  begin
    try
      UType.unify e2_type ty
    with Type_error _ ->
      type_error e.loc
        "this expression has type %a but it should have type %a"
        pp_type e2_type pp_type ty
  end;
  e2

(* type e as ty
   return (env2, perms2, prereq, e2) *)
let expr_as env perms e ty =
  (* type e *)
  let e1, e1_type = expr env perms e in
  (* find needed adoptions *)
  let env, perms, prereq = unify_with_adoptions e.loc env perms e1_type ty in
  (* re-type e in the new environment
     adoptions won't require recursion, we finish now *)
  let e2 = expr_as_no_adopt env perms e ty in
  env, perms, prereq, e2

(******************************************************************************)

(* Return (perms, prereq) where prereq may contain some valid unfocuses.
   Returned permissions are [perms] without those about out of scope regions,
   and if S->R and S^c are available but S is not in scope in [env],
   [prereq] will contain [unfocus S as R]. *)
let scope_perms loc env perms =
  log "scope_perms %a@." pp_permissions perms;
  let out_of_scope r =
    match URegion.find r with
      | RRoot r -> not (Env.mem env.regions r)
      | ROwn (v, _) -> not (Env.mem env.variables v)
      | RVar ->
          type_error loc
            "region %a was not unified, cannot know its scope" pp_region r
  in
  let focus, perms =
    List.fold_left
      (fun ((focus, perms) as acc) perm ->
        match perm with
          | PEmpty r
          | PClosed r
          | POpen r
          | PGroup r
          | PFocus (_, r) when out_of_scope r -> acc
          | PEmpty _
          | PClosed _
          | POpen _
          | PGroup _ -> focus, perm :: perms
          | PFocus (s, r) when out_of_scope s -> (s, r) :: focus, perms
          | PFocus _ -> focus, perm :: perms)
      ([], [])
      perms
  in
  List.fold_left
    (fun ((perms, prereq) as acc) (s, r) ->
      try
        let perms = simple_consume (PClosed s) perms in
        let unfocus = make_unfocus loc env perms s r in
        (PGroup r) :: perms, prereq @ [ unfocus ]
      with Not_found ->
        acc)
    (perms, [])
    focus

let positive_region = function
  | PEmpty r
  | POpen r
  | PClosed r
  | PGroup r
  | PFocus (_, r) ->
      r

let perm_of_region perms reg =
  List.find (fun perm -> eq_region (positive_region perm) reg) perms

(* Assume [perm1] and [perm2] are about the same region.
   Return [None] if [perm1] and [perm2] cannot be joined.
   Return [Some ((op1, op2), perms1', perms2')] where:
     - [op1] must be applied to [perms1] to get [perms1'] instead
     - [op2] idem
   (op1 and op2 are lists)
     - [perms1'] and [perms2'] have joined [perm1] and [perm2]
   This version is not symmetric, use [join_candidate] instead. *)
let join_candidate_aux env env1 env2 perms1 perms2 perm1 perm2 =
  try
    match perm1, perm2 with
      | PEmpty r, PGroup s ->
          assert (eq_region r s);
          let perms1 = PGroup r :: simple_consume (PEmpty r) perms1 in
          Some (([ IWeakenEmpty r ], []), perms1, perms2)
      | PClosed r, PGroup s ->
          assert (eq_region r s);
          let perms1 = PGroup r :: simple_consume (PClosed r) perms1 in
          Some (([ IWeakenSingle r ], []), perms1, perms2)
      | PClosed r, POpen s ->
          assert (eq_region r s);

          begin
            try
              let v = var_of_region env1 r in
              log "var_of_region env1 %a = %a@." pp_region r Ident.pp v
            with Not_found ->
              log "var_of_region env1 %a = Not_found@." pp_region r
          end;
          begin
            try
              let v = var_of_region env2 r in
              log "var_of_region env2 %a = %a@." pp_region r Ident.pp v
            with Not_found ->
              log "var_of_region env2 %a = Not_found@." pp_region r
          end;

          begin
            try
              let var = var_of_region env1 r in
              let owned =
                try
                  owned_permissions_of_var dummy_location env1 var
                with Types.Type_error _ ->
                  raise Not_found
              in
              let perms1 =
                POpen r :: owned @ simple_consume (PClosed r) perms1
              in
              Some (([ IUnpack var ], []), perms1, perms2)
            with Not_found ->
              try
                (* No variable for region r. Maybe we can pack instead. *)
                (* USELESS BECAUSE OF SCOPE_PERMISSIONS.
                   TODO: fix *)
                let var = var_of_region env2 r in
                let owned =
                  try
                    owned_permissions_of_var dummy_location env2 var
                  with Types.Type_error _ ->
                    raise Not_found
                in
                let perms2 =
                  PClosed r :: simple_consume_list (PClosed r :: owned) perms2
                in
                let pack = make_pack Lang.dummy_location env perms1 var in
                Some (([ pack ], []), perms1, perms2)
              with Not_found ->
                None
          end
      | _ ->
          None
  with Not_found ->
    assert false (* impossible *)

let join_candidate env env1 env2 perms1 perms2 perm1 perm2 =
  log "join_candidate %a and %a@." pp_permission perm1 pp_permission perm2;
  match join_candidate_aux env env1 env2 perms1 perms2 perm1 perm2 with
    | Some _ as x -> x
    | None ->
        match join_candidate_aux env env2 env1 perms2 perms1 perm2 perm1 with
          | Some ((op1, op2), perms1, perms2) -> Some ((op2, op1), perms2, perms1)
          | None -> None

(* I don't want to sort the lists because comparing regions is odd with unification variables. *)
let same_permissions perms1 perms2 =
  try
    let rem =
      List.fold_left
        (fun perms perm ->
          list_removef (eq_permission perm) perms)
        perms1
        perms2
    in
    rem = []
  with Not_found ->
    false

(* Return [perms, prereq1, prereq2] where, if [prereq1] are applied to [perms1]
   and [prereq2] to [perms2], both become [perms]. *)
let rec join ?(acc = []) loc env env1 env2 perms1 perms2 =
  try
    let ops, perms1, perms2 =
      list_findf
        (fun perm ->
          let reg = positive_region perm in
          try
            let perm2 = perm_of_region perms2 reg in
            join_candidate env env1 env2 perms1 perms2 perm perm2
          with Not_found ->
            None)
        perms1
    in
    join ~acc: (ops :: acc) loc env env1 env2 perms1 perms2
  with Not_found ->
    if same_permissions perms1 perms2 then
      begin
        let acc = List.rev acc in
        let ops1, ops2 = List.split acc in
        perms1, List.flatten ops1, List.flatten ops2
      end
    else
      type_error loc "@[<hv 2>cannot join permissions:@ %a@ and@ %a@]"
        pp_permissions perms1 pp_permissions perms2

let join a b c d e f =
  log "join %a and %a@." pp_permissions e pp_permissions f;
  join a b c d e f

(******************************************************************************)

(* Return (env2, perms2, seq) where [seq] is a list of instructions
   equivalent to [i], consuming [perms] and producing [perms2]. *)
let rec instruction env perms i =
  let find_var name = lookup_var env name in
  match i.node with

    | P.IAffect (id, field, e) ->
        let var, var_type = find_var id in
        let env, perms, prereq1, actual_region =
          consume_open id.loc env perms var
        in
        (* consumed POpen, now reproduce it *)
        let perms = POpen actual_region :: perms in
        (* type the field *)
        let region = region_of_var id.loc env var in
        let ce = class_expr_of_region id.loc env region in
        let field, field_ty = field_of_class_expr env ce field var in
        (* type the expression *)
        let env, perms, prereq2, e = expr_as env perms e field_ty in
        let path = path i.loc env perms var in
        env,
        perms,
        prereq1 @ prereq2 @ [ IAffect (var, field, path, e) ]

    | P.ILet (id, e) ->
        let id = Ident.create id in
        let e, te = expr env perms e in
        let env = { env with variables = Env.add env.variables id te } in
        env,
        perms,
        [ ILet (id, e) ]

    | P.IUnpack id ->
        let var, var_ty = lookup_var env id in
        let env, perms, prereq1, actual_region =
          consume_closed id.loc env perms var
        in
        (* Instead of applying the typing rule here, we apply it using the
           inference mechanism. No need to duplicate work! *)
        let perms = PClosed actual_region :: perms in
        let env, perms, prereq2, _ =
          consume_open id.loc env perms var
        in
        let perms = POpen actual_region :: perms in
        env,
        perms,
        prereq1 @ prereq2

    | P.IPack id ->
        let var, var_ty = lookup_var env id in
        let env, perms, prereq1, actual_region =
          consume_open id.loc env perms var
        in
        (* Instead of applying the typing rule here, we apply it using the
           inference mechanism. No need to duplicate work! *)
        let perms = POpen actual_region :: perms in
        let env, perms, prereq2, _ =
          consume_closed id.loc env perms var
        in
        let perms = PClosed actual_region :: perms in
        env,
        perms,
        prereq1 @ prereq2

    | P.IWeakenSingle reg ->
        let reg = region env reg in
        (* No sense in trying to infer a focus here. *)
        let env, perms, prereq = last_consume i.loc env perms (PClosed reg) in
        let perms = PGroup reg :: perms in
        env,
        perms,
        prereq @ [ IWeakenSingle reg ]

    | P.IWeakenEmpty reg ->
        let reg = region env reg in
        (* No sense in trying to infer a focus here. *)
        let env, perms, prereq = last_consume i.loc env perms (PEmpty reg) in
        let perms = PGroup reg :: perms in
        env,
        perms,
        prereq @ [ IWeakenEmpty reg ]

    | P.INew (id, r) ->
        let reg = region env r in
        let env, perms, prereq = last_consume r.loc env perms (PEmpty reg) in
        let var = Ident.create id in
        let ty = UType.create (TPointer reg) in
        let env = { env with variables = Env.add env.variables var ty } in
        let decl = class_decl_of_region r.loc env reg in
        let singles, groups = owned_regions_of_decl var decl in
        let owned = List.map (fun r -> PEmpty r) (singles @ groups) in
        let perms = POpen reg :: owned @ perms in
        let let_new = make_new i.loc env perms var reg in
        env,
        perms,
        prereq @ [ let_new ]

    | P.ILetRegion (id, ce) ->
        let ce = class_expr env ce in
        let var = Ident.create id in
        let env = { env with regions = Env.add env.regions var ce } in
        let perms = PEmpty (URegion.create (RRoot var)) :: perms in
        env,
        perms,
        [ ILetRegion (var, ce) ]

    | P.IAdopt (s, r) ->
        let s = region env s in
        let r = region env r in
        make_adopt i.loc env perms s r

    | P.IFocus (id, reg) ->
        let reg = region env reg in
        let var, var_ty = lookup_var env id in
        let var_region = region_of_var id.loc env var in
        let perms1 = perms in
        let env, perms, prereq1 = last_consume i.loc env perms (PEmpty reg) in
        let env, perms, prereq2 =
          last_consume i.loc env perms (PGroup var_region)
        in
        let perms = PClosed reg :: PFocus (reg, var_region) :: perms in
        let focus = make_focus i.loc env perms1 var reg in
        let env = apply_operation env focus in
        env,
        perms,
        prereq1 @ prereq2 @ [ focus ]

    | P.IUnfocus (s, r) ->
        let s = region env s in
        let r = region env r in
        let env, perms, prereq1 = last_consume i.loc env perms (PClosed s) in
        let env, perms, prereq2 =
          last_consume i.loc env perms (PFocus (s, r))
        in
        let perms = PGroup r :: perms in
        let unfocus = make_unfocus i.loc env perms s r in
        let env = apply_operation env unfocus in
        env,
        perms,
        prereq1 @ prereq2 @ [ unfocus ]

    | P.IIf (e, s1, s2) ->
        let env, perms, prereq_e, e = expr_as env perms e tbool in
        let env1, perms1, s1 = sequence env perms s1 in
        let env2, perms2, s2 = sequence env perms s2 in
        let perms1, prereq1 = scope_perms i.loc env perms1 in
        let perms2, prereq2 = scope_perms i.loc env perms2 in
        let perms, prereq_left, prereq_right =
          join i.loc env env1 env2 perms1 perms2
        in
        let if_then =
          IIf (e, s1 @ prereq1 @ prereq_left, s2 @ prereq2 @ prereq_right)
        in
        env,
        perms,
        prereq_e @ [ if_then ]

    | P.ICall (var_s, fun_s, regions, args) ->
        (* TODO: check class expr unification *)
        let var_id = Ident.create var_s in
        let fun_id, fun_decl = lookup_function env fun_s in
        (* instantiate function: poly becomes uvar *)
        let instantiated_regions =
          List.map
            (fun _ -> URegion.create RVar)
            fun_decl.f_region_parameters
        in
        let sreg =
          List.fold_left2
            (fun acc (poly_reg, _) inst_reg ->
              IMap.add poly_reg inst_reg acc)
            IMap.empty
            fun_decl.f_region_parameters
            instantiated_regions
        in
        let styp =
          List.fold_left
            (fun acc poly_typ ->
              IMap.add poly_typ (UType.create TVar) acc)
            IMap.empty
            fun_decl.f_type_parameters
        in
        let svar =
          List.fold_left2
            (fun acc (formal, _) arg ->
              let res =
                match arg.node with
                  | P.EVar var -> Some (fst (lookup_var env var))
                  | _ -> None
              in
              IMap.add formal res acc)
            IMap.empty
            fun_decl.f_arguments
            args
        in
        let instantiated_regions =
          List.map2
            (fun (_, ce) ireg ->
              let ce = subst_class_expr i.loc sreg styp svar ce in
              ireg, ce)
            fun_decl.f_region_parameters
            instantiated_regions
        in
        let instantiated_arg_types =
          List.map
            (fun (_, t) -> subst_type i.loc sreg styp svar t)
            fun_decl.f_arguments
        in
        let instantiated_return_type =
          subst_type i.loc sreg styp svar fun_decl.f_return_type
        in
        (* unify regions if given *) (* TODO: infer adoptions *)
        begin
          match regions with
            | None ->
                ()
            | Some regions ->
                if List.length regions <> List.length instantiated_regions then
                  type_error i.loc "wrong region argument count";
                List.iter2
                  (fun param (formal, formal_class) ->
                    URegion.unify param formal;
                    let param_class = class_expr_of_region i.loc env param in
                    unify_class_expr i.loc formal_class param_class)
                  (List.map (region env) regions)
                  instantiated_regions
        end;
        (* unify arguments *)
        let env, perms, prereq, args =
          List.fold_left2
            (fun (env, perms, prereq, acc) arg ety ->
              let env', perms', prereq', arg' =
                expr_as env perms arg ety
              in
              env', perms', prereq @ prereq', acc @ [ arg' ])
            (env, perms, [], [])
            args
            instantiated_arg_types
        in
        (* compute paths *)
        let instantiated_regions =
          List.map
            (fun (r, _) -> r, region_path i.loc env perms r)
            instantiated_regions
        in
        (* consume and produce permissions *)
        let consumed_perms =
          List.map (subst_permission i.loc sreg svar) fun_decl.f_consumes
        in
        let produced_perms =
          List.map (subst_permission i.loc sreg svar) fun_decl.f_produces
        in
        let env, perms, prereq = consume_list env perms consumed_perms in
        let perms = perms @ produced_perms in
        (* put var_id in environment *)
        let env =
          { env with variables =
              Env.add env.variables var_id instantiated_return_type }
        in
        log "RETURN TYPE of %a = %a(...): %a@."
          Ident.pp var_id Ident.pp fun_id
          Pp.pp_type instantiated_return_type;
        (* construct typed call *)
        let call =
          ICall (
            var_id,
            fun_id,
            instantiated_regions,
            args,
            instantiated_return_type
          )
        in
        (* finish *)
        env,
        perms,
        prereq @ [ call ]

    | P.IAssert p ->
        let always, p = predicate env perms p in
        begin
          match always with
            | `True ->
                warning "%a@[<hov 2>this@ predicate@ is@ always@ true,@ \
consider@ using@ a@ \"use\"@ instead@ of@ an@ \"assert\"@]"
                  print_location i.loc
            | `False ->
                warning "%a@[<hov 2>this@ predicate@ is@ always@ false@]"
                  print_location i.loc
            | `Unknown ->
                ()
        end;
        env, perms, [ IAssert p ]

    | P.IAssume p ->
        let always, p = predicate env perms p in
        begin
          match always with
            | `False ->
                warning "%a@[<hov 2>this@ predicate@ is@ always@ false@]"
                  print_location i.loc
            | `True
            | `Unknown ->
                ()
        end;
        env, perms, [ IAssume p ]

    | P.IUse p ->
        let always, p = predicate env perms p in
        begin
          match always with
            | `False ->
                type_error i.loc "@[<hov 2>this@ predicate@ is@ always@ false@]"
            | `True ->
                ()
            | `Unknown ->
                type_error i.loc "@[<hov 2>cannot@ prove@ this@ predicate@ \
using@ typing@ only@]"
        end;
        env, perms, [ IAssume p ]

    | P.ILabel l ->
        let l = Ident.create l in
        let env = { env with labels = Env.add env.labels l () } in
        env, perms, [ ILabel l ]

    | P.IUseInvariant id ->
        let var, var_ty = lookup_var env id in
        let env, perms, prereq, actual_region =
          consume_closed id.loc env perms var
        in
        let perms = PClosed actual_region :: perms in
        env,
        perms,
        prereq @ [ make_use_invariant id.loc env perms var ]

and sequence ?(acc = []) env perms s =
  match s with
    | [] ->
        env, perms, List.flatten (List.rev acc)
    | i :: rem ->
        let env, perms, i = instruction env perms i in
        sequence ~acc: (i :: acc) env perms rem

(******************************************************************************)

let rec collect_poly_in_type acc t =
  match t.node with
    | P.TUnit | P.TInt | P.TBool | P.TPointer _ | P.TLogicPointer ->
        acc
    | P.TPolyVar id ->
        let id = string_of_pident id in
        if StringSet.mem id acc then
          acc
        else
          StringSet.add id acc
    | P.TLogic (_, _, l) ->
        List.fold_left collect_poly_in_type acc (olf l)

let collect_function_type_parameters l =
  List.fold_left collect_poly_in_type StringSet.empty l

let decl_fun env loc name region_parameters arguments return_type
    consumes produces pre post body return =
  let region_parameters = olf region_parameters in
  let region_types =
    List.flatten
      (List.map (fun (_, _, l) -> olf l) (List.map snd region_parameters))
  in
  let type_parameters =
    collect_function_type_parameters
      (return_type :: List.map snd arguments @ region_types)
  in
  let type_parameters =
    List.map (fun s -> Ident.create s, ()) (StringSet.elements type_parameters)
  in
  let consumes = olf consumes in
  let produces = olf produces in
  let pre =
    Opt.if_none
      { loc = loc;
        node = P.LConst { loc = loc; node = P.CTrue } }
      pre
  in
  let post =
    Opt.if_none
      { loc = loc;
        node = P.LConst { loc = loc; node = P.CTrue } }
      post
  in
  let return =
    Opt.if_none
      { loc = loc;
        node = P.EConst { loc = loc; node = P.CUnit } }
      return
  in
  (* enter type parameters in environment *)
  let ienv =
    { env with poly_type_variables =
        Env.add_list env.poly_type_variables type_parameters }
  in
  (* first type regions and enter them in environment *)

  let ienv, region_parameters =
    list_fold_map
      (fun ienv region_parameter ->
        let a, b = named_region ienv region_parameter in
        let ienv = { ienv with regions = Env.add ienv.regions a b } in
        ienv, (a, b))
      ienv
      region_parameters
  in

(*  let region_parameters = List.map (named_region ienv) region_parameters in
  let ienv =
    { ienv with regions = Env.add_list ienv.regions region_parameters }
  in *)

  (* then type everything but the body, pre and post *)
  let name = Ident.create name in
  let ienv, arguments =
    list_fold_map
      (fun env (n, t) ->
        let n = Ident.create n in
        let t = type_expr env t in
        let env = { env with variables = Env.add env.variables n t } in
        env, (n, t))
      ienv
      arguments
  in
  let return_type = type_expr ienv return_type in
  let consumes = List.map (permission ienv) consumes in
  let produces = List.map (permission ienv) produces in
  let f =
    {
      f_name = name;
      f_region_parameters = region_parameters;
      f_type_parameters = List.map fst type_parameters;
      f_arguments = arguments;
      f_return_type = return_type;
      f_consumes = consumes;
      f_produces = produces;
      f_pre = PTrue;
      f_post = PTrue;
      f_body = [];
      f_return = EConst CUnit;
    }
  in
  (* local environment to type the body *)
  let ienv =
    {
      ienv with
(*        variables = Env.add_list ienv.variables arguments;*)
        functions = Env.add ienv.functions name f;
    }
  in
  let _, pre = predicate ienv consumes pre in
  let post_env =
    { ienv with variables = Env.add ienv.variables result return_type }
  in
  let _, post = predicate post_env produces post in
  let final_env, final_perms, body = sequence ienv consumes body in
  (* type return value and consume "produces" permissions *)
  let return_loc = return.loc in
  let final_env, final_perms, return_prereq, _ = (* infer adoptions *)
    expr_as final_env final_perms return return_type
  in
  let final_env, remaining_perms, prereq = (* check consumes *)
    last_consume_list ~where: "at the end of the function"
      return_loc final_env final_perms produces
  in
  let return = expr_as_no_adopt final_env final_perms return return_type in
  (* finish *)
  begin
    let scoped = scope_perms return_loc ienv remaining_perms in
    match fst scoped with
      | [] -> ()
      | _ ->
          warning
            "%aWarning: more permissions are produced than those you gave.@.%a"
            print_location loc
            pp_available remaining_perms
  end;
  let f =
    {
      f with
        f_body = body @ prereq;
        f_return = return;
        f_pre = pre;
        f_post = post;
    }
  in
  let env = { env with functions = Env.add env.functions name f } in
  env, DFun f

(******************************************************************************)

(*
let invariant env class_decl pred =
  let this_region = Ident.create ("region_of_" ^ this) in
  let this = Ident.create this in
  let region = URegion.create (RRoot this_region) in
  let ty = UType.create (TPointer region) in
  (* create local environment and class expr for "this" *)
  let env, ce_regions =
    list_fold_map
      (fun env (id, ce) ->
        { env with regions = Env.add env.regions id ce },
        URegion.create (RRoot id))
      env
      class_decl.c_region_parameters
  in
  let env, ce_types =
    list_fold_map
      (fun env id ->
        { env with poly_type_variables =
            Env.add env.poly_type_variables id () },
        UType.create (TPoly id))
      env
      class_decl.c_type_parameters
  in
  let ce = class_decl.c_name, ce_regions, ce_types in
  (* declare things in local environment *)
  let env =
    {
      env with
        variables = Env.add env.variables this ty;
        regions = Env.add env.regions this_region ce;
    }
  in
  let pred = predicate env pred in
  this, pred
*)

let invariant env cd inv =
  (* put fields and owned regions in environment *)
  let regions = Env.add_list env.regions cd.c_singles in
  let regions = Env.add_list regions cd.c_groups in
  let variables = Env.add_list env.variables cd.c_fields in
  let ienv = { env with regions = regions; variables = variables } in
  (* type invariant *)
  let _, p = predicate ienv [] inv in
  p

let decl_class env name region_parameters type_parameters singles groups
    fields inv =
  let region_parameters = olf region_parameters in
  let type_parameters = olf type_parameters in
  (* type region parameters first - THIS MEANS WE CAN'T USE THE CLASS BEING
     DEFINED IN ITS OWN REGION PARAMETERS - fixpoint? *)
  let region_parameters = List.map (named_region env) region_parameters in
  let type_parameters = List.map Ident.create type_parameters in
  (* dummy version of the class for recursion *)
  let c =
    {
      c_name = Ident.create name;
      c_region_parameters = region_parameters;
      c_type_parameters = type_parameters;
      c_singles = [];
      c_groups = [];
      c_fields = [];
      c_invariant = PTrue;
    }
  in
  (* internal environment uses dummy version of the class *)
  let ienv = { env with classes = Env.add env.classes c.c_name c } in
  let ienv = { ienv with regions = Env.add_list env.regions region_parameters } in

  log "****** Environnement des regions apres ajout region parameters *******@.%a@."
    (Env.pp pp_class_expr) ienv.regions;

  (* also declare type parameters *)
  let ienv =
    { ienv with poly_type_variables =
        Env.add_list env.poly_type_variables
          (List.map (fun i -> i, ()) type_parameters) }
  in
  (* internal environment with dummy owned regions *)
  let dummy_region (n, _) = Ident.create n, (Ident.dummy, [], []) in
  let dummy_singles = List.map dummy_region singles in
  let dummy_groups = List.map dummy_region groups in
  let ienv =
    { ienv with regions =
        Env.add_list ienv.regions (dummy_singles @ dummy_groups) }
  in
  (* intermediate version of the class with owned regions *)
  let type_owned (_, ce) (id, _) = id, class_expr ienv ce in
  let singles = List.map2 type_owned singles dummy_singles in
  let groups = List.map2 type_owned groups dummy_groups in
  let c =
    {
      c with
        c_singles = singles;
        c_groups = groups;
    }
  in
  let ienv = { ienv with classes = Env.replace env.classes c.c_name c } in

(*  (* intermediate version of the class with owned regions *)
  let singles = List.map (named_region ienv) singles in
  let groups = List.map (named_region ienv) groups in
  let c =
    {
      c with
        c_singles = singles;
        c_groups = groups;
    }
  in
  let ienv = { ienv with classes = Env.replace env.classes c.c_name c } in *)

  (* intermediate version of the class with fields *)
  let iienv =
    List.fold_left
      (fun env (name, ce) ->
        { env with regions = Env.add env.regions name ce })
      ienv
      (singles @ groups)
  in

  log "****** Environnement des regions pour typer les champs *******@.%a@."
    (Env.pp pp_class_expr) iienv.regions;

  let fields = List.map (named_type_expr iienv) fields in
  let c = { c with c_fields = fields } in
  let ienv = { env with classes = Env.add env.classes c.c_name c } in
  (* now we can type the invariant *)
  let invariant =
    match inv with
      | None -> PTrue
      | Some inv -> invariant ienv c inv
  in
  let c = { c with c_invariant = invariant } in
  let env = { env with classes = Env.add env.classes c.c_name c } in
  env, T.DClass c

(******************************************************************************)

let decl_type env name args =
  let args = olf args in
  let name = Ident.create name in
  let parameters = List.map Ident.create args in
  let td =
    {
      t_name = name;
      t_parameters = parameters;
    }
  in
  let env = { env with types = Env.add env.types name td } in
  env, T.DType td

let decl_logic env loc name regions args other_types =
  let regions = olf regions in
  let args = olf args in
  let name = Ident.create name in

  let ienv = env in

  (* type parameters *)
  let region_types =
    List.flatten
      (List.map (fun (_, _, l) -> olf l) (List.map snd regions))
  in
  let poly_type_variables =
    List.fold_left
      collect_poly_in_type
      StringSet.empty
      (other_types @ region_types @ List.map snd args)
  in
  let poly_type_variables = StringSet.elements poly_type_variables in
  let ienv, type_params =
    list_fold_map
      (fun env tp ->
        let tp = Ident.create tp in
        let env =
          { env with poly_type_variables =
              Env.add env.poly_type_variables tp () } in
        env, tp)
      ienv
      poly_type_variables
  in

  (* regions *)
  let ienv, regions =
    list_fold_map
      (fun env (rn, ce) ->
        let ce = class_expr env ce in
        let rn = Ident.create rn in
        let env = { env with regions = Env.add env.regions rn ce } in
        env, (rn, ce))
      ienv
      regions
  in

  (* parameters *)
  let ienv, parameters =
    list_fold_map
      (fun env (pn, pty) ->
        let pty = type_expr env pty in
        let pn = Ident.create pn in
        let env = { env with variables = Env.add env.variables pn pty } in
        env, (pn, pty))
      ienv
      args
  in

  name, type_params, regions, parameters, ienv

let decl_logic_fun env loc name regions args ret body =
  let name, type_params, regions, parameters, ienv =
    decl_logic env loc name regions args [ ret ]
  in
  let ret = type_expr ienv ret in
  let body = Opt.map (fun t -> term_as ienv t ret) body in
  let lfd =
    {
      lf_name = name;
      lf_type_parameters = type_params;
      lf_regions = regions;
      lf_parameters = parameters;
      lf_return_type = ret;
      lf_body = body;
    }
  in
  let env =
    { env with logic_functions = Env.add env.logic_functions name lfd }
  in
  env, T.DLogicFun lfd

let decl_predicate env loc name regions args body =
  let name, type_params, regions, parameters, ienv =
    decl_logic env loc name regions args []
  in
  let body = Opt.map (fun t -> predicate ienv [] t) body in
  let p =
    {
      p_name = name;
      p_type_parameters = type_params;
      p_regions = regions;
      p_parameters = parameters;
      p_body = Opt.map snd body;
    }
  in
  let env =
    { env with predicates = Env.add env.predicates name p }
  in
  env, T.DPredicate p

let rec collect_poly_in_logic acc p =
  match p.node with
    | P.LVar _
    | P.LConst _ ->
        acc
    | P.LIn (a, _)
    | P.LNot a
    | P.LGet (_, a, _)
    | P.LAt (a, _)
    | P.LAtPre a ->
        collect_poly_in_logic acc a
    | P.LBinOp (a, _, b) ->
        let acc = collect_poly_in_logic acc a in
        collect_poly_in_logic acc b
    | P.LApp (_, _, l) ->
        List.fold_left collect_poly_in_logic acc l
    | P.LForallRegion (_, (_, _, tl), a)
    | P.LExistsRegion (_, (_, _, tl), a) ->
        let acc = List.fold_left collect_poly_in_type acc (olf tl) in
        collect_poly_in_logic acc a
    | P.LForall (_, t, a)
    | P.LExists (_, t, a) ->
        let acc = collect_poly_in_type acc t in
        collect_poly_in_logic acc a

let decl_axiom env loc lemma name body =
  let type_params = collect_poly_in_logic StringSet.empty body in
  let type_params = StringSet.elements type_params in
  let type_params = List.map (fun x -> Ident.create x, ()) type_params in
  let ienv =
    { env with poly_type_variables =
        Env.add_list env.poly_type_variables type_params }
  in
  let body = predicate ienv [] body in
  let decl =
    {
      a_lemma = lemma;
      a_name = Ident.create name;
      a_type_parameters = List.map fst type_params;
      a_body = snd body;
    }
  in
  env, DAxiom decl

(******************************************************************************)

let decl env d =
  match d.node with
    | P.DFun (name, region_parameters, arguments, return_type,
              consumes, produces, pre, post, body, return) ->
        decl_fun env d.loc name
          region_parameters arguments return_type
          consumes produces pre post body return
    | P.DClass (name, region_parameters, type_parameters, owned_regions,
                fields, invariant) ->
        let singles =
          list_filterf
            (function
              | { node = Single }, a, b -> Some (a, b)
              | _ -> None)
            owned_regions
        in
        let groups =
          list_filterf
            (function
              | { node = Group }, a, b -> Some (a, b)
              | _ -> None)
            owned_regions
        in
        decl_class env name
          region_parameters type_parameters singles groups
          fields invariant
    | P.DType (name, args) ->
        decl_type env name args
    | P.DLogic (name, regions, args, Some ret, body) ->
        decl_logic_fun env d.loc name regions args ret body
    | P.DLogic (name, regions, args, None, body) ->
        decl_predicate env d.loc name regions args body
    | P.DAxiom (name, body) ->
        decl_axiom env d.loc false name body
    | P.DLemma (name, body) ->
        decl_axiom env d.loc true name body

let file f =
  list_fold_map decl empty_env f
