(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Misc
open Lang_ast
open Why
open Tast
open Types
open Pp
open Typinterp
open Icommon

type interp_env = {
  iregions: bool Env.t; (* region -> is reference (i.e. from program) *)
  hat: Model.hat;
}

(******************************************************************************)

let logic_path_of_path (root, rem) =
  let rem = List.map (fun (id, pp, own) -> LVar id, pp, own) rem in
  root, rem

let region_is_from_program ienv id =
  try
    Env.find ienv.iregions id
  with Not_found ->
    error "Don't know whether %a comes from the program or not." Ident.pp id

let region_term_var at env ienv id =
  if region_is_from_program ienv id then
    Model.term_region at env ienv.hat (id, [])
  else
    term_var id, []

(*
type path = Ident.t * (Ident.t * path_perm * Ident.t) list
*)
let rec path_of_logic_region_if_from_program ienv r: logic_path option =
  match r with
    | LRoot id ->
        begin
          try
            let is_program = Env.find ienv.iregions id in
            if is_program then
              Some (id, [])
            else
              None
          with Not_found ->
            None (* not in the environment -> not a program region *)
        end
    | LOwn (p_region, p, owned) ->
        match path_of_logic_region_if_from_program ienv p_region with
          | None ->
              None
          | Some (path_root, path_rem) ->
              let pp = PPGroup (* todo *) in
              Some (path_root, path_rem @ [ p, pp, owned ])

let rec term at env ienv t =
  let trm t = term at env ienv t in
  match t with
    | LConst CUnit ->
        term_void, []
    | LConst CTrue ->
        term_bool true, []
    | LConst CFalse ->
        term_bool false, []
    | LConst (CInt i) ->
        term_int i, []
    | LBinOp (a, op, b) ->
        let a, qa = trm a in
        let b, qb = trm b in
        let f =
          match op with
            | LSub -> term_sub
            | LAdd -> term_add
            | LMul -> term_mul
            | LDiv -> term_div
        in
        f a b, qa @ qb
    | LAt (t, l) ->
        term (Some l) env ienv t
    | LVar x ->
        term_var x, []

    (* Region-dependent *)
    | LGet (r, p, f) ->
        let p, qp = trm p in
        begin
          match path_of_logic_region_if_from_program ienv r with
            | None ->
                let r, qr = region_term at env ienv r in
                term_get f r p, qp @ qr
            | Some path ->
                let path, qpath = why_path_of_logic_path trm path in
                Model.term_field at ienv.hat path p f, qp @ qpath
        end
    | LApp (n, r, a) ->
(*
        log "APPLICATION LApp(%a, ..., ...)@." Ident.pp n;
*)
        let r = List.map (reconstruct_region_term at env ienv) r in
        let r, qr = List.split r in
        let a = List.map trm a in
        let a, qa = List.split a in
        let quantif = List.flatten (qr @ qa) in
(*
        log "-> QUANTIFICATIONS";
        List.iter (fun (x, _, _) -> log " %a" Ident.pp x) quantif;
        log ".@.";
*)
        term_app n (r @ a), quantif

(* Region-dependent *)
and region_term at env ienv r =
  match r with
    | LRoot id ->
        region_term_var at env ienv id
    | LOwn (s, p, f) ->
        let p, qp = term at env ienv p in
        let s, qs = region_term at env ienv s in
        term_get f s p, qp @ qs

and reconstruct_region_term at env ienv r =
  match path_of_logic_region_if_from_program ienv r with
    | None ->
        region_term at env ienv r
    | Some path ->
        let path, qpath = why_path_of_logic_path (term at env ienv) path in
        let term, quantif = Model.term_region at env ienv.hat path in
        term, quantif @ qpath

and why_path_of_logic_path trm ((root, rem): logic_path): why_path * _ list =
  let rem =
    List.map
      (fun (t, pp, own) ->
        let t, qt = trm t in
        (t, pp, own), qt)
      rem
  in
  let rem, q = List.split rem in
  (root, rem), List.flatten q

(* TODO: simplifier *)
let why_path_of_path at env ienv p =
  why_path_of_logic_path (term at env ienv) (logic_path_of_path p)

let quantify p q =
  List.fold_left
    (fun acc (x, t, p) -> a_forall x t (a_implies p acc))
    p
    q

let rec predicate_aux at env ienv p =
  let trm t = term at env ienv t in
  let pred ?(ienv = ienv) ?(at = at) p = predicate_aux at env ienv p in
  let quantify a b = a, b in
  match p with
    | PTrue ->
        a_true, []
    | PFalse ->
        a_false, []
    | PCompare (a, op, b) ->
        let f =
          match op with
            | LNeq -> a_neq
            | LEq -> a_eq
            | LLt -> a_lt
            | LLe -> a_le
            | LGt -> a_gt
            | LGe -> a_ge
        in
        let a, qa = trm a in
        let b, qb = trm b in
        quantify (f a b) (qa @ qb)
    | PLogicOp (a, op, b) ->
        let f =
          match op with
            | LOr -> a_or
            | LAnd -> a_and
            | LImpl -> a_implies
            | LIff -> a_iff
        in
        let a, qa = pred a in
        let b, qb = pred b in
        quantify (f a b) (qa @ qb)
    | PAt (p, at) ->
        pred ~at: (Some at) p
    | PQuantify (q, x, t, p) ->
        let t = typ env t in
        let p, qp = pred p in
        let q =
          match q with
            | Forall -> a_forall
            | Exists -> a_exists
        in
        quantify (q x t p) qp
    | PNot p ->
        let p, qp = pred p in
        quantify (a_not p) qp

    (* Region-dependent *)
    | PQuantifyRegion (q, x, ce, p) ->
        let argty = class_tuple_type env ce in
        let t = Whylib.region argty in
        let ienv =
          { ienv with iregions = Env.add ienv.iregions x false }
        in
        let p, qp = pred ~ienv p in
        let q =
          match q with
            | Forall -> a_forall
            | Exists -> a_exists
        in
        quantify (q x t p) qp
    | PIn (t, r) ->
        let t, qt = trm t in
        let r, qr = reconstruct_region_term at env ienv r in
        quantify (Whylib.in_region t r) (qt @ qr)
    | PApp (n, r, a) ->
        let r = List.map (reconstruct_region_term at env ienv) r in
        let r, qr = List.split r in
        let a = List.map trm a in
        let a, qa = List.split a in
        quantify (a_app n (r @ a)) (List.flatten (qr @ qa))

(* We compare quantifications based on the identifier being used.
   Indeed, quantifications are always reconstructed regions, and
   if the same region is reconstructed twice, it is reconstructed
   in the same way. The type is equal. The body is not actually equal,
   but this is only due to the fact that it contains a quantification
   on a pointer aux_p, whose identifier is recreated every time. *)
let compare_quantifications (a, _, _) (b, _, _) =
  Ident.compare a b

(*
let simplify_quantifications q =
(*  log "SIMPLIFY_QUANTIFICATIONS [";
  List.iter
    (fun (id, _, _) ->
      log " %a" Ident.upp id)
    q;
  log " ]@."; *)
  let q = List.sort compare_quantifications q in
  let rec aux acc a l =
    match l with
      | [] ->
          List.rev acc
      | x :: rem ->
          if compare_quantifications x a = 0 then
            aux acc a rem
          else
            aux (a :: acc) x rem
  in
  match q with
    | [] -> []
    | x :: rem -> aux [ x ] x rem
*)

let predicate at env ienv p =
  let p, q = predicate_aux at env ienv p in
(*  let q = simplify_quantifications q in*)
(*  quantify p q*)
  p, q

(******************************************************************************)

let invariant_predicate_name_of_class =
  imap_memo (fun cname -> Ident.create (Ident.name cname ^ "_inv"))

let invariant_predicate_of_class env cd =
  let name = invariant_predicate_name_of_class cd.c_name in
  let region_param (id, ce) =
    let argty = class_tuple_type env ce in
    id, Whylib.region argty
  in
  let params =
    List.map region_param cd.c_singles
    @ List.map region_param cd.c_groups
    @ List.map (fun (id, ty) -> id, typ env ty) cd.c_fields
  in
  let ienv_regions =
    List.map (fun (id, _) -> id, false) cd.c_singles
    @ List.map (fun (id, _) -> id, false) cd.c_groups
  in
  let ienv =
    {
      hat = Model.empty_hat;
      iregions = Env.add_list Env.empty ienv_regions
    }
  in
  let body, decls = predicate None env ienv cd.c_invariant in
  assert (decls = []);
(*  decls @ [ decl_predicate name params (Some body) ]*)
  decl_predicate name params (Some body)

let apply_invariant env ienv x (x_path: path) pp (cn, _, _) =
  let cd = find_class_decl env cn in
  let name = invariant_predicate_name_of_class cn in
  (* Region-dependent *)
  let path id = fst x_path, snd x_path @ [ x, pp, id ] in
  let make_regions l =
    List.map
      (fun (id, _) ->
        let path, qpath = why_path_of_path None env ienv (path id) in
        let r, qr = Model.term_region None env ienv.hat path in
        r, (qpath @ qr))
      l
  in
  let singles = make_regions cd.c_singles in
  let groups = make_regions cd.c_groups in
  let regions = singles @ groups in
  let regions, decls = List.split regions in
  let fields =
    List.map
      (fun (id, _) ->
        let path, qpath = why_path_of_path None env ienv x_path in
        Model.term_field None ienv.hat path (term_var x) id, qpath)
      (cd.c_fields)
  in
  let fields, qfields = List.split fields in
  let main = a_app name (regions @ fields) in
(*  quantify main (List.flatten (quantifiers @ qfields))*)
  main, List.flatten (decls @ qfields)

(******************************************************************************)

let rec expr env ienv e =
  let ex e = expr env ienv e in
  match e with
    | EConst CUnit ->
        const_void
    | EConst CTrue ->
        const_bool true
    | EConst CFalse ->
        const_bool false
    | EConst (CInt i) ->
        const_int i
    | EVar id ->
        var id
    | EBinOp (a, Sub, b) ->
        expr_sub (ex a) (ex b)
    | EBinOp (a, Neq, b) ->
        expr_not (expr_eq (ex a) (ex b))
    | EBinOp (a, Mul, b) ->
        expr_mul (ex a) (ex b)
    | EBinOp (a, Lt, b) ->
        expr_lt (ex a) (ex b)
    | EBinOp (a, Le, b) ->
        expr_le (ex a) (ex b)
    | EBinOp (a, Gt, b) ->
        expr_gt (ex a) (ex b)
    | EBinOp (a, Ge, b) ->
        expr_ge (ex a) (ex b)
    | EBinOp (a, Eq, b) ->
        expr_eq (ex a) (ex b)
    | EBinOp (a, Div, b) ->
        expr_div (ex a) (ex b)
    | EBinOp (a, Add, b) ->
        expr_add (ex a) (ex b)
    | EBinOp (a, And, b) ->
        expr_and (ex a) (ex b)
    | EBinOp (a, Or, b) ->
        expr_or (ex a) (ex b)

    (* Region-dependent *)
    | ESelect (x, f, path) ->
        Model.expr_field ienv.hat path (var x) f
    | EApp (n, r, a) ->
        let r = List.map expr_get_path (List.map snd r) in
        let a = List.map ex a in
        app_list (var n) (r @ a)

let assume_invariant env ienv x x_path pp x_class =
  let a_inv, decls = apply_invariant env ienv x x_path pp x_class in
  Why.assume a_inv, decls

let rec subst_term region_subst regular_subst label_subst term =
  let trm = subst_term region_subst regular_subst label_subst in
  let regtrm = subst_region_term region_subst regular_subst label_subst in
  match term with
    | LConst _ ->
        term
    | LBinOp (a, op, b) ->
        LBinOp (trm a, op, trm b)
    | LVar id ->
        begin
          try
            IMap.find id regular_subst
          with Not_found ->
            term
        end
    | LAt (a, l) ->
        let l =
          try
            IMap.find l label_subst
          with Not_found ->
            l
        in
        LAt (trm a, l)
    | LGet (r, a, f) ->
        LGet (regtrm r, trm a, f)
    | LApp (fname, regions, args) ->
        LApp (fname, List.map regtrm regions, List.map trm args)

and subst_region_term region_subst regular_subst label_subst regterm =
  let trm = subst_term region_subst regular_subst label_subst in
  let regtrm = subst_region_term region_subst regular_subst label_subst in
  match regterm with
    | LRoot id ->
        begin
          try
            IMap.find id region_subst
          with Not_found ->
            regterm
        end
    | LOwn (r, a, s) ->
        LOwn (regtrm r, trm a, s)

let rec subst_predicate region_subst regular_subst label_subst pred =
  let trm = subst_term region_subst regular_subst label_subst in
  let regtrm = subst_region_term region_subst regular_subst label_subst in
  let prd = subst_predicate region_subst regular_subst label_subst in
  match pred with
    | PTrue
    | PFalse ->
        pred
    | PCompare (a, op, b) ->
        PCompare (trm a, op, trm b)
    | PLogicOp (p, op, q) ->
        PLogicOp (prd p, op, prd q)
    | PIn (a, r) ->
        PIn (trm a, regtrm r)
    | PAt (p, l) ->
        let l =
          try
            IMap.find l label_subst
          with Not_found ->
            l
        in
        PAt (prd p, l)
    | PQuantify (q, x, t, p) ->
        PQuantify (q, x, t, prd p)
    | PQuantifyRegion (q, x, ce, p) ->
        PQuantifyRegion (q, x, ce, prd p)
    | PApp (fname, regs, args) ->
        PApp (fname, List.map regtrm regs, List.map trm args)
    | PNot p ->
        PNot (prd p)

let rec region_term_of_path f_ptr (root, path) =
  List.fold_left
    (fun acc (ptr, _, s) -> LOwn (acc, f_ptr ptr, s))
    (LRoot root)
    path

let term_op_of_binop op =
  match op with
    | Sub -> LSub
    | Mul -> LMul
    | Div -> LDiv
    | Add -> LAdd
    | _ -> assert false (* no equivalent expression *)

let rec term_of_expr expr =
  match expr with
    | EConst c ->
        LConst c
    | EVar x ->
        LVar x
    | ESelect (x, f, path) ->
        LGet (region_term_of_path (fun x -> LVar x) path, LVar x, f)
    | EBinOp (a, op, b) ->
        LBinOp (term_of_expr a, term_op_of_binop op, term_of_expr b)
    | EApp (fname, regs, args) -> (* of Ident.t * (region * path) list * expr list*)
        let regs =
          List.map
            (fun (_, path) -> region_term_of_path (fun x -> LVar x) path)
            regs
        in
        let args = List.map term_of_expr args in
        LApp (fname, regs, args)

let declarations: (Why.decl * Why.decl) IMap.t ref = ref IMap.empty

let add_declaration (id, d, axiom) =
  try
    let d_prev = IMap.find id !declarations in
(*    assert (d_prev = d) *)
    () (* TODO: verifier *)
  with Not_found ->
    declarations := IMap.add id (d, axiom) !declarations

let add_declarations l =
  List.iter add_declaration l

let make_declarations () =
  let decls =
    IMap.fold
      (fun _ (d, a) acc -> a :: d :: acc)
      !declarations
      []
  in
  declarations := IMap.empty;
  decls

let rec statement env wenv ienv s =
  let simple ?(ienv = ienv) x = seq2 x, ienv in
  let lets x y = expr_let x y, ienv in
  (* TODO: put regions in both environments *)
  let let_region r _ =
    let region_vars = Model.variables_of_region env ienv.hat r in
    let k body =
      List.fold_left
        (fun acc (r, _) ->
          expr_let r (expr_ref Whylib.expr_empty) acc)
        body
        region_vars
    in
    k,
    { ienv with iregions = Env.add ienv.iregions r true }
  in
  match s with
    | IAssert p ->
        let p, d = predicate None env ienv p in
        add_declarations d;
        simple (expr_assert p)
    | IAssume p ->
        let p, d = predicate None env ienv p in
        add_declarations d;
        simple (Why.assume p)
    | IIf (e, a, b) ->
        let e = expr env ienv e in
        let a, _ = sequence env wenv ienv a const_void in
        let b, _ = sequence env wenv ienv b const_void in
        simple (expr_if e a b)
    | ILet (x, e) ->
        lets x (expr env ienv e)
    | ILabel l ->
        label l, ienv
    | IUnpack _ ->
        simple const_void

    (* Region-dependent *)
    | ILetRegion (x, y) ->
        let_region x y

    | IAffect (x, f, path, e) ->
        let e = expr env ienv e in
        let assign = Model.assign_field ienv.hat path (var x) f e in
        simple assign

    | IPack (v, ce, path) ->
        let inv, d = apply_invariant env ienv v path PPSingle ce in
        add_declarations d;
        simple (expr_assert inv)

    | IFocus (v, var_path, _, target_path)
    | IUnfocus (_, var_path, _, target_path, v)
    | IAdopt (_, var_path, _, target_path, v) ->
        let copy = Model.copy_pointer ienv.hat env var_path v target_path in
        simple copy

(*
        (* expr_get_path var_path: C region (in which there is v of class C) *)
        (* we want to put the value of v in this C region,
           in the value of v of the target_path C region *)
        let set =
          expr_set_path (var target_root) target_path
            (fun _ ->
              let root = Whylib.expr_empty in
              Whylib.expr_set root (var v)
                (Whylib.expr_get
                   (expr_get_path var_path)
                   (var v)))
        in
        simple (assign target_root set)
*)

(*
    | IUnfocus (_, s_path, _, (target_root, target_path), focused_variable)
    | IAdopt (_, s_path, _, (target_root, target_path), focused_variable) ->
        let set =
          expr_set_path (deref target_root) target_path
            (fun target ->
              Whylib.expr_set target (var focused_variable)
                (Whylib.expr_get
                   (expr_get_path s_path)
                   (var focused_variable)))
        in
        simple (assign target_root set)
*)

    | INew (x, _, (*(target_root, target_path)*) target_path, ((cn, _, _) (*as ce*))) ->
(*
        let cd = find_class_decl env cn in
        let count =
          List.length cd.c_singles
          + List.length cd.c_groups
          + List.length cd.c_fields
        in
        let post =
          list_mapi
            (fun i _ ->
              a_eq
                (term_proj count i term_result)
                Whylib.term_empty)
            (cd.c_singles @ cd.c_groups)
        in
        let post = List.fold_left a_and a_true post in
        (* TODO: use model *)
        (* R := (((set !R) res) [ {true} (int, 'a larray) couple {true} ]));
           should become something like
           R_f := ...
           R_g := ... *)
        let empty_tuple =
          black_box
            (type_annot
               ~post
               (type_base (class_tuple_type env ce)))
        in
        let set =
          expr_set_path
            (deref target_root)
            target_path
            (fun target -> Whylib.expr_set target (var x) empty_tuple)
        in
        let assign = assign target_root set in
*)
        let assign = Model.allocate_pointer ienv.hat target_path (var x) in

        let fresh = black_box (type_base Whylib.pointer) in
        let k, ienv = lets x fresh in
        (fun next -> k (seq2 assign next)), ienv

    | IWeakenSingle _
    | IWeakenEmpty _ ->
        simple const_void

    | IUseInvariant (x, x_path, x_class) ->
        (* TODO: other than PPSingle? *)
        let inv, d = assume_invariant env ienv x x_path PPSingle x_class in
        add_declarations d;
        simple inv

    | ICall (x, fname, regs, args, return_type) ->
        (* TODO: pour que l'option -hat 0 serve a quelque chose il faut
           faire comme on faisait avant : copier les regions dans des
           references intermediaires et les remettre avec une egalite,
           pour mieux controler les effets *)
        let fd = find_fun_decl env fname in
        let label_before_call = Ident.create ("before_" ^ Ident.name x) in

        (* substitute pre and post *)
        let region_subst =
          List.fold_left2
            (fun acc (id, _) (reg, path) ->
              let reg = region_term_of_path (fun x -> LVar x) path in
              IMap.add id reg acc)
            IMap.empty
            fd.f_region_parameters
            regs
        in
        let regular_subst =
          List.fold_left2
            (fun acc (id, _) arg ->
              let arg = term_of_expr arg in
              IMap.add id arg acc)
            IMap.empty
            fd.f_arguments
            args
        in
        let label_subst =
          IMap.add Why.at_old label_before_call IMap.empty
        in
        let pre =
          subst_predicate region_subst regular_subst label_subst fd.f_pre
        in
        let post =
          subst_predicate region_subst regular_subst label_subst fd.f_post
        in

        (* interpret pre and post *)
        let pre, d = predicate None env ienv pre in
        add_declarations d;
        let post, d = predicate None env ienv post in
        add_declarations d;

        (* instanciate effects *)
        let paths =
          List.combine
            (List.map fst fd.f_region_parameters)
            (List.map snd regs)
        in
        let effects =
          try
            Env.find wenv fname
          with Not_found ->
            log "interp.ml, interpretation of function calls: \
cannot find effects for %a"
              Ident.pp fname;
            assert false (* error *)
        in
        let writes =
          List.flatten
            (List.map
               (fun r ->
                 let path =
                   try
                     List.assoc r paths
                   with Not_found ->
                     assert false (* error *)
                 in
                 List.map fst
                   (Model.variables_of_path env ienv.hat path))
               (ISet.elements effects))
        in

        (* TODO *)
        let reads = [] in

        let return_type = type_base (typ env return_type) in
        let call = black_box (type_annot ~pre ~post ~reads ~writes return_type) in
        lets x (label label_before_call call)

(*
        let fd = find_fun_decl env fname in
        let tmp_regs =
          List.map
            (fun (id, ce) ->
              Ident.create (Ident.name fname ^ "_" ^ Ident.name id))
            fd.f_region_parameters
        in
        let regs_why = List.map var tmp_regs in
        let args_why = List.map (expr env ienv) args in
        let call = app_list (var fname) (regs_why @ args_why) in
        let writes =
          try
            Env.find wenv fname
          with Not_found ->
            assert false (* error *)
        in
        let assigns =
          list_map3
            (fun id (reg, (root, path)) (formal, _) ->
              if ISet.mem formal writes then
                begin
                  let set =
                    expr_set_path (deref root) path (fun _ -> deref id)
                  in
                  Why.assign root set
                end
              else
                const_void)
            tmp_regs
            regs
            fd.f_region_parameters
        in
        let call_with_sets =
          let tmp = Ident.create "tmp" in
          expr_let tmp call (seq2 (seq assigns) (var tmp))
        in
        let call_full =
          List.fold_left2
            (fun acc id get_reg -> expr_let id (expr_ref get_reg) acc)
            call_with_sets
            tmp_regs
            (List.map expr_get_path (List.map snd regs))
        in
        lets x call_full
*)

and sequence env wenv ienv seq return =
  let k, ienv =
    List.fold_left
      (fun (acc, ienv) s ->
        let sk, ienv = statement env wenv ienv s in
        (fun k -> acc (sk k)), ienv)
      ((fun x -> x), ienv)
      seq
  in
  k return, ienv

(******************************************************************************)

let why_parameter ~name ~pre ~reads ~writes ~post ~args ~rty =
  let t =
    type_annot
      ~pre
      ~reads
      ~writes
      ~post
      rty
  in
  let t_full =
    List.fold_left
      (fun acc (id, ty) ->
        type_fun ~arg_name: id ~arg_type: ty acc)
      t
      (List.rev args)
  in
  decl_parameter name t_full

(*
let fun_parameter env fd pre reads writes post args rty =
  why_parameter
    ~name: fd.f_name
    ~pre
    ~reads
    ~writes
    ~post
    ~args
    ~rty
*)

let why_fun ~name ~pre ~post ~args ~body =
  decl_let name (expr_fun ~args ~pre ~post body)

let fun_goal env wenv ienv fd pre post args =
  let variables = fd.f_arguments in
  let env = { env with variables = Env.add_list env.variables variables } in
  let return = expr env ienv fd.f_return in
  let body, _ = sequence env wenv ienv fd.f_body return in
  why_fun
    ~name: (Ident.create (Ident.name fd.f_name ^ "_goal"))
    ~pre
    ~post
    ~args
    ~body

(* TODO: put regions in both environments *)
let decl_fun env wenv acc fd =
(*
  log "************ HAT FOR FUNCTION: %a ***************@." pp_ident fd.f_name;
  let hat = Model.hat_of_fun env fd in
  Model.print_hat hat;
  log "************ RESTRICT HAT WITH DEPTH = %d ***************@."
    !Options.hat_depth;
  let hat = Model.restrict_hat !Options.hat_depth hat in
  Model.print_hat hat;
  log "************ END OF HAT ***************@.";
*)
  let hat = Model.hat_of_fun env fd in
  let hat = Model.restrict_hat !Options.hat_depth hat in

  (* header *)
  let ienv_regions =
    List.map (fun (id, _) -> id, true) fd.f_region_parameters
  in
  let ienv =
    {
      iregions = Env.add_list Env.empty ienv_regions;
      hat = hat;
    }
  in
  let pre, d = predicate None env ienv fd.f_pre in
  add_declarations d;
  let post, d = predicate None env ienv fd.f_post in
  add_declarations d;

(*
  let reads = List.map fst fd.f_region_parameters in
*)
  let writes = Effects.fun_writes env wenv fd in

(*
  let region_args =
    List.map
      (fun (id, ce) ->
        let argty = class_tuple_type env ce in
        id, type_ref (type_base (Whylib.region argty)))
      fd.f_region_parameters
  in
*)
  let region_args =
    List.flatten
      (List.map
         (fun (id, _) -> Model.variables_of_region env hat id)
         fd.f_region_parameters)
  in
  let region_args =
    List.map (fun (n, t) -> n, type_ref (type_base t)) region_args
  in

  let variable_args =
    List.map
      (fun (id, ty) -> id, type_base (typ env ty))
      fd.f_arguments
  in
  let args = region_args @ variable_args in
(*
  let rty = type_base (typ env fd.f_return_type) in
  (* finish *)
  let writes_list = ISet.elements writes in
  let parameter = fun_parameter env fd pre reads writes_list post args rty in
*)
  let wenv = Env.add wenv fd.f_name writes in
  let goal = fun_goal env wenv ienv fd pre post args in
  let decls = make_declarations () in
  wenv, goal(* :: parameter*) :: decls @ acc

let define_get_using_tuple tuple_type count name i ty =
  let x = Ident.create "x" in
  decl_logic_def
    name
    ~args: [ x, tuple_type ]
    ty
    (term_proj count i (term_var x))

let define_set_using_tuple tuple_type count get set i ty =
  let x = Ident.create "x" in
  let y = Ident.create "y" in
  let rec tuple acc n =
    if n < 0 then
      acc
    else if n = i then
      tuple (term_var y :: acc) (n - 1)
    else
      tuple (term_proj count n (term_var x) :: acc) (n - 1)
  in
  decl_logic_def
    set
    ~args: [ x, tuple_type; y, ty ]
    tuple_type
    (term_tuple (tuple [] (count - 1)))

(* TODO: put regions in both environments *)
let decl_class env acc cd =
  let regions = List.map fst cd.c_region_parameters in
  let regions = List.map (fun r -> URegion.create (RRoot r)) regions in
  let types = List.map (fun t -> UType.create (TPoly t)) cd.c_type_parameters in
  let ce = cd.c_name, regions, types in
  let tuple_type = class_tuple_type env ce in

  let owneds =
    List.map
      (fun (owned_name, owned_ce) ->
        let get_name = region_get_name owned_name in
        let set_name = region_set_name owned_name in
        let owned_ty = class_tuple_type env owned_ce in
        let ty = Whylib.region owned_ty in
        get_name, set_name, ty)
      (cd.c_singles @ cd.c_groups)
  in
  let fields =
    List.map
      (fun (field_name, ty) ->
        let get_name = region_get_name field_name in
        let set_name = region_set_name field_name in
        let ty = typ env ty in
        get_name, set_name, ty)
      (cd.c_fields)
  in
  let projs = owneds @ fields in
  let count = List.length projs in
  let gets =
    list_mapi
      (fun i (name, _, ty) -> define_get_using_tuple tuple_type count name i ty)
      projs
  in
  let sets =
    list_mapi
      (fun i (get, set, ty) ->
        define_set_using_tuple tuple_type count get set i ty)
      projs
  in

  (* TODO: define these gets and sets using tuples *)
  let invariant_predicate = invariant_predicate_of_class env cd in
  invariant_predicate :: gets @ sets @ acc

(******************************************************************************)

let decl_type env acc td =
  decl_logic_type ~args: td.t_parameters td.t_name :: acc

let logic_parameters env acc regions parameters =
  let regions' =
    List.map
      (fun (id, x) -> id, Whylib.region (class_tuple_type env x))
      regions
  in
  let parameters' =
    List.map
      (fun (id, ty) -> id, typ env ty)
      parameters
  in
  let iregions = List.map (fun (id, _) -> id, false) regions in
  let ienv =
    {
      iregions = Env.add_list Env.empty iregions;
      hat = Model.empty_hat;
    }
  in
  regions' @ parameters', ienv

let decl_logic_fun env acc lfd =
  (* Use ienv for TODO below *)
  let args, _ = logic_parameters env acc lfd.lf_regions lfd.lf_parameters in
  let args = List.map snd args in
  let ret = typ env lfd.lf_return_type in
  let decl = Why.decl_logic lfd.lf_name ~args ret in
  begin
    match lfd.lf_body with
      | None -> ()
      | Some _ -> assert false (* TODO *)
  end;
  decl :: acc

let decl_predicate env acc p =
  let args, ienv = logic_parameters env acc p.p_regions p.p_parameters in
  let body =
    match p.p_body with
      | None ->
          None
      | Some body ->
          let p, d = predicate None env ienv body in
          assert (d = []);
          Some p
  in
  let decl = Why.decl_predicate p.p_name args body in
  decl :: acc

let decl_axiom env acc ad =
  let ienv =
    {
      iregions = Env.empty;
      hat = Model.empty_hat;
    }
  in
  let body, d = predicate None env ienv ad.a_body in
  assert (d = []);
  let decl = Why.decl_axiom ad.a_lemma ad.a_name body in
  decl :: acc

(******************************************************************************)

let decl env (wenv, acc) d =
  match d with
    | DFun fd -> decl_fun env wenv acc fd
    | DClass cd -> wenv, decl_class env acc cd
    | DType td -> wenv, decl_type env acc td
    | DLogicFun lfd -> wenv, decl_logic_fun env acc lfd
    | DPredicate p -> wenv, decl_predicate env acc p
    | DAxiom ad -> wenv, decl_axiom env acc ad

let file env ast =
  let wenv = Env.empty in
  Why.file (List.rev (snd (List.fold_left (decl env) (wenv, []) ast)))
