open Lexing

let print_location fmt (b, e) =
  Format.fprintf fmt "File \"%s\", line %d, characters %d-%d:@."
    b.pos_fname b.pos_lnum (b.pos_cnum - b.pos_bol) (e.pos_cnum - b.pos_bol)

let dummy_location = Lexing.dummy_pos, Lexing.dummy_pos

let of_lexbuf entry lexbuf =
  try
    entry Lang_lexer.token lexbuf
  with
    | Parsing.Parse_error ->
        let loc = lexeme_start_p lexbuf, lexeme_end_p lexbuf in
        raise (Lang_ast.Parse_error (loc, "Parse error"))

let of_channel entry ch =
  of_lexbuf entry (Lexing.from_channel ch)

let of_string entry str =
  of_lexbuf entry (Lexing.from_string str)

let of_function entry f =
  of_lexbuf entry (Lexing.from_function f)

let of_file entry file =
  let ch = open_in file in
  try
    let lexbuf = Lexing.from_channel ch in
    lexbuf.lex_curr_p <- { lexbuf.lex_curr_p with pos_fname = file };
    let result = of_lexbuf entry lexbuf in
    close_in ch;
    result
  with exn ->
    close_in ch;
    raise exn

let region_of_channel = of_channel Lang_parser.region
let region_of_file = of_file Lang_parser.region
let region_of_function = of_function Lang_parser.region
let region_of_string = of_string Lang_parser.region

let type_expr_of_channel = of_channel Lang_parser.type_expr
let type_expr_of_file = of_file Lang_parser.type_expr
let type_expr_of_function = of_function Lang_parser.type_expr
let type_expr_of_string = of_string Lang_parser.type_expr

let expr_of_channel = of_channel Lang_parser.expr
let expr_of_file = of_file Lang_parser.expr
let expr_of_function = of_function Lang_parser.expr
let expr_of_string = of_string Lang_parser.expr

let const_of_channel = of_channel Lang_parser.const
let const_of_file = of_file Lang_parser.const
let const_of_function = of_function Lang_parser.const
let const_of_string = of_string Lang_parser.const

let instruction_of_channel = of_channel Lang_parser.instruction
let instruction_of_file = of_file Lang_parser.instruction
let instruction_of_function = of_function Lang_parser.instruction
let instruction_of_string = of_string Lang_parser.instruction

let logic_expr_of_channel = of_channel Lang_parser.logic_expr
let logic_expr_of_file = of_file Lang_parser.logic_expr
let logic_expr_of_function = of_function Lang_parser.logic_expr
let logic_expr_of_string = of_string Lang_parser.logic_expr

let logic_region_of_channel = of_channel Lang_parser.logic_region
let logic_region_of_file = of_file Lang_parser.logic_region
let logic_region_of_function = of_function Lang_parser.logic_region
let logic_region_of_string = of_string Lang_parser.logic_region

let permission_of_channel = of_channel Lang_parser.permission
let permission_of_file = of_file Lang_parser.permission
let permission_of_function = of_function Lang_parser.permission
let permission_of_string = of_string Lang_parser.permission

let class_expr_of_channel = of_channel Lang_parser.class_expr
let class_expr_of_file = of_file Lang_parser.class_expr
let class_expr_of_function = of_function Lang_parser.class_expr
let class_expr_of_string = of_string Lang_parser.class_expr

let decl_of_channel = of_channel Lang_parser.decl
let decl_of_file = of_file Lang_parser.decl
let decl_of_function = of_function Lang_parser.decl
let decl_of_string = of_string Lang_parser.decl

let file_of_channel = of_channel Lang_parser.file
let file_of_file = of_file Lang_parser.file
let file_of_function = of_function Lang_parser.file
let file_of_string = of_string Lang_parser.file

let ident_of_channel = of_channel Lang_parser.ident
let ident_of_file = of_file Lang_parser.ident
let ident_of_function = of_function Lang_parser.ident
let ident_of_string = of_string Lang_parser.ident

open Format

let print_int _ fmt x = fprintf fmt "%d" x
let print_string _ fmt x = fprintf fmt "%S" x
let print_bool _ fmt x = fprintf fmt "%B" x
let print_unit _ fmt () = fprintf fmt "()"
let print_node a par fmt x = a par fmt x.Lang_ast.node

let print_list a _ fmt x =
  match x with
    | [] ->
        fprintf fmt "[]";
    | x :: r ->
        fprintf fmt "@[<hv 2>[ ";
        fprintf fmt "%a" (a false) x;
        List.iter (fprintf fmt ";@ %a" (a false)) r;
        fprintf fmt " ]@]"

let print_option a par fmt x =
  match x with
    | None -> fprintf fmt "None"
    | Some x ->
        if par then fprintf fmt "(";
        fprintf fmt "@[<hv 2>Some@ %a@]" (a true) x;
        if par then fprintf fmt ")"

let rec print_region par fmt x =
  (print_node print_region_case) par fmt x

and print_type_expr par fmt x =
  (print_node print_type_expr_case) par fmt x

and print_expr par fmt x =
  (print_node print_expr_case) par fmt x

and print_const par fmt x =
  (print_node print_expr_econst) par fmt x

and print_instruction par fmt x =
  (print_node print_instruction_case) par fmt x

and print_sequence par fmt x =
  (print_list (print_node print_instruction_case)) par fmt x

and print_logic_expr par fmt x =
  (print_node print_instruction_iassert) par fmt x

and print_logic_region par fmt x =
  (print_node print_instruction_iassert_lapp) par fmt x

and print_permission par fmt x =
  (print_node print_permission_case) par fmt x

and print_class_expr par fmt x =
  (fun par fmt (x1, x2, x3) ->
    fprintf fmt "@[<hv 2>";
    if par then fprintf fmt "(";
    fprintf fmt "%a" (print_string false) x1;
    fprintf fmt ",@ %a" ((print_option
                           (print_list (print_node print_region_case))) false) x2;
    fprintf fmt ",@ %a" ((print_option
                           (print_list (print_node print_type_expr_case))) false) x3;
    fprintf fmt "@]";
    if par then fprintf fmt ")")
  par
  fmt
  x

and print_decl par fmt x =
  (print_node print_decl_case) par fmt x

and print_file par fmt x =
  (print_list (print_node print_decl_case)) par fmt x

and print_ident par fmt x =
  (print_node print_region_rown) par fmt x

and print_region_rown par fmt = function
  | Lang_ast.Ident x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>Ident@ %a@]" (print_string true) x1;
      if par then fprintf fmt ")"

and print_decl_case par fmt = function
  | Lang_ast.DAxiom (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>DAxiom (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.DClass (x1, x2, x3, x4, x5, x6) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>DClass (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_option
                             (print_list
                               (fun par fmt (x1, x2) ->
                                 fprintf fmt "@[<hv 2>";
                                 if par then fprintf fmt "(";
                                 fprintf fmt "%a" (print_string false) x1;
                                 fprintf fmt ",@ %a" ((fun par fmt (x1, x2,
                                                                    x3) ->
                                                        fprintf fmt "@[<hv 2>";
                                                        if par then fprintf fmt "(";
                                                        fprintf fmt "%a" (print_string false) x1;
                                                        fprintf fmt ",@ %a" (
                                                        (print_option
                                                          (print_list
                                                            (print_node
                                                              print_region_case))) false) x2;
                                                        fprintf fmt ",@ %a" (
                                                        (print_option
                                                          (print_list
                                                            (print_node
                                                              print_type_expr_case))) false) x3;
                                                        fprintf fmt "@]";
                                                        if par then fprintf fmt ")") false) x2;
                                 fprintf fmt "@]";
                                 if par then fprintf fmt ")"))) false) x2;
      fprintf fmt ",@ %a" ((print_option (print_list print_string)) false) x3;
      fprintf fmt ",@ %a" ((print_list
                             (fun par fmt (x1, x2, x3) ->
                               fprintf fmt "@[<hv 2>";
                               if par then fprintf fmt "(";
                               fprintf fmt "%a" ((print_node
                                                   print_decl_dclass) false) x1;
                               fprintf fmt ",@ %a" (print_string false) x2;
                               fprintf fmt ",@ %a" ((fun par fmt (x1, x2, x3) ->
                                                      fprintf fmt "@[<hv 2>";
                                                      if par then fprintf fmt "(";
                                                      fprintf fmt "%a" (print_string false) x1;
                                                      fprintf fmt ",@ %a" (
                                                      (print_option
                                                        (print_list
                                                          (print_node
                                                            print_region_case))) false) x2;
                                                      fprintf fmt ",@ %a" (
                                                      (print_option
                                                        (print_list
                                                          (print_node
                                                            print_type_expr_case))) false) x3;
                                                      fprintf fmt "@]";
                                                      if par then fprintf fmt ")") false) x3;
                               fprintf fmt "@]";
                               if par then fprintf fmt ")")) false) x4;
      fprintf fmt ",@ %a" ((print_list
                             (fun par fmt (x1, x2) ->
                               fprintf fmt "@[<hv 2>";
                               if par then fprintf fmt "(";
                               fprintf fmt "%a" (print_string false) x1;
                               fprintf fmt ",@ %a" ((print_node
                                                      print_type_expr_case) false) x2;
                               fprintf fmt "@]";
                               if par then fprintf fmt ")")) false) x5;
      fprintf fmt ",@ %a" ((print_option
                             (print_node print_instruction_iassert)) false) x6;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.DFun (x1, x2, x3, x4, x5, x6, x7, x8, x9, x10) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>DFun (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_option
                             (print_list
                               (fun par fmt (x1, x2) ->
                                 fprintf fmt "@[<hv 2>";
                                 if par then fprintf fmt "(";
                                 fprintf fmt "%a" (print_string false) x1;
                                 fprintf fmt ",@ %a" ((fun par fmt (x1, x2,
                                                                    x3) ->
                                                        fprintf fmt "@[<hv 2>";
                                                        if par then fprintf fmt "(";
                                                        fprintf fmt "%a" (print_string false) x1;
                                                        fprintf fmt ",@ %a" (
                                                        (print_option
                                                          (print_list
                                                            (print_node
                                                              print_region_case))) false) x2;
                                                        fprintf fmt ",@ %a" (
                                                        (print_option
                                                          (print_list
                                                            (print_node
                                                              print_type_expr_case))) false) x3;
                                                        fprintf fmt "@]";
                                                        if par then fprintf fmt ")") false) x2;
                                 fprintf fmt "@]";
                                 if par then fprintf fmt ")"))) false) x2;
      fprintf fmt ",@ %a" ((print_list
                             (fun par fmt (x1, x2) ->
                               fprintf fmt "@[<hv 2>";
                               if par then fprintf fmt "(";
                               fprintf fmt "%a" (print_string false) x1;
                               fprintf fmt ",@ %a" ((print_node
                                                      print_type_expr_case) false) x2;
                               fprintf fmt "@]";
                               if par then fprintf fmt ")")) false) x3;
      fprintf fmt ",@ %a" ((print_node print_type_expr_case) false) x4;
      fprintf fmt ",@ %a" ((print_option
                             (print_list (print_node print_permission_case))) false) x5;
      fprintf fmt ",@ %a" ((print_option
                             (print_list (print_node print_permission_case))) false) x6;
      fprintf fmt ",@ %a" ((print_option
                             (print_node print_instruction_iassert)) false) x7;
      fprintf fmt ",@ %a" ((print_option
                             (print_node print_instruction_iassert)) false) x8;
      fprintf fmt ",@ %a" ((print_list (print_node print_instruction_case)) false) x9;
      fprintf fmt ",@ %a" ((print_option (print_node print_expr_case)) false) x10;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.DLemma (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>DLemma (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.DLogic (x1, x2, x3, x4, x5) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>DLogic (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_option
                             (print_list
                               (fun par fmt (x1, x2) ->
                                 fprintf fmt "@[<hv 2>";
                                 if par then fprintf fmt "(";
                                 fprintf fmt "%a" (print_string false) x1;
                                 fprintf fmt ",@ %a" ((fun par fmt (x1, x2,
                                                                    x3) ->
                                                        fprintf fmt "@[<hv 2>";
                                                        if par then fprintf fmt "(";
                                                        fprintf fmt "%a" (print_string false) x1;
                                                        fprintf fmt ",@ %a" (
                                                        (print_option
                                                          (print_list
                                                            (print_node
                                                              print_region_case))) false) x2;
                                                        fprintf fmt ",@ %a" (
                                                        (print_option
                                                          (print_list
                                                            (print_node
                                                              print_type_expr_case))) false) x3;
                                                        fprintf fmt "@]";
                                                        if par then fprintf fmt ")") false) x2;
                                 fprintf fmt "@]";
                                 if par then fprintf fmt ")"))) false) x2;
      fprintf fmt ",@ %a" ((print_option
                             (print_list
                               (fun par fmt (x1, x2) ->
                                 fprintf fmt "@[<hv 2>";
                                 if par then fprintf fmt "(";
                                 fprintf fmt "%a" (print_string false) x1;
                                 fprintf fmt ",@ %a" ((print_node
                                                        print_type_expr_case) false) x2;
                                 fprintf fmt "@]";
                                 if par then fprintf fmt ")"))) false) x3;
      fprintf fmt ",@ %a" ((print_option (print_node print_type_expr_case)) false) x4;
      fprintf fmt ",@ %a" ((print_option
                             (print_node print_instruction_iassert)) false) x5;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.DType (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>DType (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_option (print_list print_string)) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"

and print_decl_dclass par fmt = function
  | Lang_ast.Group -> fprintf fmt "Group"
  | Lang_ast.Single -> fprintf fmt "Single"

and print_permission_case par fmt = function
  | Lang_ast.PClosed x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>PClosed@ %a@]" ((print_node print_region_case) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.PEmpty x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>PEmpty@ %a@]" ((print_node print_region_case) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.PFocus (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>PFocus (@,";
      ((print_node print_region_case) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_case) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.PGroup x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>PGroup@ %a@]" ((print_node print_region_case) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.POpen x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>POpen@ %a@]" ((print_node print_region_case) true) x1;
      if par then fprintf fmt ")"

and print_instruction_iassert_lapp par fmt = function
  | Lang_ast.LOwn (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LOwn (@,";
      ((print_node print_instruction_iassert_lapp) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x2;
      fprintf fmt ",@ %a" (print_string false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LRoot x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LRoot@ %a@]" ((print_node print_region_rown) true) x1;
      if par then fprintf fmt ")"

and print_instruction_iassert_lbinop par fmt = function
  | Lang_ast.LAdd -> fprintf fmt "LAdd"
  | Lang_ast.LAnd -> fprintf fmt "LAnd"
  | Lang_ast.LDiv -> fprintf fmt "LDiv"
  | Lang_ast.LEq -> fprintf fmt "LEq"
  | Lang_ast.LGe -> fprintf fmt "LGe"
  | Lang_ast.LGt -> fprintf fmt "LGt"
  | Lang_ast.LIff -> fprintf fmt "LIff"
  | Lang_ast.LImpl -> fprintf fmt "LImpl"
  | Lang_ast.LLe -> fprintf fmt "LLe"
  | Lang_ast.LLt -> fprintf fmt "LLt"
  | Lang_ast.LMul -> fprintf fmt "LMul"
  | Lang_ast.LNeq -> fprintf fmt "LNeq"
  | Lang_ast.LOr -> fprintf fmt "LOr"
  | Lang_ast.LSub -> fprintf fmt "LSub"

and print_instruction_iassert par fmt = function
  | Lang_ast.LApp (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LApp (@,";
      ((print_node print_region_rown) false) fmt x1;
      fprintf fmt ",@ %a" ((print_option
                             (print_list
                               (print_node print_instruction_iassert_lapp))) false) x2;
      fprintf fmt ",@ %a" ((print_list
                             (print_node print_instruction_iassert)) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LAt (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LAt (@,";
      ((print_node print_instruction_iassert) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_rown) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LAtPre x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LAtPre@ %a@]" ((print_node
                                            print_instruction_iassert) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.LBinOp (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LBinOp (@,";
      ((print_node print_instruction_iassert) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert_lbinop) false) x2;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LConst x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LConst@ %a@]" ((print_node print_expr_econst) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.LExists (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LExists (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_type_expr_case) false) x2;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LExistsRegion (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LExistsRegion (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((fun par fmt (x1, x2, x3) ->
                             fprintf fmt "@[<hv 2>";
                             if par then fprintf fmt "(";
                             fprintf fmt "%a" (print_string false) x1;
                             fprintf fmt ",@ %a" ((print_option
                                                    (print_list
                                                      (print_node
                                                        print_region_case))) false) x2;
                             fprintf fmt ",@ %a" ((print_option
                                                    (print_list
                                                      (print_node
                                                        print_type_expr_case))) false) x3;
                             fprintf fmt "@]";
                             if par then fprintf fmt ")") false) x2;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LForall (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LForall (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_type_expr_case) false) x2;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LForallRegion (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LForallRegion (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((fun par fmt (x1, x2, x3) ->
                             fprintf fmt "@[<hv 2>";
                             if par then fprintf fmt "(";
                             fprintf fmt "%a" (print_string false) x1;
                             fprintf fmt ",@ %a" ((print_option
                                                    (print_list
                                                      (print_node
                                                        print_region_case))) false) x2;
                             fprintf fmt ",@ %a" ((print_option
                                                    (print_list
                                                      (print_node
                                                        print_type_expr_case))) false) x3;
                             fprintf fmt "@]";
                             if par then fprintf fmt ")") false) x2;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LGet (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LGet (@,";
      ((print_node print_instruction_iassert_lapp) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert) false) x2;
      fprintf fmt ",@ %a" (print_string false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LIn (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LIn (@,";
      ((print_node print_instruction_iassert) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_instruction_iassert_lapp) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.LNot x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LNot@ %a@]" ((print_node
                                          print_instruction_iassert) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.LVar x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>LVar@ %a@]" ((print_node print_region_rown) true) x1;
      if par then fprintf fmt ")"

and print_instruction_case par fmt = function
  | Lang_ast.IAdopt (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IAdopt (@,";
      ((print_node print_region_case) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_case) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.IAffect (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IAffect (@,";
      ((print_node print_region_rown) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_rown) false) x2;
      fprintf fmt ",@ %a" ((print_node print_expr_case) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.IAssert x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IAssert@ %a@]" ((print_node
                                             print_instruction_iassert) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.IAssume x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IAssume@ %a@]" ((print_node
                                             print_instruction_iassert) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.ICall (x1, x2, x3, x4) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>ICall (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_rown) false) x2;
      fprintf fmt ",@ %a" ((print_option
                             (print_list (print_node print_region_case))) false) x3;
      fprintf fmt ",@ %a" ((print_list (print_node print_expr_case)) false) x4;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.IFocus (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IFocus (@,";
      ((print_node print_region_rown) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_case) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.IIf (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IIf (@,";
      ((print_node print_expr_case) false) fmt x1;
      fprintf fmt ",@ %a" ((print_list (print_node print_instruction_case)) false) x2;
      fprintf fmt ",@ %a" ((print_list (print_node print_instruction_case)) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.ILabel x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>ILabel@ %a@]" (print_string true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.ILet (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>ILet (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_expr_case) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.ILetRegion (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>ILetRegion (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((fun par fmt (x1, x2, x3) ->
                             fprintf fmt "@[<hv 2>";
                             if par then fprintf fmt "(";
                             fprintf fmt "%a" (print_string false) x1;
                             fprintf fmt ",@ %a" ((print_option
                                                    (print_list
                                                      (print_node
                                                        print_region_case))) false) x2;
                             fprintf fmt ",@ %a" ((print_option
                                                    (print_list
                                                      (print_node
                                                        print_type_expr_case))) false) x3;
                             fprintf fmt "@]";
                             if par then fprintf fmt ")") false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.INew (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>INew (@,";
      (print_string false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_case) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.IPack x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IPack@ %a@]" ((print_node print_region_rown) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.IUnfocus (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IUnfocus (@,";
      ((print_node print_region_case) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_case) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.IUnpack x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IUnpack@ %a@]" ((print_node print_region_rown) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.IUse x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IUse@ %a@]" ((print_node
                                          print_instruction_iassert) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.IUseInvariant x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IUseInvariant@ %a@]" ((print_node
                                                   print_region_rown) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.IWeakenEmpty x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IWeakenEmpty@ %a@]" ((print_node
                                                  print_region_case) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.IWeakenSingle x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>IWeakenSingle@ %a@]" ((print_node
                                                   print_region_case) true) x1;
      if par then fprintf fmt ")"

and print_expr_econst par fmt = function
  | Lang_ast.CFalse -> fprintf fmt "CFalse"
  | Lang_ast.CInt x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>CInt@ %a@]" (print_int true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.CTrue -> fprintf fmt "CTrue"
  | Lang_ast.CUnit -> fprintf fmt "CUnit"

and print_expr_ebinop par fmt = function
  | Lang_ast.Add -> fprintf fmt "Add"
  | Lang_ast.And -> fprintf fmt "And"
  | Lang_ast.Div -> fprintf fmt "Div"
  | Lang_ast.Eq -> fprintf fmt "Eq"
  | Lang_ast.Ge -> fprintf fmt "Ge"
  | Lang_ast.Gt -> fprintf fmt "Gt"
  | Lang_ast.Le -> fprintf fmt "Le"
  | Lang_ast.Lt -> fprintf fmt "Lt"
  | Lang_ast.Mul -> fprintf fmt "Mul"
  | Lang_ast.Neq -> fprintf fmt "Neq"
  | Lang_ast.Or -> fprintf fmt "Or"
  | Lang_ast.Sub -> fprintf fmt "Sub"

and print_expr_case par fmt = function
  | Lang_ast.EApp (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>EApp (@,";
      ((print_node print_region_rown) false) fmt x1;
      fprintf fmt ",@ %a" ((print_option
                             (print_list (print_node print_region_case))) false) x2;
      fprintf fmt ",@ %a" ((print_list (print_node print_expr_case)) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.EBinOp (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>EBinOp (@,";
      ((print_node print_expr_case) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_expr_ebinop) false) x2;
      fprintf fmt ",@ %a" ((print_node print_expr_case) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.EConst x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>EConst@ %a@]" ((print_node print_expr_econst) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.ESelect (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>ESelect (@,";
      ((print_node print_region_rown) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_rown) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.EVar x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>EVar@ %a@]" ((print_node print_region_rown) true) x1;
      if par then fprintf fmt ")"

and print_type_expr_case par fmt = function
  | Lang_ast.TBool -> fprintf fmt "TBool"
  | Lang_ast.TInt -> fprintf fmt "TInt"
  | Lang_ast.TLogic (x1, x2, x3) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>TLogic (@,";
      ((print_node print_region_rown) false) fmt x1;
      fprintf fmt ",@ %a" ((print_option
                             (print_list (print_node print_region_case))) false) x2;
      fprintf fmt ",@ %a" ((print_option
                             (print_list (print_node print_type_expr_case))) false) x3;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.TLogicPointer -> fprintf fmt "TLogicPointer"
  | Lang_ast.TPointer x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>TPointer@ %a@]" ((print_node print_region_case) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.TPolyVar x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>TPolyVar@ %a@]" ((print_node print_region_rown) true) x1;
      if par then fprintf fmt ")"
  | Lang_ast.TUnit -> fprintf fmt "TUnit"

and print_region_case par fmt = function
  | Lang_ast.ROwn (x1, x2) ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>ROwn (@,";
      ((print_node print_region_rown) false) fmt x1;
      fprintf fmt ",@ %a" ((print_node print_region_rown) false) x2;
      fprintf fmt ")@]";
      if par then fprintf fmt ")"
  | Lang_ast.RRoot x1 ->
      if par then fprintf fmt "(";
      fprintf fmt "@[<hv 2>RRoot@ %a@]" ((print_node print_region_rown) true) x1;
      if par then fprintf fmt ")"
