(** Lang abstract syntax tree *)

(** {2 Locations} *)

type location = Lexing.position * Lexing.position

type 'a node = {
  node: 'a;
  loc: location;
}

exception Parse_error of location * string

(** {2 Entry Points Types} *)

type region_rown = | Ident of string

and decl_case =
  | DType of string * string list option
  | DLogic of
      string *
      (string * class_expr) list option *
      (string * type_expr) list option *
      type_expr option *
      logic_expr option
  | DLemma of string * logic_expr
  | DFun of
      string *
      (string * class_expr) list option *
      (string * type_expr) list *
      type_expr *
      permission list option *
      permission list option *
      logic_expr option *
      logic_expr option *
      instruction list *
      expr option
  | DClass of
      string *
      (string * class_expr) list option *
      string list option *
      (decl_dclass node * string * class_expr) list *
      (string * type_expr) list *
      logic_expr option
  | DAxiom of string * logic_expr

and decl_dclass = | Single | Group

and permission_case =
  | POpen of region
  | PGroup of region
  | PFocus of region * region
  | PEmpty of region
  | PClosed of region

and instruction_iassert_lapp =
  | LRoot of ident | LOwn of logic_region * logic_expr * string

and instruction_iassert_lbinop =
  | LSub
  | LOr
  | LNeq
  | LMul
  | LLt
  | LLe
  | LImpl
  | LIff
  | LGt
  | LGe
  | LEq
  | LDiv
  | LAnd
  | LAdd

and instruction_iassert =
  | LVar of ident
  | LNot of logic_expr
  | LIn of logic_expr * logic_region
  | LGet of logic_region * logic_expr * string
  | LForallRegion of string * class_expr * logic_expr
  | LForall of string * type_expr * logic_expr
  | LExistsRegion of string * class_expr * logic_expr
  | LExists of string * type_expr * logic_expr
  | LConst of const
  | LBinOp of logic_expr * instruction_iassert_lbinop node * logic_expr
  | LAtPre of logic_expr
  | LAt of logic_expr * ident
  | LApp of ident * logic_region list option * logic_expr list

and instruction_case =
  | IWeakenSingle of region
  | IWeakenEmpty of region
  | IUseInvariant of ident
  | IUse of logic_expr
  | IUnpack of ident
  | IUnfocus of region * region
  | IPack of ident
  | INew of string * region
  | ILetRegion of string * class_expr
  | ILet of string * expr
  | ILabel of string
  | IIf of expr * instruction list * instruction list
  | IFocus of ident * region
  | ICall of string * ident * region list option * expr list
  | IAssume of logic_expr
  | IAssert of logic_expr
  | IAffect of ident * ident * expr
  | IAdopt of region * region

and expr_econst = | CUnit | CTrue | CInt of int | CFalse

and expr_ebinop =
  | Sub | Or | Neq | Mul | Lt | Le | Gt | Ge | Eq | Div | And | Add

and expr_case =
  | EVar of ident
  | ESelect of ident * ident
  | EConst of const
  | EBinOp of expr * expr_ebinop node * expr
  | EApp of ident * region list option * expr list

and type_expr_case =
  | TUnit
  | TPolyVar of ident
  | TPointer of region
  | TLogicPointer
  | TLogic of ident * region list option * type_expr list option
  | TInt
  | TBool

and region_case = | RRoot of ident | ROwn of ident * ident

and region = region_case node

and type_expr = type_expr_case node

and expr = expr_case node

and const = expr_econst node

and instruction = instruction_case node

and logic_expr = instruction_iassert node

and logic_region = instruction_iassert_lapp node

and permission = permission_case node

and class_expr = (string * region list option * type_expr list option)

and decl = decl_case node

and file = decl list

and ident = region_rown node
