{
  open Lexing
  open Lang_ast
  open Lang_parser

  let parse_error lexbuf s =
    Printf.ksprintf
      (fun s ->
         let loc = lexeme_start_p lexbuf, lexeme_end_p lexbuf in
         raise (Parse_error (loc, s)))
      s

  let identifier lexbuf = function
    | "adopt" -> KWadopt
    | "as" -> KWas
    | "assert" -> KWassert
    | "assume" -> KWassume
    | "axiom" -> KWaxiom
    | "bool" -> KWbool
    | "class" -> KWclass
    | "consumes" -> KWconsumes
    | "else" -> KWelse
    | "empty" -> KWempty
    | "exists" -> KWexists
    | "false" -> KWfalse
    | "focus" -> KWfocus
    | "forall" -> KWforall
    | "fun" -> KWfun
    | "get" -> KWget
    | "group" -> KWgroup
    | "if" -> KWif
    | "in" -> KWin
    | "int" -> KWint
    | "invariant" -> KWinvariant
    | "label" -> KWlabel
    | "lemma" -> KWlemma
    | "let" -> KWlet
    | "logic" -> KWlogic
    | "new" -> KWnew
    | "not" -> KWnot
    | "pack" -> KWpack
    | "post" -> KWpost
    | "pre" -> KWpre
    | "produces" -> KWproduces
    | "region" -> KWregion
    | "return" -> KWreturn
    | "single" -> KWsingle
    | "then" -> KWthen
    | "true" -> KWtrue
    | "type" -> KWtype
    | "unfocus" -> KWunfocus
    | "unit" -> KWunit
    | "unpack" -> KWunpack
    | "use" -> KWuse
    | "weaken" -> KWweaken
    | s -> IDENT s
}

rule token = parse
  | [' ' '\t' '\r']* { token lexbuf }
  | '\n' { new_line lexbuf; token lexbuf }
  | "(*" { comment lexbuf; token lexbuf }
  | ['a'-'z' 'A'-'Z' '_'] ['a'-'z' 'A'-'Z' '_' '0'-'9']* as x
      { identifier lexbuf x }
  | "#" { SHARP }
  | "&&" { AMPERSAND_AMPERSAND }
  | "'" { QUOTE }
  | "(" { LPAR }
  | ")" { RPAR }
  | "*" { STAR }
  | "+" { PLUS }
  | "," { COMMA }
  | "-" { MINUS }
  | "->" { MINUS_GT }
  | "." { DOT }
  | "/" { DIV }
  | ":" { COLON }
  | ";" { SEMI }
  | "<" { LT }
  | "<-" { LT_MINUS }
  | "<=" { LT_EQUAL }
  | "<=>" { LT_EQUAL_GT }
  | "<>" { LT_GT }
  | "=" { EQUAL }
  | "==>" { EQUAL_EQUAL_GT }
  | ">" { GT }
  | ">=" { GT_EQUAL }
  | "?" { QUESTION }
  | "@" { AROBASE }
  | "[" { LBRACKET }
  | "]" { RBRACKET }
  | "^c" { HAT_c }
  | "^e" { HAT_e }
  | "^g" { HAT_g }
  | "^o" { HAT_o }
  | "{" { LBRACE }
  | "||" { VERT_VERT }
  | "}" { RBRACE }
  | ['0'-'9']+ as x
      { try
          INT (int_of_string x)
        with Failure "int_of_string" ->
          parse_error lexbuf "Invalid integer literal: `%s'" x }
  | eof { EOF }
  | _ as c { parse_error lexbuf "Invalid character: `%c'" c }

and comment = parse
  | "*)" { }
  | "(*" { comment lexbuf; comment lexbuf }
  | '\n' { new_line lexbuf; comment lexbuf }
  | eof { parse_error lexbuf "Unterminated comment" }
  | _ { comment lexbuf }
