(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Format
open Lexing

module StringSet = Set.Make(String)
module StringMap = Map.Make(String)

module IMap = Map.Make(Ident)
module ISet = Set.Make(Ident)

let imap_memo f =
  let tbl = ref IMap.empty in
  fun id ->
    try
      IMap.find id !tbl
    with Not_found ->
      let r = f id in
      tbl := IMap.add id r !tbl;
      r

let error s =
  let buf = Buffer.create 128 in
  let fmt = formatter_of_buffer buf in
  kfprintf
    (fun _ ->
       fprintf fmt "@?";
       printf "@?";
       eprintf "%s@." (Buffer.contents buf);
       exit 1)
    fmt
    s

let warnings = ref []

let warning s =
  let buf = Buffer.create 128 in
  let fmt = formatter_of_buffer buf in
  kfprintf
    (fun _ ->
      fprintf fmt "@?";
      warnings := Buffer.contents buf :: !warnings)
    fmt
    s

let print_warnings () =
  List.iter
    (printf "%s@.")
    (List.rev !warnings)

let () =
  at_exit print_warnings

let rec list_fold_map f acc env = function
  | [] -> env, List.rev acc
  | x :: r ->
      let env, x = f env x in
      list_fold_map f (x :: acc) env r
let list_fold_map f env l = list_fold_map f [] env l

let rec list_remove acc x = function
  | [] -> raise Not_found
  | y :: rem when x = y -> List.rev_append acc rem
  | y :: rem -> list_remove (y :: acc) x rem
let list_remove x = list_remove [] x

let rec list_removef acc f = function
  | [] -> raise Not_found
  | y :: rem when f y -> List.rev_append acc rem
  | y :: rem -> list_removef (y :: acc) f rem
let list_removef f x = list_removef [] f x

let rec list_findf f = function
  | [] ->
      raise Not_found
  | x :: r ->
      match f x with
        | None ->
            list_findf f r
        | Some y ->
            y

let rec list_filterf ?(acc = []) f = function
  | [] ->
      List.rev acc
  | x :: r ->
      let acc =
        match f x with
          | None -> acc
          | Some y -> y :: acc
      in
      list_filterf ~acc f r

let rec list_iteri ?(i = 0) f = function
  | [] -> ()
  | x :: r -> f i x; list_iteri ~i: (i + 1) f r

let rec list_mapi ?(i = 0) ?(acc = []) f = function
  | [] -> List.rev acc
  | x :: r -> list_mapi ~i: (i + 1) ~acc: (f i x :: acc) f r

let rec list_map3 ?(acc = []) f a b c =
  match a, b, c with
    | [], [], [] ->
        List.rev acc
    | xa :: ra, xb :: rb, xc :: rc ->
        list_map3 ~acc: (f xa xb xc :: acc) f ra rb rc
    | _ ->
        invalid_arg "list_map3"

let (|>) x f = f x

module Opt = struct
  let iter f = function
    | None -> ()
    | Some x -> f x

  let map f = function
    | None -> None
    | Some x -> Some (f x)

  let fold f x = function
    | None -> x
    | Some y -> f x y

  let filter f = function
    | None -> None
    | Some y -> if f y then Some y else None

  let if_none v = function
    | None -> v
    | Some x -> x
end

(* option list flattening *)
let olf = function
  | None -> []
  | Some l -> l

let log s = kfprintf (fun fmt -> ()) std_formatter s

let loge s = kfprintf (fun fmt -> ()) err_formatter s

let fresh =
  let i = ref (-1) in
  fun () ->
    incr i;
    !i

let memoize f =
  let tbl = Hashtbl.create 42 in
  fun x ->
    try
      Hashtbl.find tbl x
    with Not_found ->
      let r = f x in
      Hashtbl.add tbl x r;
      r

let dummy_loc name =
  let p = { Lexing.dummy_pos with Lexing.pos_fname = name } in
  p, p
