(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Pp
open Misc
open Tast
open Format
open Typinterp
open Icommon

type label = Ident.t option

type hat = region_info IMap.t

and region_info = {
  region_name: Ident.t;
  region_flattened_name: Ident.t;
  class_expr: class_expr;
  class_fields: (Ident.t * Why.logic_type) list;
  class_singles: Ident.t list;
  class_groups: Ident.t list;
  class_tuple_type: Why.logic_type;
  model: model;
}

and model =
  | Border
  | Flatten of flattening

and flattening = {
  fields: field_info IMap.t;
  owns: hat;
}

and field_info = {
  field_name: Ident.t;
  field_flattened_name: Ident.t;
  field_type: typ;
}

(******************************************************************************)

let rec print_hat ?(indent = 0) hat =
  let indent_string = String.make indent ' ' in
  IMap.iter
    (fun _ info ->
      printf "%s- %a: %a (class_expr: %a)"
        indent_string
        pp_ident info.region_name
        pp_ident info.region_flattened_name
        pp_class_expr info.class_expr;
      match info.model with
        | Border ->
            printf "@."
        | Flatten flat ->
            printf " (flatten)@.";
            IMap.iter
              (fun name field ->
                printf "%s  * %a: %a (type: %a)@."
                  indent_string
                  pp_ident field.field_name
                  pp_ident field.field_flattened_name
                  pp_type field.field_type)
              flat.fields;
            print_hat ~indent: (indent + 2) flat.owns)
    hat

(******************************************************************************)

let rec restrict_hat depth hat =
  IMap.mapi (restrict_info depth) hat

and restrict_info depth _ info =
  { info with model = restrict_model depth info.model }

and restrict_model depth model =
  if depth <= 0 then
    Border
  else
    match model with
      | Border _ ->
          model
      | Flatten flat ->
          Flatten (restrict_flattening depth flat)

and restrict_flattening depth flat =
  {
    fields = flat.fields;
    owns = restrict_hat (depth - 1) flat.owns;
  }

(******************************************************************************)

let concat_names a b =
  if a = "" then
    b
  else
    a ^ "_" ^ b

let make_region_info env current_name name class_expr =
  let cd =
    try
      Env.find env.classes (let (cn, _, _) = class_expr in cn)
    with Not_found ->
      assert false
  in
  let flattened_name = concat_names current_name (Ident.name name) in
  {
    region_name = name;
    region_flattened_name = Ident.create flattened_name;
    class_expr = class_expr;
    class_fields = List.map (fun (n, t) -> n, typ env t) cd.c_fields;
    class_singles = List.map fst cd.c_singles;
    class_groups = List.map fst cd.c_groups;
    class_tuple_type = class_tuple_type env class_expr;
    model = Border;
  }

let enter_new_region env current_name hat name class_expr =
  let info = make_region_info env current_name name class_expr in
  IMap.add name info hat

let flattening_of_info env info =
  let owner = Ident.create "dummy_owner" in (* not important for types *)
  let singles, groups, fields =
    repr_of_class_expr env info.class_expr owner
  in
  let fields =
    List.fold_left
      (fun acc (name, typ) ->
        let flattened_name =
          concat_names
            (Ident.name info.region_flattened_name)
            (Ident.name name)
        in
        let fi =
          {
            field_name = name;
            field_flattened_name = Ident.create flattened_name;
            field_type = typ;
          }
        in
        IMap.add name fi acc)
      IMap.empty
      fields
  in
  let owns =
    List.fold_left
      (fun acc (name, ce) ->
        enter_new_region
          env
          (Ident.name info.region_flattened_name)
          acc
          name
          ce)
      IMap.empty
      (singles @ groups)
  in
  {
    fields = fields;
    owns = owns;
  }

let expand_info env info =
  match info.model with
    | Border _ ->
        { info with model = Flatten (flattening_of_info env info) }
    | Flatten _ ->
        info

let rec insert_in_info flatten env info path =
  match path with
    | [] ->
        if flatten then
          expand_info env info
        else
          info
    | _ ->
        let info = expand_info env info in
        match info.model with
          | Border _ ->
              assert false (* was expanded just above *)
          | Flatten flat ->
              let owns = insert_in_hat flatten env flat.owns path in
              let flat = { flat with owns = owns } in
              { info with model = Flatten flat }

(* requires path not to be empty, and its root to already be in the hat *)
and insert_in_hat flatten env hat path =
  match path with
    | [] ->
        assert false (* pre *)
    | root :: rem ->
        let info =
          try
            IMap.find root hat
          with Not_found ->
            assert false (* pre *)
        in
        let info = insert_in_info flatten env info rem in
        IMap.add root info hat

let simplify_path (root, path) =
  let path =
    List.map
      (fun (_, _, own) -> own)
      path
  in
  root :: path

let enter_path_border env hat path =
  insert_in_hat false env hat (simplify_path path)

let enter_path_flatten env hat path =
  insert_in_hat true env hat (simplify_path path)

(******************************************************************************)

let empty_hat = IMap.empty

let rec hat_of_term env hat t =
  hat
(*
  match t with
    | _ -> assert false (* TODO *)
*)

let rec hat_of_predicate env hat p =
  hat
(*
  match p with
    | _ -> assert false (* TODO *)
*)

let hat_of_region_path_list (env: Tast.env) hat rpl =
  List.fold_left
    (fun acc (_, path) ->
      enter_path_border env acc path)
    hat
    rpl

let rec hat_of_expr (env: Tast.env) hat e =
  let expr = hat_of_expr env in
  match e with
    | EConst _ | EVar _ ->
        hat
    | ESelect (_, _, path_to_x) ->
        enter_path_flatten env hat path_to_x
    | EBinOp (a, _, b) ->
        expr (expr hat a) b
    | EApp (f, rpl, el) ->
        let hat = hat_of_expr_list env hat el in
        hat_of_region_path_list env hat rpl

and hat_of_expr_list (env: Tast.env) hat el =
  List.fold_left
    (hat_of_expr env)
    hat
    el

let rec hat_of_instruction (env: Tast.env) (hat: hat) i: hat =
  let flatten = enter_path_flatten env in
  let border = enter_path_border env in
  let expr = hat_of_expr env in
  let sequence = hat_of_sequence env in
  match i with
    | IWeakenSingle _
    | IWeakenEmpty _
    | IUnpack _
    | ILabel _
      ->
        (* TODO if those operation actually do something, which they do
           not right now *)
        hat

    | IUnfocus (_, path1, _, path2, _)
    | IFocus (_, path1, _, path2)
    | IAdopt (_, path1, _, path2, _)
      ->
        border (border hat path1) path2

    | IPack (_, _, path)
    | IUseInvariant (_, path, _)
      ->
        flatten hat path

    | IAffect (_, _, path, e) ->
        let hat = expr hat e in
        flatten hat path

    | INew (_, _, path, _) ->
        border hat path
    | ILet (_, e) ->
        expr hat e
    | IIf (e, s1, s2) ->
        let hat = expr hat e in
        let hat = sequence hat s1 in
        sequence hat s2

    | IAssert p
    | IAssume p
      ->
        hat_of_predicate env hat p

    | ICall (_, _, rpl, el, _) ->
        (* TODO: pre / post (substituted) *)
        let hat = hat_of_expr_list env hat el in
        hat_of_region_path_list env hat rpl

    | ILetRegion (id, ce) ->
        enter_new_region env "" hat id ce

and hat_of_sequence (env: Tast.env) hat s =
  List.fold_left (hat_of_instruction env) hat s

let hat_of_fun (env: Tast.env) fd =
  let hat =
    List.fold_left
      (fun acc (name, ce) -> enter_new_region env "" acc name ce)
      IMap.empty
      fd.f_region_parameters
  in
  let hat = hat_of_predicate env hat fd.f_pre in
  let hat = hat_of_predicate env hat fd.f_post in
  let hat = hat_of_sequence env hat fd.f_body in
  let hat = hat_of_expr env hat fd.f_return in
  hat

(******************************************************************************)

(* met [depth] fois le type region autour de [t] *)
let rec regionize t depth =
  if depth <= 0 then
    t
  else
    Whylib.region (regionize t (depth - 1))

let variables_of_fields env depth fields =
  IMap.fold
    (fun _ fi acc ->
      (fi.field_flattened_name,
       regionize (typ env fi.field_type) depth) :: acc)
    fields
    []

let rec variables_of_hat env depth hat =
  IMap.fold
    (fun _ info acc ->
      variables_of_info env depth info @ acc)
    hat
    []

and variables_of_info env depth info =
  match info.model with
    | Border ->
        [ info.region_flattened_name,
          regionize (class_tuple_type env info.class_expr) depth ]
    | Flatten flat ->
        let fields = variables_of_fields env depth flat.fields in
        let owns = variables_of_hat env (depth + 1) flat.owns in
        owns @ fields

let variables_of_region env hat region =
  let info =
    try
      IMap.find region hat
    with Not_found ->
      assert false (* misconstructed hat *)
  in
  variables_of_info env 1 info

let rec variables_of_path depth env hat (root, path) =
(*  log "depth %d, variables_of_path %a (rem %d)@." depth Ident.pp root (List.length path);*)
  let root_info =
    try
      IMap.find root hat
    with Not_found ->
      assert false (* misconstructed hat *)
  in
  let result =
    match path, root_info.model with
      | [], _ | _, Border ->
          variables_of_info env depth root_info
      | (_, _, own) :: rem, Flatten flat ->
          variables_of_path (depth + 1) env flat.owns (own, rem)
  in
(*  log "depth %d, variables_of_path %a (rem %d) DONE@." depth Ident.pp root (List.length path);*)
  result

let variables_of_path = variables_of_path 1

(******************************************************************************)

(* Split a path using a hat.
   Return [info, pointers, remaining_path]
   where
   - [info: region_info] describes where the path leads in the hat;
   - [pointers: Ident.t list] is the part of the path which is in the hat;
   - [remaining_path: path] is the part of the path which is not. *)
let rec split_path hat (root, path) =
  let info =
    try
      IMap.find root hat
    with Not_found ->
      assert false
  in
  match info.model with
    | Border _ ->
        info, [], path
    | Flatten flat ->
        match path with
          | [] ->
              info, [], []
(*          | (_, PPNone, _) :: _ ->
              error "split_path: cannot access regions with no permissions"
          | (_, PPFocus, _) :: _ ->
              error "split_path: cannot access focused regions"*)
          | (pointer, _, own) :: rem ->
              let info, pointers, rem_path = split_path flat.owns (own, rem) in
              info, pointer :: pointers, rem_path

(*
let split_path hat (root, path) =
  let info, pointers, rem_path = split_path hat (root, path) in
  log "@[<hv 2>split_path of@ %a, [@[<hov 2>" pp_ident root;
  List.iter
    (fun (x, _, r) ->
      log "@ %a, %a;" pp_ident x pp_ident r)
    path;
  log "@]]@ info.region_flattened_name = %a@ pointers =@ [@[<hov 2>"
    pp_ident info.region_flattened_name;
  List.iter (log "@ %a;" pp_ident) pointers;
  log "@]]@ rem_path =@ [@[<hov 2>";
  List.iter
    (fun (x, _, r) ->
      log "@ %a, %a;" pp_ident x pp_ident r)
    rem_path;
  log "@]]@]@.";
  info, pointers, rem_path
*)

(******************************************************************************)

let get_path expr_get var root path =
  List.fold_left
    (fun acc (v, perm, own) ->
      match perm with
        | PPNone ->
            error "Model.get_path: no permission (own %a)"
              pp_ident own
        | PPFocus ->
            error
              "Model.get_path: focus permission (cannot read in focus yet)"
        | PPEmpty
        | PPGroup
        | PPSingle ->
            expr_get own acc (var v))
    root
    (List.rev path)

let get_field
    get get_pointer var deref
    hat path pointer field =
  let info, pointers, rem_path = split_path hat path in
  let get_pointers e =
    List.fold_left
      (fun acc p -> get_pointer acc (var p))
      e
      pointers
  in
  match info.model with
    | Border ->
        (* start from id, get pointers, apply remaining path, then get field *)
        let e = deref info.region_flattened_name in
        let e = get_pointers e in
        let e = get_path get var e rem_path in
        get field e pointer
    | Flatten flat ->
        assert (rem_path = []);
        (* start from field id, then get pointers *)
        let fi =
          try
            IMap.find field flat.fields
          with Not_found ->
            assert false
        in
        get_pointer (get_pointers (deref fi.field_flattened_name)) pointer

let expr_field hat path pointer field =
  get_field expr_get Whylib.expr_get Why.var Why.deref
    hat path pointer field

let term_field at hat (path: why_path) pointer field =
  get_field term_get Whylib.term_get (fun x -> x) (Why.term_deref ?at)
    hat path pointer field

let why_path_of_path (root, rem) =
  let rem = List.map (fun (id, pp, own) -> Why.term_var id, pp, own) rem in
  root, rem

(*
let quantify p q =
  List.fold_left
    (fun acc (x, t, p) -> Why.a_forall x t (Why.a_implies p acc))
    p
    q
*)

let quantify p q =
  List.fold_left
    (fun acc (x, t) -> Why.a_forall x t acc)
    p
    q

let rec region_constraint at env hat (path: why_path) flat reg get_pointers =
  let p_id = Ident.create "aux_p" in
  let p_ty = Whylib.pointer in
  let fields =
    IMap.fold
      (fun _ fi acc ->
        let root = Why.term_deref ?at fi.field_flattened_name in
        let root = get_pointers root in
        let eq =
          Why.a_eq
            (Whylib.term_get root (Why.term_var p_id))
            (term_get fi.field_name reg (Why.term_var p_id))
        in
        eq :: acc)
      flat.fields
      []
  in
  let owns, decls =
    IMap.fold
      (fun own_id ri (acc_constraints, acc_decls) ->
        (* TODO: PPGroup ?? *)
        let path_root, path_rem = path in
        let path_with_p = path_root, path_rem @ [ Why.term_var p_id, PPGroup, own_id ] in
        let own_reg_term, decls = term_region at env hat path_with_p in

        (* pas sur que ca soit correct pour tous les cas... *)
        let reconstructed_own = term_get own_id reg (Why.term_var p_id) in

        let eq = Why.a_eq own_reg_term reconstructed_own in
        eq :: acc_constraints, decls @ acc_decls)
      flat.owns
      ([], [])
  in
  let body =
    List.fold_left Why.a_and Why.a_true (fields @ owns)
  in
  Why.a_forall p_id p_ty body, decls

and term_region at env hat (path: why_path) =
  let info, pointers, rem_path = split_path hat path in
  let get_pointers root =
    List.fold_left
      (fun acc p -> Whylib.term_get acc p)
      root
      pointers
  in
  match info.model with
    | Border ->
        let root = Why.term_deref ?at info.region_flattened_name in
        let root = get_pointers root in
        let t = get_path term_get (fun x -> x) root rem_path in
        t, []
    | Flatten flat ->
        assert (rem_path = []);
(*        let reg_id = Ident.create (Ident.name info.region_flattened_name) in*)
        (* I chose not to recreate the identifier, as quantifiers which overlap
           are actually the same reconstructions of the same regions.
           This allows to avoid using multiple reconstructions of the same
           region (see Interp.simplify_quantifications). *)
        let reg_id = info.region_flattened_name in
        let reg_id_unique = Ident.create (Ident.uname reg_id) in (* HACK *)
        let reg_ty = Whylib.region info.class_tuple_type in
(*        let pred =
          region_constraint at env hat path flat (Why.term_var reg_id)
            get_pointers
        in *)
        let variables = variables_of_info env 1 info in
        let args = List.map snd variables in
        let d_fun =
          Why.decl_logic
            reg_id_unique
            ~args
            reg_ty
        in
        let pred, d =
          region_constraint at env hat path flat
            (Why.term_var reg_id)
            (fun x -> x)
        in
        assert (d = []);
        let make_arg (var, _) =
          let root = Why.term_deref ?at var in
          root
        in
        let pred =
          Why.a_let reg_id
            (Why.term_app reg_id_unique (List.map make_arg variables))
            pred
        in
        let pred = quantify pred variables in
        (*
          let pred =
          Why.a_forall reg_id reg_ty pred
        in
        *)
        let d_axiom =
          Why.decl_axiom ~is_lemma: false reg_id pred
        in
        let make_arg (var, _) =
          let root = Why.term_deref ?at var in
          let root = get_pointers root in
          root
        in
        Why.term_app reg_id_unique (List.map make_arg variables),
        [ reg_id, d_fun, d_axiom ]

let expr_region (hat: hat) env (path: path) =
  let info, pointers, rem_path = split_path hat path in
  let get_pointers get var root =
    List.fold_left
      (fun acc p -> get acc (var p))
      root
      pointers
  in
  match info.model with
    | Border ->
        let root = Why.deref info.region_flattened_name in
        let root = get_pointers Whylib.expr_get Why.var root in
        let t = get_path expr_get Why.var root rem_path in
        t
    | Flatten flat ->
        assert (rem_path = []);
        let reg_ty = Whylib.region info.class_tuple_type in
        let why_path = why_path_of_path path in
        let post, decls =
          region_constraint None env hat why_path flat Why.term_result
            (get_pointers Whylib.term_get Why.term_var)
        in
        assert (decls = []);
        Why.black_box
          (Why.type_annot
             ~post
             (Why.type_base reg_ty))

(******************************************************************************)

type gen_path =
  | Expr of Why.expr
  | ExprField of Why.expr * Ident.t

let rec deep_set path expr value =
  match path with
    | Expr e :: rem ->
        let expr' = Whylib.expr_get expr e in
        let value' = deep_set rem expr' value in
        Whylib.expr_set expr e value'
    | ExprField (e, f) :: rem ->
        let expr' = expr_get f expr e in
        let value' = deep_set rem expr' value in
        expr_set f expr e value'
    | [] ->
        value

let make_gen_path pointers path =
  List.map (fun e -> Expr e) pointers
  @ List.map (fun (x, _, f) -> ExprField (x, f)) path

let make_gen_path' pointers path =
  List.map (fun e -> Expr (Why.var e)) pointers
  @ List.map (fun (x, _, f) -> ExprField (Why.var x, f)) path

let rec deep_set_pointers_path pointer_path rem_path expr value =
  let path = make_gen_path pointer_path rem_path in
  deep_set path expr value

let assign_field
    (hat: hat) (path: path) (x: Why.expr) (f: Ident.t) (v: Why.expr) =
  let path =
    fst path,
    List.map (fun (a, b, c) -> Why.var a, b, c) (snd path)
  in
  let info, pointers, rem_path = split_path hat path in
  match info.model with
    | Border ->
        (* TODO: PPGroup ??? *)
        let root = info.region_flattened_name in
        Why.assign
          root
          (deep_set_pointers_path
             pointers (rem_path @ [ x, PPGroup, f ]) (Why.deref root) v)
    | Flatten flat ->
        assert (rem_path = []);
        let fi =
          try
            IMap.find f flat.fields
          with Not_found ->
            assert false
        in
        let root = fi.field_flattened_name in
        Why.assign
          root
          (deep_set_pointers_path
             (pointers @ [ x ]) [] (Why.deref root) v)

let copy_pointer (hat: hat) env (from_path: path) (p: Ident.t) (to_path: path) =
  let to_info, to_pointers, to_rem_path = split_path hat to_path in
  match to_info.model with
    | Border ->
        let fields =
          List.map
            (fun (f, _) -> expr_field hat from_path (Why.var p) f)
            to_info.class_fields
        in
        (* TODO: PPGroup ?? *)
        let owns =
          List.map
            (fun r ->
              let path =
                fst from_path,
                snd from_path @ [ p, PPGroup, r ]
              in
              expr_region hat env path)
            (to_info.class_singles @ to_info.class_groups)
        in
        let tuple = Why.expr_tuple (fields @ owns) in
        let root = to_info.region_flattened_name in
        let path = make_gen_path' to_pointers to_rem_path @ [ Expr (Why.var p) ] in
        Why.assign
          root
          (deep_set path (Why.deref root) tuple)
    | Flatten flat ->
        let fields =
          List.map
            (fun (f, _) ->
              assign_field hat to_path (Why.var p) f
                (expr_field hat from_path (Why.var p) f))
            to_info.class_fields
        in
        (* TODO: owns *)
        let owns =
          List.map
            (fun _ -> assert false (* TODO *))
            (to_info.class_singles @ to_info.class_groups)            
        in
        Why.seq (fields @ owns)

let rec variables_of_hat_without_type (acc: Ident.t list) (hat: hat): Ident.t list =
  IMap.fold
    (fun _ info acc ->
      match info.model with
        | Border ->
            info.region_flattened_name :: acc
        | Flatten flat ->
            let acc =
              IMap.fold
                (fun _ info acc ->
                  info.field_flattened_name :: acc)
                flat.fields
                acc
            in
            variables_of_hat_without_type acc flat.owns)
    hat
    acc

let assign_every_region_to_empty hat pointers =
  let vars = variables_of_hat_without_type [] hat in
  let empty =
    List.fold_left
      (fun acc p ->
        Whylib.expr_set acc p Whylib.expr_empty)
      Whylib.expr_empty
      (List.rev pointers)
  in
  List.map
    (fun var ->
      Why.assign var empty)
    vars

(* ressemble beaucoup a [copy_pointer], pourraient etre factorises *)
let allocate_pointer (hat: hat) (path: path) (p: Why.expr): Why.expr =
  let info, pointers, rem_path = split_path hat path in
  match info.model with
    | Border ->
        let fields =
          List.map
            (fun (f, t) ->
              Ident.create (Ident.name f ^ "_ini"),
              Why.black_box (Why.type_base t))
            info.class_fields
        in
        let owns =
          List.map
            (fun r -> Whylib.expr_empty)
            (info.class_singles @ info.class_groups)
        in
        let field_vars = List.map (fun (id, _) -> Why.var id) fields in
        let tuple = Why.expr_tuple (owns @ field_vars) in
        let root = info.region_flattened_name in
        let gen_path = make_gen_path' pointers rem_path @ [ Expr p ] in
        let assign =
          Why.assign
            root
            (deep_set gen_path (Why.deref root) tuple)
        in
        List.fold_left
          (fun acc (id, value) ->
            Why.expr_let id value acc)
          assign
          fields
    | Flatten flat ->
        (* pas très intéressant d'affecter à une boîte noire... *)
        let fields =
(*          List.map
            (fun (f, t) ->
              assign_field hat path p f
                (Why.black_box (Why.type_base t)))
            info.class_fields*)
          []
        in
        let owns =
          assign_every_region_to_empty flat.owns (List.map Why.var pointers @ [ p ])
        in
        Why.seq (fields @ owns)
