(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Ocamlbuild_plugin

let native = false
let menhir = false
let hat_depth = max_int 

let nativify x =
  if native then x ^ ".native" else x ^ ".byte"

let capucine_main = nativify "main"

let capucine ?o ?(mode: [ `Why2 | `Why3 | `Boron ] = `Why2) file =
  S [
    begin
      if not native then
        S [ A "ocamlrun"; A "-bt" ]
      else
        N
    end;
    A ("./" ^ capucine_main);
    A "-hat"; A (string_of_int hat_depth);
    begin
      match o with
        | None -> N
        | Some o -> S [ A "-o"; P o ]
    end;
    begin
      match mode with
        | `Why2 -> A "-why2"
        | `Why3 -> A "-why3"
        | `Boron -> A "-boron"
    end;
    P file;
  ]

(* --coq-preamble has no effect on my computer :( *)
let coq_preamble = "Require Export Why.
Require Export capucine_why."

let () = dispatch begin function
  | Before_options ->
      if menhir then
        begin
          Options.use_menhir := true;
          tag_any [ "explain" ]
        end
  | After_rules ->
      flag [ "ocamlyacc" ] (A "-v");

      rule "parsini: X.pini -> X_lexer.mll, X_parser.mly, X_ast.ml, X.ml, X.mli"
        ~deps: [
          "%.pini";
        ]
        ~prods: [
          "%_lexer.mll";
          "%_parser.mly";
          "%_ast.ml";
          "%.ml";
          "%.mli";
        ]
        (fun env _ ->
           let pini = env "%.pini" in
           Cmd (S [ A "ocamlrun"; A "-bt";
                    A "parsini"; A "-variant"; A "mono";
                    P pini ]));

      rule ".cat"
        ~dep: "%"
        ~prod: "%.cat"
        (fun env _ ->
          let f = env "%" in
          Cmd (S [ A "cat"; P f ]));

      rule ".test"
        ~deps: [ capucine_main; "%.cap" ]
        ~prod: "%.test"
        (fun env _ ->
          let cap = env "%.cap" in
          Cmd (capucine cap));

      (* la suite est copiee de capucine 1 *)

      rule "capucine: cap -> why"
        ~deps: [ capucine_main; "%.cap" ]
        ~prod: "%.why"
        (fun env _ ->
           let cap = env "%.cap" and why = env "%.why" in
           Cmd (capucine ~o: why ~mode: `Why2 cap));

      rule "capucine: cap -> mlw"
        ~deps: [ capucine_main; "%.cap" ]
        ~prod: "%.mlw"
        (fun env _ ->
           let cap = env "%.cap" and why = env "%.mlw" in
           Cmd (capucine ~o: why ~mode: `Why3 cap));

      rule "gwhy: why -> gwhy"
        ~deps: [
          "lib/capucine.why";
          "%.why";
        ]
        ~prod: "%.gwhy"
        (fun env _ ->
(*           enter_build_dir ();*)
           let why = env "%.why" in
           Cmd (S [ A "gwhy-bin"; (*A "--exp"; A "goal";*)
                    A "--split-user-conj"; (*A "--all-vc";*)
                    P "lib/capucine.why"; P why ]));

      rule "capucine.why -> capucine_why3.why"
        ~dep: "lib/capucine.why"
        ~prod: "lib/capucine_why3.why"
        (fun _ _ -> Cmd (S [ A "why"; A "-why3"; P "lib/capucine.why" ]));

      rule "why3: mlw -> prouveurs"
        ~deps: [
          "lib/capucine_why3.why";
          "%.mlw";
        ]
        ~prod: "%.why3"
        (fun env _ ->
(*           enter_build_dir ();*)
           let mlw = env "%.mlw" in
           Cmd (S [ A "whyml3"; A "-a"; A "split_goal";
                    A "-I"; P "lib"; A "-P"; A "alt-ergo"; P mlw ]));

      rule "why3ide"
        ~deps: [
          "lib/capucine_why3.why";
          "%.mlw";
        ]
        ~prod: "%.why3ide"
        (fun env _ ->
(*           enter_build_dir ();*)
           let mlw = env "%.mlw" in
           Cmd (S [ A "why3ide"; A "-I"; P "lib"; P mlw ]));

      rule "coq: why -> coq"
        ~deps: [
          "lib/capucine_why.v";
          "%.why";
        ]
        ~prod: "%.coq"
        (fun env _ ->
(*           enter_build_dir ();*)
           let why = env "%.why" in
           Cmd (S [ A "why"; A "--coq"; (*A "--all-vc";*)
                    P "lib/capucine.why";
                    A "--coq-preamble"; A coq_preamble;
                    P why ]));

      rule "capucine.why -> capucine_why.v"
        ~dep: "lib/capucine.why"
        ~prod: "lib/capucine_why.v"
        (fun env _ ->
          Cmd (S [ A "why"; A "--coq"; P "lib/capucine.why" ]));

      rule "v -> vo"
        ~dep: "%.v"
        ~prod: "%.vo"
        (fun env _ ->
          let file = env "%.v" in
          Cmd (S [ A "coqc"; P file ]));

      rule "coqide"
        ~deps: [
          "lib/capucine_why.vo";
          "%.coq";
        ]
        ~prod: "%.coqide"
        (fun env _ ->
(*           enter_build_dir ();*)
           let file = env "%_why.v" in
           Cmd (S [ A "coqide"; A "-I"; P "lib"; P file ]))
  | _ -> ()
end
