(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

open Lang_ast
open Format
open Tast
open Types

let pp_type = Types.pp_type
let pp_region = Types.pp_region
let pp_ident = Ident.pp
let pp_env = Env.pp

let pp_unit fmt () = fprintf fmt "()"

let pp_permission fmt perm =
  match perm with
    | PEmpty r -> fprintf fmt "%a^e" pp_region r
    | POpen r -> fprintf fmt "%a^o" pp_region r
    | PClosed r -> fprintf fmt "%a^c" pp_region r
    | PGroup r -> fprintf fmt "%a^g" pp_region r
    | PFocus (s, r) -> fprintf fmt "%a->%a" pp_region s pp_region r

let pp_permissions fmt perms =
  fprintf fmt "@[<hov>";
  begin
    match perms with
      | [] ->
          fprintf fmt "(none)"
      | x :: rem ->
          pp_permission fmt x;
          List.iter (fprintf fmt ",@ %a" pp_permission) rem
  end;
  fprintf fmt "@]"

let pp_available fmt perms =
  fprintf fmt "@[<hv 2>Available permissions:@ %a@]"
    pp_permissions perms

let pp_binop fmt op =
  match op with
    | Add -> fprintf fmt "+"
    | Mul -> fprintf fmt "*"
    | Sub -> fprintf fmt "-"
    | Div -> fprintf fmt "/"
    | Le -> fprintf fmt "<="
    | Lt -> fprintf fmt "<"
    | Ge -> fprintf fmt ">="
    | Gt -> fprintf fmt ">"
    | Eq -> fprintf fmt "="
    | Neq -> fprintf fmt "<>"
    | And -> fprintf fmt "&&"
    | Or -> fprintf fmt "||"

let pp_const fmt c =
  match c with
    | CUnit -> fprintf fmt "()"
    | CTrue -> fprintf fmt "true"
    | CFalse -> fprintf fmt "false"
    | CInt i -> fprintf fmt "%d" i

let rec pp_expr fmt e =
  match e with
    | EConst c -> pp_const fmt c
    | EVar id -> Ident.pp fmt id
    | ESelect (v, f, _) -> fprintf fmt "%a.%a" Ident.pp v Ident.pp f
    | EBinOp (a, op, b) ->
        fprintf fmt "(@[<hv 2>%a@ %a %a@])" pp_expr a pp_binop op pp_expr b
    | EApp (id, r, a) ->
        fprintf fmt "@[<hov 2>%a" Ident.pp id;
        begin
          match r with
            | [] ->
                ()
            | (x, _) :: r ->
                fprintf fmt "@ [@[<hov 2>%a" pp_region x;
                List.iter (fprintf fmt ",@ %a" pp_region) (List.map fst r);
                fprintf fmt "@]]"
        end;
        fprintf fmt "@ (@[<hov 2>";
        begin
          match a with
            | [] ->
                ()
            | x :: a ->
                fprintf fmt "%a" pp_expr x;
                List.iter (fprintf fmt ",@ %a" pp_expr) a
        end;
        fprintf fmt "@])@]"

let pp_class_expr fmt (name, regions, types) =
  fprintf fmt "@[<hv 2>%a" Ident.pp name;
  begin
    match regions with
      | [] -> ()
      | x :: rem ->
          fprintf fmt "[@[<hv 2>%a" pp_region x;
          List.iter (fprintf fmt ",@ %a" pp_region) rem;
          fprintf fmt "@]]"
  end;
  begin
    match types with
      | [] -> ()
      | x :: rem ->
          fprintf fmt "(@[<hv 2>%a" pp_type x;
          List.iter (fprintf fmt ",@ %a" pp_type) rem;
          fprintf fmt "@])"
  end;
  fprintf fmt "@]"

let pp_compare_op fmt o =
  match o with
    | LNeq -> fprintf fmt "<>"
    | LLt -> fprintf fmt "<"
    | LLe -> fprintf fmt "<="
    | LGt -> fprintf fmt ">"
    | LGe -> fprintf fmt ">="
    | LEq -> fprintf fmt "="

let pp_logic_op fmt o =
  match o with
    | LOr -> fprintf fmt "||"
    | LImpl -> fprintf fmt "==>"
    | LIff -> fprintf fmt "<=>"
    | LAnd -> fprintf fmt "&&"

let pp_term_op fmt o =
  match o with
    | LSub -> fprintf fmt "-"
    | LMul -> fprintf fmt "*"
    | LDiv -> fprintf fmt "/"
    | LAdd -> fprintf fmt "+"

let rec pp_term fmt t =
  match t with
    | LConst c ->
        pp_const fmt c
    | LBinOp (a, op, b) ->
        fprintf fmt "(@[<hv 2>%a %a@ %a@])" pp_term a pp_term_op op pp_term b
    | LVar id ->
        Ident.pp fmt id
    | LAt (t, l) ->
        fprintf fmt "@[%a@@%a@]" pp_term t Ident.pp l
    | LGet (r, p, f) ->
        fprintf fmt "@[get(%a, %a, %a)@]" pp_region_term r
          pp_term p Ident.pp f
    | LApp (id, r, a) ->
        fprintf fmt "@[<hov 2>%a" Ident.pp id;
        begin
          match r with
            | [] ->
                ()
            | x :: r ->
                fprintf fmt "@ [@[<hov 2>%a" pp_region_term x;
                List.iter (fprintf fmt ",@ %a" pp_region_term) r;
                fprintf fmt "@]]"
        end;
        fprintf fmt "@ (@[<hov 2>";
        begin
          match a with
            | [] ->
                ()
            | x :: a ->
                fprintf fmt "%a" pp_term x;
                List.iter (fprintf fmt ",@ %a" pp_term) a
        end;
        fprintf fmt "@])@]"

and pp_region_term fmt r =
  match r with
    | LRoot i ->
        Ident.pp fmt i
    | LOwn (s, p, f) ->
        fprintf fmt "@[<hov 2>get(%a,@ %a,@ %a)@]" pp_region_term s
          pp_term p Ident.pp f

let pp_quantifier fmt q =
  match q with
    | Forall -> fprintf fmt "forall"
    | Exists -> fprintf fmt "exists"

let rec pp_predicate fmt p =
  match p with
    | PTrue ->
        fprintf fmt "true"
    | PFalse ->
        fprintf fmt "false"
    | PCompare (a, op, b) ->
        fprintf fmt "(@[<hv 2>%a %a@ %a@])" pp_term a pp_compare_op op pp_term b
    | PLogicOp (a, op, b) ->
        fprintf fmt "(@[<hv 2>%a %a@ %a@])"
          pp_predicate a pp_logic_op op pp_predicate b
    | PIn (t, r) ->
        fprintf fmt "@[(%a in %a)@]" pp_term t pp_region_term r
    | PAt (p, l) ->
        fprintf fmt "@[%a@%a@]" pp_predicate p Ident.pp l
    | PQuantify (q, x, t, p) ->
        fprintf fmt "(@[<hv 2>%a %a: %a.@ %a@])"
          pp_quantifier q Ident.pp x pp_type t pp_predicate p
    | PQuantifyRegion (q, x, ce, p) ->
        fprintf fmt "(@[<hv 2>%a region %a: %a.@ %a@])"
          pp_quantifier q Ident.pp x pp_class_expr ce pp_predicate p
    | PApp (id, r, a) ->
        fprintf fmt "@[<hov 2>%a" Ident.pp id;
        begin
          match r with
            | [] ->
                ()
            | x :: r ->
                fprintf fmt "@ [@[<hov 2>%a" pp_region_term x;
                List.iter (fprintf fmt ",@ %a" pp_region_term) r;
                fprintf fmt "@]]"
        end;
        fprintf fmt "@ (@[<hov 2>";
        begin
          match a with
            | [] ->
                ()
            | x :: a ->
                fprintf fmt "%a" pp_term x;
                List.iter (fprintf fmt ",@ %a" pp_term) a
        end;
        fprintf fmt "@])@]"
    | PNot p ->
        fprintf fmt "(@[<hov 2>not@ %a@])" pp_predicate p

let rec pp_statement fmt s =
  match s with
    | IWeakenSingle r ->
        fprintf fmt "weaken single %a" pp_region r
    | IWeakenEmpty r ->
        fprintf fmt "weaken empty %a" pp_region r
    | IUnpack id ->
        fprintf fmt "unpack %a" Ident.pp id
    | IUnfocus (s, _, r, _, _) ->
        fprintf fmt "unfocus %a as %a" pp_region s pp_region r
    | IPack (id, _, _) ->
        fprintf fmt "pack %a" Ident.pp id
    | INew (id, r, _, _) ->
        fprintf fmt "let %a = new [%a]" Ident.pp id pp_region r
    | ILetRegion (id, ce) ->
        fprintf fmt "let region %a: %a" Ident.pp id pp_class_expr ce
    | ILet (id, e) ->
        fprintf fmt "@[<hv 2>let %a =@ %a@]" Ident.pp id pp_expr e
    | IAffect (v, f, _, e) ->
        fprintf fmt "@[<hv 2>%a.%a <-@ %a@]" Ident.pp v Ident.pp f pp_expr e
    | IFocus (var, _, region, _) ->
        fprintf fmt "focus %a as %a" Ident.pp var pp_region region
    | IAdopt (s, _, r, _, _) ->
        fprintf fmt "adopt %a as %a" pp_region s pp_region r
    | IIf (e, s1, s2) ->
        fprintf fmt "@[<hov>@[<hv 2>if@ %a@]@ then@ @[<hv 2>{@ %a@ }@]@ else@ @[<hv 2>{@ %a@ }@]"
          pp_expr e pp_sequence s1 pp_sequence s2
      (* ICall of
         Ident.t * Ident.t * region list option * expr list
         * region*)
    | ICall (x, f, rl, el, _) ->
        fprintf fmt "@[<hv 2>let %a = %a@ [@[<hov>"
          Ident.pp x Ident.pp f;
        begin
          match rl with
            | [] -> ()
            | (x, _) :: rem ->
                fprintf fmt "%a" pp_region x;
                List.iter (fprintf fmt ",@ %a" pp_region) (List.map fst rem)
        end;
        fprintf fmt "@]]@ (@[<hov>";
        begin
          match el with
            | [] -> ()
            | x :: rem ->
                fprintf fmt "%a" pp_expr x;
                List.iter (fprintf fmt ",@ %a" pp_expr) rem
        end;
        fprintf fmt "@])@]";
    | IAssert p ->
        fprintf fmt "@[<hv 2>assert@ %a@]" pp_predicate p
    | IAssume p ->
        fprintf fmt "@[<hv 2>assume@ %a@]" pp_predicate p
    | ILabel l ->
        fprintf fmt "label %a" Ident.pp l
    | IUseInvariant (x, _, _) ->
        fprintf fmt "use invariant %a" Ident.pp x

and pp_sequence fmt s =
  match s with
    | [] ->
        ()
    | x :: rem ->
        fprintf fmt "%a" pp_statement x;
        List.iter (fprintf fmt ";@ %a" pp_statement) rem

let rec pp_statement_in_decl fmt s =
  fprintf fmt "  %a;@." pp_statement s

let pp_sequence_in_decl fmt s =
  List.iter (pp_statement_in_decl fmt) s

let pp_inferred_decl fmt decl =
  match decl with
    | DClass _ | DType _ | DLogicFun _ | DPredicate _ | DAxiom _ ->
        ()
    | DFun fd ->
        fprintf fmt
          "@.fun %a@.{@.%a  @[<hv 2>return@ %a@]@.}@."
          Ident.pp fd.f_name pp_sequence_in_decl fd.f_body pp_expr fd.f_return

let pp_inferred fmt decls =
  List.iter (pp_inferred_decl fmt) decls;
  fprintf fmt "@."

let pp_operation_producing fmt (consumes, operation, produces) =
  fprintf fmt "@[<hv>@[<hv 2>{%a}@ %a@]@ {%a}@]"
    pp_permissions consumes
    pp_statement operation
    pp_permissions produces
