(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

type typ = Types.UType.t
type region = Types.URegion.t

type binop = Lang_ast.expr_ebinop

type const =
  | CUnit
  | CTrue
  | CFalse
  | CInt of int

type path_perm =
  | PPNone
  | PPEmpty
  | PPSingle
  | PPGroup
  | PPFocus

type 'a path_gen = Ident.t * ('a * path_perm * Ident.t) list
  (* x.r where x: [R] and R^c is encoded as (R, [x, PPSingle, r])
     the list starts with the var of the root region  *)

type path = Ident.t path_gen

type expr =
  | EConst of const
  | EVar of Ident.t
  | ESelect of Ident.t * Ident.t * path
  | EBinOp of expr * binop * expr
  | EApp of Ident.t * (region * path) list * expr list

type compare_op = LNeq | LLt | LLe | LGt | LGe | LEq
type logic_op = LOr | LImpl | LIff | LAnd
type term_op = LSub | LMul | LDiv | LAdd

type term =
  | LConst of const
  | LBinOp of term * term_op * term
  | LVar of Ident.t
  | LAt of term * Ident.t
  | LGet of region_term * term * Ident.t
  | LApp of Ident.t * region_term list * term list

and region_term =
  | LRoot of Ident.t
  | LOwn of region_term * term * Ident.t

type logic_path = term path_gen
type why_path = Why.term path_gen

type quantifier = Forall | Exists

type class_expr = Ident.t * region list * typ list

type predicate =
  | PTrue
  | PFalse
  | PCompare of term * compare_op * term
  | PLogicOp of predicate * logic_op * predicate
  | PIn of term * region_term
  | PAt of predicate * Ident.t
  | PQuantify of quantifier * Ident.t * typ * predicate
  | PQuantifyRegion of quantifier * Ident.t * class_expr * predicate
  | PApp of Ident.t * region_term list * term list
  | PNot of predicate

type instruction =
  | IWeakenSingle of region
  | IWeakenEmpty of region
  | IUnpack of Ident.t (* variable *)
  | IUnfocus of region * path * region * path * Ident.t
      (* singleton region, its path, focused region, its path,
         variable which was focused *)
  | IPack of Ident.t * class_expr * path
      (* variable, class of variable, path to variable *)
  | INew of Ident.t * region * path * class_expr
  | ILetRegion of Ident.t * class_expr
  | ILet of Ident.t * expr
  | IIf of expr * instruction list * instruction list
  | IFocus of Ident.t * path * region * path
  | ICall of Ident.t * Ident.t * (region * path) list * expr list * typ
      (* let ident1 = ident2[regions](expressions) *)
  | IAffect of Ident.t * Ident.t * path * expr
      (* ident1.ident2 <- expr *)
  | IAdopt of region * path * region * path * Ident.t
  | IAssert of predicate
  | ILabel of Ident.t
  | IUseInvariant of Ident.t * path * class_expr
  | IAssume of predicate

type permission =
  | POpen of region
  | PGroup of region
  | PFocus of region * region
  | PEmpty of region
  | PClosed of region

type class_decl = {
  c_name: Ident.t;
  c_region_parameters: (Ident.t * class_expr) list;
  c_type_parameters: Ident.t list;
  c_singles: (Ident.t * class_expr) list;
  c_groups: (Ident.t * class_expr) list;
  c_fields: (Ident.t * typ) list;
  c_invariant: predicate;
}

type fun_decl = {
  f_name: Ident.t;
  f_region_parameters: (Ident.t * class_expr) list;
  f_type_parameters: Ident.t list;
  f_arguments: (Ident.t * typ) list;
  f_return_type: typ;
  f_consumes: permission list;
  f_produces: permission list;
  f_pre: predicate;
  f_post: predicate;
  f_body: instruction list;
  f_return: expr;
}

type type_decl = {
  t_name: Ident.t;
  t_parameters: Ident.t list;
}

type logic_fun_decl = {
  lf_name: Ident.t;
  lf_type_parameters: Ident.t list;
  lf_regions: (Ident.t * class_expr) list;
  lf_parameters: (Ident.t * typ) list;
  lf_return_type: typ;
  lf_body: term option;
}

type predicate_decl = {
  p_name: Ident.t;
  p_type_parameters: Ident.t list;
  p_regions: (Ident.t * class_expr) list;
  p_parameters: (Ident.t * typ) list;
  p_body: predicate option;
}

type axiom_decl = {
  a_name: Ident.t;
  a_lemma: bool;
  a_type_parameters: Ident.t list;
  a_body: predicate;
}

type env = {
  classes: class_decl Env.t;
  functions: fun_decl Env.t;
  types: type_decl Env.t;
  logic_functions: logic_fun_decl Env.t;
  predicates: predicate_decl Env.t;
  regions: class_expr Env.t;
  variables: typ Env.t;
  poly_type_variables: unit Env.t;
  labels: unit Env.t;
}

let empty_env =
  {
    classes = Env.empty;
    functions = Env.empty;
    regions = Env.empty;
    variables = Env.empty;
    poly_type_variables = Env.empty;
    labels = Env.empty;
    types = Env.empty;
    logic_functions = Env.empty;
    predicates = Env.empty;
  }

type decl =
  | DFun of fun_decl
  | DClass of class_decl
  | DType of type_decl
  | DLogicFun of logic_fun_decl
  | DPredicate of predicate_decl
  | DAxiom of axiom_decl

type file = decl list
