(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

module type UNIFIABLE =
sig
  type t
  val unify: t -> t -> t
end

module type UNIFY =
sig
  type t
  type data

  val create: data -> t
  val unify: t -> t -> unit
  val find: t -> data
  val compare: t -> t -> int
    (** This does not compare the associated data, only variables.
        They may become equal after unification. *)
  val reprs: unit -> t list
  val uid: t -> int
end

module Make(X: UNIFIABLE): UNIFY with type data = X.t =
struct
  type t = int
  type data = X.t

  let data_table: (t, data) Hashtbl.t = Hashtbl.create 17
  let union_table: (t, t) Hashtbl.t = Hashtbl.create 17

  (* Invariant: there is no cycle in [union_table], and only repr variables
     have data in [data_table]. *)

  let fresh = let c = ref (-1) in fun () -> incr c; !c

  let uid x = x

  let create data =
    let var = fresh () in
    Hashtbl.add data_table var data;
    var

  let rec repr var =
    try
      let r = repr (Hashtbl.find union_table var) in
      Hashtbl.replace union_table var r;
      r
    with Not_found ->
      var

  let compare a b = (repr a) - (repr b)

  let find var =
    Hashtbl.find data_table (repr var)

  (* Unify maintains the invariant above.
     Moreover, each time you call unify, either it terminates directly,
     or it lowers a measure before calling X.unify.
     The measure is the number of equivalent variable classes.
     So if [X.unify] does not create any variable, the algorithm terminates. *)
  let unify a b =
    let a = repr a in
    let b = repr b in
    if a <> b then
      begin
        let data_a = find a in
        let data_b = find b in
        (* Break invariant. *)
        Hashtbl.add union_table a b;
        Hashtbl.remove data_table a;
        (* Invariant restored. *)
        let unified_data =
          try
            X.unify data_a data_b
          with exn ->
            (* Re-break unification, so that errors are reported correctly *)
            (* Break invariant. *)
            Hashtbl.remove union_table a;
            Hashtbl.add data_table a data_a;
            (* Invariant restored. *)
            raise exn
        in
        Hashtbl.replace data_table b unified_data
      end

  module IntSet = Set.Make (struct type t = int let compare = (-) end)

  let reprs () =
    IntSet.elements
      (Hashtbl.fold
         (fun i _ acc -> IntSet.add i acc)
         data_table
         IntSet.empty)
end
