(**************************************************************************)
(* Copyright (c) 2010, Romain BARDOU                                      *)
(* All rights reserved.                                                   *)
(*                                                                        *)
(* Redistribution and  use in  source and binary  forms, with  or without *)
(* modification, are permitted provided that the following conditions are *)
(* met:                                                                   *)
(*                                                                        *)
(* * Redistributions  of  source code  must  retain  the above  copyright *)
(*   notice, this list of conditions and the following disclaimer.        *)
(* * Redistributions in  binary form  must reproduce the  above copyright *)
(*   notice, this list of conditions  and the following disclaimer in the *)
(*   documentation and/or other materials provided with the distribution. *)
(* * Neither the  name of Capucine nor  the names of its contributors may *)
(*   be used  to endorse or  promote products derived  from this software *)
(*   without specific prior written permission.                           *)
(*                                                                        *)
(* THIS SOFTWARE  IS PROVIDED BY  THE COPYRIGHT HOLDERS  AND CONTRIBUTORS *)
(* "AS  IS" AND  ANY EXPRESS  OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT *)
(* LIMITED TO, THE IMPLIED  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR *)
(* A PARTICULAR PURPOSE  ARE DISCLAIMED. IN NO EVENT  SHALL THE COPYRIGHT *)
(* OWNER OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, *)
(* SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL  DAMAGES (INCLUDING,  BUT  NOT *)
(* LIMITED TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, *)
(* DATA, OR PROFITS; OR BUSINESS  INTERRUPTION) HOWEVER CAUSED AND ON ANY *)
(* THEORY OF  LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY,  OR TORT *)
(* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING  IN ANY WAY OUT OF THE USE *)
(* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.   *)
(**************************************************************************)

let boron = ref false
let use_boron b = boron := b

module ISet = Set.Make(Ident)

(******************************************************************************)
(*                                    AST                                     *)
(******************************************************************************)

type logic_type_normal = {
  lt_args: logic_type list;
  lt_name: Ident.t;
}

and logic_type =
  | LTNormal of logic_type_normal
  | LTTuple of logic_type list
  | LTVar of Ident.t

let logic_type ?(args = []) name =
  LTNormal { lt_args = args; lt_name = name }

let type_variable x =
  LTVar x

let unit = logic_type (Ident.create "unit")
let int = logic_type (Ident.create "int")
let bool = logic_type (Ident.create "bool")
let string = logic_type (Ident.create "string")

let tuple l = LTTuple l

type term =
  | LVoid
  | LInt of string
  | LReal of string
  | LBool of bool
  | LTuple of term list
  | LProj of int * int * term
  | LApp of Ident.t * term list
  | LVar of Ident.t
  | LDeref of Ident.t * Ident.t option
  | LIf of term * term * term
  | LResult
  | LEq of term * term
  | LLet of Ident.t * term * term

  | LAnd of term * term
  | LOr of term * term
  | LXor of term * term
  | LAdd of term * term
  | LSub of term * term
  | LDiv of term * term
  | LMul of term * term
  | LMod of term * term
  | LNot of term
  | LNeg of term
  | LLt of term * term
  | LGt of term * term
  | LLe of term * term
  | LGe of term * term

let at_old = Ident.create ""
let at_init = Ident.create "Init"

let term_void = LVoid
let term_ints x = LInt x
let term_reals x = LReal x
let term_int x = LInt (string_of_int x)
let term_real x = LReal (string_of_float x)
let term_bool x = LBool x
let term_app f args = LApp (f, args)
let term_var x = LVar x
let term_deref ?at x = LDeref (x, at)
let term_if x y z = LIf (x, y, z)
let term_result = LResult
let term_eq x y = LEq (x, y)
let term_let x y z = LLet (x, y, z)

let term_and x y = LAnd (x, y)
let term_or x y = LOr (x, y)
let term_xor x y = LXor (x, y)
let term_add x y = LAdd (x, y)
let term_sub x y = LSub (x, y)
let term_div x y = LDiv (x, y)
let term_mul x y = LMul (x, y)
let term_mod x y = LMod (x, y)
let term_not x = LNot x
let term_neg x = LNeg x
let term_lt x y = LLt (x, y)
let term_gt x y = LGt (x, y)
let term_le x y = LLe (x, y)
let term_ge x y = LGe (x, y)

type assertion =
  | ABool of bool
  | APred of Ident.t * term list
  | AAnd of assertion * assertion
  | AOr of assertion * assertion
  | AIff of assertion * assertion
  | AImplies of assertion * assertion
  | ANot of assertion
  | AIf of term * assertion * assertion
  | ALet of Ident.t * term * assertion
  | AForall of Ident.t * logic_type * assertion
  | AExists of Ident.t * logic_type * assertion
  | AEq of term * term
  | ANeq of term * term
  | AGt of term * term
  | ALt of term * term
  | AGe of term * term
  | ALe of term * term

let a_bool b = ABool b
let a_true = a_bool true
let a_false = a_bool false
let a_app x args = APred (x, args)

let a_and x y =
  match x, y with
    | ABool true, z
    | z, ABool true -> z
    | _ -> AAnd (x, y)

let a_or x y = AOr (x, y)
let a_iff x y = AIff (x, y)
let a_implies x y = AImplies (x, y)
let a_not x = ANot x
let a_if x y z = AIf (x, y, z)
let a_let x y z = ALet (x, y, z)
let a_forall x t y = AForall (x, t, y)
let a_exists x t y = AExists (x, t, y)
let a_eq x y = AEq (x, y)
let a_neq x y = ANeq (x, y)
let a_gt x y = AGt (x, y)
let a_lt x y = ALt (x, y)
let a_ge x y = AGe (x, y)
let a_le x y = ALe (x, y)

type typ =
  | TBase of logic_type
  | TRef of typ
  | TFun of Ident.t * typ * typ
  | TAnnot of assertion * Ident.t list * Ident.t list * assertion *
      (Ident.t * assertion) list * typ

let type_base lt = TBase lt
let type_ref t = TRef t
let type_fun ~arg_name ~arg_type t = TFun (arg_name, arg_type, t)
let type_annot ?(pre = a_true) ?(reads = []) ?(writes = []) ?(post = a_true)
    ?(exceptions = []) t =
  TAnnot (pre, reads, writes, post, exceptions, t)

type expr =
  | EVoid
  | EInt of string
  | EReal of string
  | EBool of bool
  | EVar of Ident.t
  | EAnd of expr * expr
  | EOr of expr * expr
  | EXor of expr * expr
  | ENot of expr
  | ENeg of expr
  | EAdd of expr * expr
  | ESub of expr * expr
  | EMul of expr * expr
  | EDiv of expr * expr
  | EMod of expr * expr
  | ELt of expr * expr
  | EGt of expr * expr
  | ELe of expr * expr
  | EGe of expr * expr
  | EEq of expr * expr
  | EDeref of Ident.t
  | EIf of expr * expr * expr
  | ELoop of expr * assertion * (term * Ident.t option) option * expr list
  | ESeq of expr list
  | EApp of expr * expr
  | ERaise of Ident.t * expr option
  | ETry of expr * Ident.t * Ident.t option * expr
  | EFun of (Ident.t * typ) list * assertion * assertion *
      (Ident.t * assertion) list * expr
  | EAssert of assertion
  | ELabel of Ident.t * expr
  | EBlackBox of typ
  | EAbsurd
  | ELet of Ident.t * expr * expr
  | EAssign of Ident.t * expr
  | ETuple of expr list
  | EProj of int * int * expr
  | ERef of expr

let rec may_have_side_effects = function
  | EVoid | EInt _ | EReal _ | EBool _ | EVar _ | EDeref _ | ERef _ ->
      false
  | EAnd (a, b)
  | EOr (a, b)
  | EXor (a, b)
  | EAdd (a, b)
  | ESub (a, b)
  | EDiv (a, b)
  | EMul (a, b)
  | EMod (a, b)
  | ELt (a, b)
  | EGt (a, b)
  | ELe (a, b)
  | EGe (a, b)
  | EEq (a, b)
  | ETry (a, _, _, b)
  | ELet (_, a, b) ->
      may_have_side_effects a || may_have_side_effects b
  | ENot a
  | ENeg a ->
      may_have_side_effects a
  | EIf (a, b, c) ->
      may_have_side_effects a
      || may_have_side_effects b
      || may_have_side_effects c
  | ELoop (a, _, _, l) ->
      may_have_side_effects a
      || (List.exists may_have_side_effects l)
  | ESeq l ->
      List.exists may_have_side_effects l
  | ETuple l -> List.exists may_have_side_effects l
  | EProj (_, _, e) -> may_have_side_effects e
  | EApp _ | ERaise _ | EFun _ | EAssert _ | ELabel _ | EBlackBox _ | EAbsurd
  | EAssign _ ->
      true

(*
let rec expr_of_term = function
  | LVoid ->
      EVoid
  | LInt s ->
      EInt s
  | LReal s ->
      EReal s
  | LBool b ->
      EBool b
  | LApp (id, l) ->
      List.fold_left
        (fun acc x -> EApp (acc, expr_of_term x))
        (EVar id)
        l
  | LVar id
  | LDeref (id, None) ->
      EVar id
  | LIf (a, b, c) ->
      EIf (expr_of_term a, expr_of_term b, expr_of_term c)
  | LDeref (_, Some _) | LResult ->
      raise (Invalid_argument "Why.expr_of_term")
  | LEq (a, b) ->
      EEq (expr_of_term a, expr_of_term b)
  | LLet (x, a, b) ->
      ELet (x, expr_of_term a, expr_of_term b)
  | LNeg a ->
      ENeg (expr_of_term a)
  | LNot a ->
      ENot (expr_of_term a)
  | LMod (a, b) ->
      EMod (expr_of_term a, expr_of_term b)
  | LMul (a, b) ->
      EMul (expr_of_term a, expr_of_term b)
  | LDiv (a, b) ->
      EDiv (expr_of_term a, expr_of_term b)
  | LSub (a, b) ->
      ESub (expr_of_term a, expr_of_term b)
  | LAdd (a, b) ->
      EAdd (expr_of_term a, expr_of_term b)
  | LXor (a, b) ->
      EXor (expr_of_term a, expr_of_term b)
  | LOr (a, b) ->
      EOr (expr_of_term a, expr_of_term b)
  | LAnd (a, b) ->
      EAnd (expr_of_term a, expr_of_term b)
  | LLt (a, b) ->
      ELt (expr_of_term a, expr_of_term b)
  | LGt (a, b) ->
      EGt (expr_of_term a, expr_of_term b)
  | LLe (a, b) ->
      ELe (expr_of_term a, expr_of_term b)
  | LGe (a, b) ->
      EGe (expr_of_term a, expr_of_term b)
*)

let const_void = EVoid
let const_ints x = EInt x
let const_int x = EInt (string_of_int x)
let const_reals x = EReal x
let const_real x = EReal (string_of_float x)
let const_bool x = EBool x
let var x = EVar x
let expr_and x y = EAnd (x, y)
let expr_or x y = EOr (x, y)
let expr_xor x y = EXor (x, y)
let expr_add x y = EAdd (x, y)
let expr_sub x y = ESub (x, y)
let expr_div x y = EDiv (x, y)
let expr_mul x y = EMul (x, y)
let expr_mod x y = EMod (x, y)
let expr_not x = ENot x
let expr_neg x = ENeg x
let expr_lt x y = ELt (x, y)
let expr_gt x y = EGt (x, y)
let expr_le x y = ELe (x, y)
let expr_ge x y = EGe (x, y)
let expr_eq x y = EEq (x, y)
let deref x = EDeref x
let expr_if x y z = EIf (x, y, z)

let loop x ?(invariant = a_true) ?variant y =
  let y = match y with
    | ESeq l -> l
    | _ -> [ y ]
  in
  ELoop (x, invariant, variant, y)

let rec seq = function
  | [] ->
      const_void
  | [ x ] ->
      x
  | EVoid :: r ->
      seq r
  | ESeq l :: r ->
      seq (l @ r)
  | x :: r ->
      begin match seq r with
        | EVoid ->
            x
        | ESeq l ->
            ESeq (x :: l)
        | y ->
            ESeq [ x; y ]
      end

let seq2 x y = seq [ x; y ]
let app x y = EApp (x, y)
let app_list x l = List.fold_left app x l
let expr_raise ?arg x = ERaise (x, arg)
let expr_try x y ?arg z = ETry (x, y, arg, z)
let expr_fun ?(args = []) ?(pre = a_true) ?(post = a_true) ?(exceptions = [])
    x =
  EFun (args, pre, post, exceptions, x)
let expr_assert x = EAssert x
let label x y = ELabel (x, y)
let black_box x = EBlackBox x
let absurd = EAbsurd
let expr_let id x y = ELet (id, x, y)
let assign id x = EAssign (id, x)

let expr_ignore =
  let underscore = Ident.create "_" in
  fun a b ->
    if may_have_side_effects a then
      expr_let underscore a b
    else
      b

let term_tuple l = LTuple l
let term_proj n i t = LProj (n, i, t)

let expr_tuple l = ETuple l
let expr_proj n i e = EProj (n, i, e)

let expr_ref e = ERef e

let rec term_reads acc t =
  match t with
    | LVoid
    | LInt _
    | LReal _
    | LBool _
    | LVar _
    | LResult
      -> acc

    | LProj (_, _, a)
    | LNot a
    | LNeg a
      -> term_reads acc a

    | LEq (a, b)
    | LLet (_, a, b)
    | LAnd (a, b)
    | LOr (a, b)
    | LXor (a, b)
    | LAdd (a, b)
    | LSub (a, b)
    | LDiv (a, b)
    | LMul (a, b)
    | LMod (a, b)
    | LLt (a, b)
    | LGt (a, b)
    | LLe (a, b)
    | LGe (a, b)
      -> term_reads (term_reads acc a) b

    | LIf (a, b, c)
      -> term_reads (term_reads (term_reads acc a) b) c

    | LTuple l
    | LApp (_, l)
      -> List.fold_left term_reads acc l

    | LDeref (x, _)
      -> ISet.add x acc

let rec assertion_reads acc a =
  match a with
    | ABool _
      -> acc
    | AAnd (a, b)
    | AOr (a, b)
    | AIff (a, b)
    | AImplies (a, b)
      -> assertion_reads (assertion_reads acc a) b
    | AIf (t, a, b)
      -> assertion_reads (assertion_reads (term_reads acc t) a) b
    | ALet (_, t, a)
      -> assertion_reads (term_reads acc t) a
    | ANot a
    | AForall (_, _, a)
    | AExists (_, _, a)
      -> assertion_reads acc a
    | AEq (ta, tb)
    | ANeq (ta, tb)
    | AGt (ta, tb)
    | ALt (ta, tb)
    | AGe (ta, tb)
    | ALe (ta, tb)
      -> term_reads (term_reads acc ta) tb
    | APred (n, tl)
      -> List.fold_left term_reads acc tl

let assume = function
  | ABool true ->
      const_void
  | a ->
      let reads = assertion_reads ISet.empty a in
      let reads = ISet.elements reads in
      black_box (type_annot ~reads ~post: a (type_base unit))

type decl =
  | DLogicType of Ident.t list * Ident.t
  | DLogic of Ident.t * logic_type list * logic_type
  | DLogicDef of Ident.t * (Ident.t * logic_type) list * logic_type * term
  | DParam of Ident.t * typ
  | DLet of Ident.t * expr
  | DPredicate of Ident.t * (Ident.t * logic_type) list * assertion option
  | DAxiom of bool * Ident.t * assertion

let decl_logic_type ?(args = []) name = DLogicType (args, name)
let decl_logic name ?(args = []) t = DLogic (name, args, t)
let decl_logic_def name ?(args = []) ty t = DLogicDef (name, args, ty, t)
let decl_parameter name t = DParam (name, t)
let decl_let name e = DLet (name, e)
let decl_predicate name args body = DPredicate (name, args, body)
let decl_axiom ~is_lemma name body = DAxiom (is_lemma, name, body)

type file = decl list

let file l = l

(******************************************************************************)
(*                              Pretty-Printing                               *)
(******************************************************************************)

open Format

module Pp =
struct
  let nothing fmt () = ()
  let comma_space fmt () = fprintf fmt ",@ "
  let string s fmt () = fprintf fmt "%s" s
  let space fmt () = fprintf fmt "@ "

  let list ?(sep = nothing) ?(left = nothing) ?(right = nothing) f fmt l =
    match l with
      | [] -> ()
      | [x] ->
          left fmt ();
          f fmt x;
          right fmt ();
      | x :: rem ->
          left fmt ();
          f fmt x;
          List.iter
            (fun x ->
              sep fmt ();
              f fmt x)
            rem;
          right fmt ()

  let option ?(left = nothing) ?(right = nothing) f fmt = function
    | None -> ()
    | Some x ->
        left fmt ();
        f fmt x;
        right fmt ()

  let capitalized_identifier fmt i =
  (*  if !print_unique_identifiers then
      fprintf fmt "%s" (String.capitalize (Ident.uname i))
    else*)
      fprintf fmt "%s" (String.capitalize (Ident.name i))
end

open Pp

(******************************************************************************)

(* Identifiers should not be keywords. *)
let is_keyword = function
  | "array" -> true
  | _ -> false

let identifier fmt x =
  if is_keyword (Ident.name x) then
    fprintf fmt "_%a" Ident.pp x
  else
    Ident.pp fmt x

let prime_identifier fmt id =
  fprintf fmt "'%a" identifier id

module Why2 = struct
  let couple =
    let id_couple = Ident.create "couple" in
    fun a b -> logic_type ~args: [ a; b ] id_couple

  let rec tuple_of couple a i j =
    let c = j - i in
    if c <= 1 then
      a.(i)
    else
      let m = i + c / 2 in
      couple
        (tuple_of couple a i m)
        (tuple_of couple a m j)

  let make_tuple unit couple = function
    | [] ->
        unit
    | l ->
        let a = Array.of_list l in
        let c = Array.length a in
        tuple_of couple a 0 c

  let rec make_proj fst snd n i x =
    if n <= 1 then x else
      let m = n / 2 in
      if i < m then
        make_proj fst snd m i (fst x)
      else
        make_proj fst snd (n - m) (i - m) (snd x)

  let pair, expr_pair =
    let id = Ident.create "pair" in
    (fun a b -> term_app id [ a; b ]),
    (fun a b -> app (app (var id) a) b)

  let term_fst, expr_fst =
    let id = Ident.create "fst" in
    (fun a -> term_app id [ a ]),
    (fun a -> app (var id) a)

  let term_snd, expr_snd =
    let id = Ident.create "snd" in
    (fun a -> term_app id [ a ]),
    (fun a -> app (var id) a)

(*  let proj = make_proj fst snd*)

  let rec pp_logic_type fmt t =
    let o x = fprintf fmt x in
    match t with
      | LTNormal t ->
          begin match t.lt_args with
            | [] ->
                o "%a" identifier t.lt_name
            | [a] ->
                o "@[<hov 2>%a@ %a@]" pp_logic_type a identifier t.lt_name
            | l ->
                o "@[<hov 2>(%a)@ %a@]" (list ~sep: comma_space pp_logic_type)
                  l identifier t.lt_name
          end
      | LTTuple l ->
          pp_logic_type fmt (make_tuple unit couple l)
      | LTVar t ->
          o "'%a" identifier t

  let rec pp_term fmt t =
    let o x = fprintf fmt x in
    match t with
      | LVoid ->
          o "void"
      | LInt s
      | LReal s ->
          if s <> "" && s.[0] >= '0' && s.[0] <= '9' then
            o "%s" s
          else
            o "(%s)" s
      | LBool b ->
          o "%b" b
      | LTuple l ->
          pp_term fmt (make_tuple term_void pair l)
      | LProj (n, i, t) ->
          pp_term fmt (make_proj term_fst term_snd n i t)
      | LApp (f, args) ->
          o "@[<hv 2>%a%a@]"
            identifier f
            (list
               ~left: (fun fmt () -> fprintf fmt "(@,")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt ")")
               pp_term) args
      | LVar x ->
          o "%a" identifier x
      | LDeref (x, None) ->
          o "%a" identifier x
      | LDeref (x, Some at) ->
          if Ident.name at = "" then (* hack for old *)
            o "%a@@" identifier x
          else
            o "%a@@%a" identifier x identifier at
      | LIf (a, b, c) ->
          o "(@[<hv>@[<hv 2>if %a then@ %a@]@ @[<hv 2>else@ %a@]@])"
            pp_term a pp_term b pp_term c
      | LResult ->
          o "result"
      | LEq (a, b) ->
          o "(@[<hv 2>eq(%a,@ %a)@])" pp_term a pp_term b
      | LLet (x, a, b) ->
          o "(@[<hv>@[<hv>@[<hv 2>let %a =@ %a@]@ in@]@ %a@])"
            identifier x pp_term a pp_term b

      | LAnd (x, y) ->
          o "(@[%a &&@ %a@])" pp_term x pp_term y
      | LOr (x, y) ->
          o "(@[%a ||@ %a@])" pp_term x pp_term y
      | LXor (x, y) ->
          o "(@[%a xor@ %a@])" pp_term x pp_term y
      | LAdd (x, y) ->
          o "(@[%a +@ %a@])" pp_term x pp_term y
      | LSub (x, y) ->
          o "(@[%a -@ %a@])" pp_term x pp_term y
      | LDiv (x, y) ->
          if !boron then
            o "(@[computer_div(%a,@ %a)@])" pp_term x pp_term y
          else
            o "(@[%a /@ %a@])" pp_term x pp_term y
      | LMul (x, y) ->
          o "(@[%a *@ %a@])" pp_term x pp_term y
      | LMod (x, y) ->
          o "(@[%a mod@ %a@])" pp_term x pp_term y
      | LNot x ->
          o "(@[not %a@])" pp_term x
      | LNeg x ->
          o "(@[- %a@])" pp_term x
      | LLt (x, y) ->
          o "(@[%a <@ %a@])" pp_term x pp_term y
      | LGt (x, y) ->
          o "(@[%a >@ %a@])" pp_term x pp_term y
      | LLe (x, y) ->
          o "(@[%a <=@ %a@])" pp_term x pp_term y
      | LGe (x, y) ->
          o "(@[%a >=@ %a@])" pp_term x pp_term y

  let rec pp_assertion fmt a =
    let o x = fprintf fmt x in
    match a with
      | ABool b ->
          o "%b" b
      | APred (p, args) ->
          o "@[<hv 2>%a%a@]"
            identifier p
            (list
               ~left: (fun fmt () -> fprintf fmt "(@,")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt ")")
               pp_term) args
      | AAnd (x, y) ->
          o "@[(%a and@ %a)@]" pp_assertion x pp_assertion y
      | AOr (x, y) ->
          o "@[(%a or@ %a)@]" pp_assertion x pp_assertion y
      | AIff (x, y) ->
          o "@[(%a <->@ %a)@]" pp_assertion x pp_assertion y
      | AImplies (x, y) ->
          o "@[(%a ->@ %a)@]" pp_assertion x pp_assertion y
      | ANot x ->
          o "@[(not %a)@]" pp_assertion x
      | AIf (a, b, c) ->
          o "(@[<hv>@[<hv 2>if %a then@ %a@]@ @[<hv 2>else@ %a@]@])"
            pp_term a pp_assertion b pp_assertion c
      | ALet (x, y, z) ->
          o "(@[<hv>@[<hv>@[<hv 2>let %a =@ %a@]@ in@]@ %a@])"
            identifier x pp_term y pp_assertion z
      | AForall (x, t, a) ->
          o "(@[<hv>forall %a: %a.@ %a@])"
            identifier x pp_logic_type t pp_assertion a
      | AExists (x, t, a) ->
          o "(@[<hv>exists %a: %a.@ %a@])"
            identifier x pp_logic_type t pp_assertion a
      | AEq (x, y) ->
          o "(@[<hv>%a =@ %a@])" pp_term x pp_term y
      | ANeq (x, y) ->
          o "(@[<hv>%a <>@ %a@])" pp_term x pp_term y
      | AGt (x, y) ->
          o "(@[<hv>%a >@ %a@])" pp_term x pp_term y
      | ALt (x, y) ->
          o "(@[<hv>%a <@ %a@])" pp_term x pp_term y
      | AGe (x, y) ->
          o "(@[<hv>%a >=@ %a@])" pp_term x pp_term y
      | ALe (x, y) ->
          o "(@[<hv>%a <=@ %a@])" pp_term x pp_term y

  let rec pp_type fmt t =
    let o x = fprintf fmt x in
    match t with
      | TBase lt ->
          o "%a" pp_logic_type lt
      | TRef t ->
          o "@[<hov 2>%a@ ref@]" pp_type t
      | TFun (id, pt, rt) ->
          o "@[<hov 2>%a:@ %a ->@ %a@]" identifier id pp_type pt pp_type rt
      | TAnnot (pre, reads, writes, post, exceptions, t) ->
          o "@[<hv>{%a}@ %a%a%a%a@ @[<hv>{%a%a}@]@]"
            pp_assertion pre
            pp_type t
            (list
               ~left: (fun fmt () -> fprintf fmt "@ @[<hov 2>reads@ ")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt "@]")
               identifier) reads
            (list
               ~left: (fun fmt () -> fprintf fmt "@ @[<hov 2>writes@ ")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt "@]")
               identifier) writes
            (list
               ~left: (fun fmt () -> fprintf fmt "@ @[<hov 2>raises@ ")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt "@]")
               (fun fmt (e, _) -> fprintf fmt "%a" identifier e)) exceptions
            pp_assertion post
            (list
               ~left: (fun fmt () -> fprintf fmt "@ | ")
               ~sep: (fun fmt () -> fprintf fmt "@ | ")
               (fun fmt (e, a) ->
                  fprintf fmt "@[<hov 2>%a =>@ %a@]"
                    identifier e pp_assertion a)) exceptions

  let rec pp_expr fmt e =
    let o x = fprintf fmt x in
    match e with
      | EVoid ->
          o "void"
      | EInt s
      | EReal s ->
          if s <> "" && s.[0] >= '0' && s.[0] <= '9' then
            o "%s" s
          else
            o "(%s)" s
      | EBool b ->
          o "%b" b
      | EVar x ->
          o "%a" identifier x
      | EAnd (x, y) ->
          o "(@[%a &&@ %a@])" pp_expr x pp_expr y
      | EOr (x, y) ->
          o "(@[%a ||@ %a@])" pp_expr x pp_expr y
      | EXor (x, y) ->
          o "(@[%a xor@ %a@])" pp_expr x pp_expr y
      | EAdd (x, y) ->
          o "(@[%a +@ %a@])" pp_expr x pp_expr y
      | ESub (x, y) ->
          o "(@[%a -@ %a@])" pp_expr x pp_expr y
      | EDiv (x, y) ->
          if !boron then
            o "(@[computer_div@ (%a)@ (%a)@])" pp_expr x pp_expr y
          else
            o "(@[%a /@ %a@])" pp_expr x pp_expr y
      | EMul (x, y) ->
          o "(@[%a *@ %a@])" pp_expr x pp_expr y
      | EMod (x, y) ->
          o "(@[%a mod@ %a@])" pp_expr x pp_expr y
      | ENot x ->
          o "(@[not %a@])" pp_expr x
      | ENeg x ->
          o "(@[- %a@])" pp_expr x
      | EDeref x ->
          o "!%a" identifier x
      | EIf (a, b, c) ->
          o "(@[<hv>@[<hv 2>if %a then@ %a@]@ @[<hv 2>else@ %a@]@])"
            pp_expr a pp_expr b pp_expr c
      | ELoop (c, i, v, b) ->
          o "@[<hv>@[<hv 2>while %a do@ { invariant %a%a }@ %a@]@ done@]"
            pp_expr c
            pp_assertion i
            (option
               (fun fmt (t, o) ->
                  fprintf fmt "variant %a%a"
                    pp_term t
                    (option ~left: (string " for ") identifier) o)) v
            (list ~sep: (fun fmt () -> fprintf fmt ";@ ") pp_expr) b
      | ESeq l ->
          o "(@[<hv>%a@])"
            (list ~sep: (fun fmt () -> fprintf fmt ";@ ") pp_expr) l
      | EApp (x, y) ->
          o "(@[%a@ %a@])" pp_expr x pp_expr y
      | ERaise (x, None) ->
          o "(@[raise %a@])" identifier x
      | ERaise (x, Some y) ->
          o "(@[raise (%a %a)@])" identifier x pp_expr y
      | ETry (e, x, a, f) ->
          o "@[<hv>@[<hv 2>try@ %a@]@ @[<hv 2>with %a%a ->@ %a@]@ end@]"
            pp_expr e identifier x
            (option ~left: (string " ") identifier) a
            pp_expr f
      | EFun (args, pre, post, exns, body) ->
          o "@[<hv 2>fun %a ->@ {%a}@ %a@ {%a%a}@]"
            (list
               ~sep: (fun fmt () -> fprintf fmt "@ ")
               (fun fmt (e, t) ->
                  fprintf fmt "(%a: %a)" identifier e pp_type t)) args
            pp_assertion pre
            pp_expr body
            pp_assertion post
            (list
               ~left: (fun fmt () -> fprintf fmt "@ | ")
               ~sep: (fun fmt () -> fprintf fmt "@ | ")
               (fun fmt (e, a) ->
                  fprintf fmt "@[<hov 2>%a =>@ %a@]"
                    identifier e pp_assertion a)) exns
      | EAssert a ->
          o "(@[<hv 2>assert@ { %a };@ void@])" pp_assertion a
      | ELabel (x, e) ->
          o "(@[%a:@ %a@])" identifier x pp_expr e
      | EBlackBox (TRef t) ->
          o "@[ref [ %a ]@]" pp_type t
      | EBlackBox t ->
          o "@[[ %a ]@]" pp_type t
      | EAbsurd ->
          o "absurd"
      | ELet (x, y, z) ->
          o "(@[<hv>@[<hv>@[<hv 2>let %a =@ %a@]@ in@]@ %a@])"
            identifier x pp_expr y pp_expr z
      | EAssign (id, x) ->
          o "(@[<hv 2>%a :=@ %a@])"
            identifier id pp_expr x
      | ELt (x, y) ->
          o "(@[%a <@ %a@])" pp_expr x pp_expr y
      | EGt (x, y) ->
          o "(@[%a >@ %a@])" pp_expr x pp_expr y
      | ELe (x, y) ->
          o "(@[%a <=@ %a@])" pp_expr x pp_expr y
      | EGe (x, y) ->
          o "(@[%a >=@ %a@])" pp_expr x pp_expr y
      | EEq (x, y) ->
          o "(@[[{} bool {if result then %a =@ %a@ else not %a =@ %a}]@])"
            pp_expr x pp_expr y pp_expr x pp_expr y
      | ETuple l ->
          pp_expr fmt (make_tuple const_void expr_pair l)
      | EProj (n, i, e) ->
          pp_expr fmt (make_proj expr_fst expr_snd n i e)
      | ERef e ->
          o "@[ref@ %a@]" pp_expr e

  let pp_decl fmt d =
    let o x = fprintf fmt x in
    match d with
      | DLogicType ([], name) ->
          o "@[<hov 2>type@ %a@]@.@." identifier name
      | DLogicType ([a], name) ->
          o "@[<hov 2>type@ %a %a@]@.@." prime_identifier a identifier name
      | DLogicType (l, name) ->
          o "@[<hov 2>type@ (%a) %a@]@.@."
            (list ~sep: comma_space prime_identifier) l
            identifier name
      | DLogic (name, args, res) ->
          o "@[<hov 2>logic %a:@ %a ->@ %a@]@.@."
            identifier name
            (list ~sep: comma_space pp_logic_type) args
            pp_logic_type res
      | DLogicDef (name, args, res, body) ->
          o "@[<hov 2>function %a(%a): %a =@ %a@]@.@."
            identifier name
            (list ~sep: comma_space
               (fun fmt (id, ty) ->
                 fprintf fmt "%a: %a" identifier id pp_logic_type ty))
            args
            pp_logic_type res
            pp_term body
      | DParam (name, t) ->
          o "@[<hov 2>parameter %a:@ %a@]@.@." identifier name pp_type t
      | DLet (name, e) ->
          o "@[<hv 2>let %a =@ %a@]@.@." identifier name pp_expr e
      | DPredicate (name, args, Some body) ->
          o "@[<hv 2>predicate %a(%a) =@ %a@]@.@."
            identifier name
            (list
               ~sep: comma_space
               (fun fmt (id, ty) ->
                  fprintf fmt "%a: %a" identifier id pp_logic_type ty)) args
            pp_assertion body
      | DPredicate (name, args, None) ->
          o "@[<hv 2>logic %a:@ %a ->@ prop@]@.@."
            identifier name
            (list
               ~sep: comma_space
               (fun fmt (_, ty) ->
                  fprintf fmt "%a" pp_logic_type ty)) args
      | DAxiom (is_lemma, name, body) ->
	  if is_lemma then
	    begin
              o "@[<hv 2>goal %a_as_lemma:@ %a@]@.@."
                identifier name pp_assertion body
	    end;
          o "@[<hv 2>axiom %a:@ %a@]@.@." 
	    identifier name pp_assertion body

  let to_channel ch file =
    let fmt = formatter_of_out_channel ch in
    fprintf fmt "include \"bool.why\"@.@.";
    if !boron then fprintf fmt "include \"divisions.why\"@.@.";
    List.iter (pp_decl fmt) file
end

module Why3 = struct
  let rec pp_logic_type fmt t =
    let o x = fprintf fmt x in
    match t with
      | LTTuple [x] ->
          pp_logic_type fmt x
      | LTTuple l ->
          o "@[<hov 2>(%a)@]"
            (list ~sep: comma_space pp_logic_type) l
      | LTNormal t ->
          begin match t.lt_args with
            | [] ->
                o "%a" identifier t.lt_name
            | l ->
                o "@[<hov 2>%a@ (%a)@]"
                  identifier t.lt_name
                  (list ~sep: space pp_logic_type) l
          end
      | LTVar t ->
          o "'%a" identifier t

  let rec pp_term fmt t =
    let o x = fprintf fmt x in
    match t with
      | LVoid ->
          o "()"
      | LInt s
      | LReal s ->
          if s <> "" && s.[0] >= '0' && s.[0] <= '9' then
            o "%s" s
          else
            o "(%s)" s
      | LBool b ->
          o "%s" (if b then "True" else "False")
      | LTuple l ->
          o "(%a)" (list ~sep: comma_space pp_term) l
      | LProj (n, i, t) ->
          o "@[<hv 2>(let (";
          for j = 1 to i do o "_," done;
          o "x";
          for j = i+1 to n-1 do o ",_" done;
          o ")@ = %a in x)@]" pp_term t
      | LApp (f, args) ->
          o "@[<hv 2>(%a%a)@]"
            identifier f
            (list
               ~left: space
               ~sep: space
               pp_term) args
      | LVar x ->
          o "%a" identifier x
      | LDeref (x, None) ->
          o "!%a" identifier x
      | LDeref (x, Some at) ->
          if Ident.name at = "" then (* hack for old *)
            o "@[<hv 2>(old@ !%a)@]" identifier x
          else
            o "@[<hv 2>(at@ !%a@ %a)@]" identifier x capitalized_identifier at
      | LIf (a, b, c) ->
          o "(@[<hv>@[<hv 2>if %a = True then@ %a@]@ @[<hv 2>else@ %a@]@])"
            pp_term a pp_term b pp_term c
      | LResult ->
          o "result"
      | LEq (a, b) ->
          o "(@[<hv 2>eq@ %a@ %a@])" pp_term a pp_term b
      | LLet (x, a, b) ->
          o "(@[<hv>@[<hv>@[<hv 2>let %a =@ %a@]@ in@]@ %a@])"
            identifier x pp_term a pp_term b

      | LAnd (x, y) ->
          o "(@[%a &&@ %a@])" pp_term x pp_term y
      | LOr (x, y) ->
          o "(@[%a ||@ %a@])" pp_term x pp_term y
      | LXor (x, y) ->
          o "(@[%a xor@ %a@])" pp_term x pp_term y
      | LAdd (x, y) ->
          o "(@[%a +@ %a@])" pp_term x pp_term y
      | LSub (x, y) ->
          o "(@[%a -@ %a@])" pp_term x pp_term y
      | LDiv (x, y) ->
          o "(@[ComputerDivision.div@ %a@ %a@])" pp_term x pp_term y
      | LMul (x, y) ->
          o "(@[%a *@ %a@])" pp_term x pp_term y
      | LMod (x, y) ->
          o "(@[%a mod@ %a@])" pp_term x pp_term y
      | LNot x ->
          o "(@[not %a@])" pp_term x
      | LNeg x ->
          o "(@[- %a@])" pp_term x
      | LLt (x, y) ->
          o "(@[%a <@ %a@])" pp_term x pp_term y
      | LGt (x, y) ->
          o "(@[%a >@ %a@])" pp_term x pp_term y
      | LLe (x, y) ->
          o "(@[%a <=@ %a@])" pp_term x pp_term y
      | LGe (x, y) ->
          o "(@[%a >=@ %a@])" pp_term x pp_term y

  let rec pp_assertion fmt a =
    let o x = fprintf fmt x in
    match a with
      | ABool b ->
          o "%b" b
      | APred (p, args) ->
          o "@[<hv 2>(%a%a)@]"
            identifier p
            (list
               ~left: space
               ~sep: space
               pp_term) args
      | AAnd (x, y) ->
          o "@[(%a and@ %a)@]" pp_assertion x pp_assertion y
      | AOr (x, y) ->
          o "@[(%a or@ %a)@]" pp_assertion x pp_assertion y
      | AIff (x, y) ->
          o "@[(%a <->@ %a)@]" pp_assertion x pp_assertion y
      | AImplies (x, y) ->
          o "@[(%a ->@ %a)@]" pp_assertion x pp_assertion y
      | ANot x ->
          o "@[(not %a)@]" pp_assertion x
      | AIf (a, b, c) ->
          o "(@[<hv>@[<hv 2>if %a = True then@ %a@]@ @[<hv 2>else@ %a@]@])"
            pp_term a pp_assertion b pp_assertion c
      | ALet (x, y, z) ->
          o "(@[<hv>@[<hv>@[<hv 2>let %a =@ %a@]@ in@]@ %a@])"
            identifier x pp_term y pp_assertion z
      | AForall (x, t, a) ->
          o "(@[<hv>forall %a: %a.@ %a@])"
            identifier x pp_logic_type t pp_assertion a
      | AExists (x, t, a) ->
          o "(@[<hv>exists %a: %a.@ %a@])"
            identifier x pp_logic_type t pp_assertion a
      | AEq (x, y) ->
          o "(@[<hv>%a =@ %a@])" pp_term x pp_term y
      | ANeq (x, y) ->
          o "(@[<hv>%a <>@ %a@])" pp_term x pp_term y
      | AGt (x, y) ->
          o "(@[<hv>%a >@ %a@])" pp_term x pp_term y
      | ALt (x, y) ->
          o "(@[<hv>%a <@ %a@])" pp_term x pp_term y
      | AGe (x, y) ->
          o "(@[<hv>%a >=@ %a@])" pp_term x pp_term y
      | ALe (x, y) ->
          o "(@[<hv>%a <=@ %a@])" pp_term x pp_term y

  let rec pp_type fmt t =
    let o x = fprintf fmt x in
    match t with
      | TBase lt ->
          o "%a" pp_logic_type lt
      | TRef t ->
          o "@[<hov 2>ref@ (%a)@]" pp_type t
      | TFun (id, pt, rt) ->
          o "@[<hov 2>%a:@ %a ->@ %a@]" identifier id pp_type pt pp_type rt
      | TAnnot (pre, reads, writes, post, exceptions, t) ->
          o "@[<hv>{%a}@ %a%a%a%a@ @[<hv>{%a%a}@]@]"
            pp_assertion pre
            pp_type t
            (list
               ~left: (fun fmt () -> fprintf fmt "@ @[<hov 2>reads@ ")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt "@]")
               identifier) reads
            (list
               ~left: (fun fmt () -> fprintf fmt "@ @[<hov 2>writes@ ")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt "@]")
               identifier) writes
            (list
               ~left: (fun fmt () -> fprintf fmt "@ @[<hov 2>raises@ ")
               ~sep: comma_space
               ~right: (fun fmt () -> fprintf fmt "@]")
               (fun fmt (e, _) -> fprintf fmt "%a" identifier e)) exceptions
            pp_assertion post
            (list
               ~left: (fun fmt () -> fprintf fmt "@ | ")
               ~sep: (fun fmt () -> fprintf fmt "@ | ")
               (fun fmt (e, a) ->
                  fprintf fmt "@[<hov 2>%a =>@ %a@]"
                    identifier e pp_assertion a)) exceptions

  let rec pp_expr fmt e =
    let o x = fprintf fmt x in
    match e with
      | EVoid ->
          o "()"
      | EInt s
      | EReal s ->
          if s <> "" && s.[0] >= '0' && s.[0] <= '9' then
            o "%s" s
          else
            o "(%s)" s
      | EBool b ->
          o "%s" (if b then "True" else "False")
      | EVar x ->
          o "%a" identifier x
      | EAnd (x, y) ->
          o "(@[%a &&@ %a@])" pp_expr x pp_expr y
      | EOr (x, y) ->
          o "(@[%a ||@ %a@])" pp_expr x pp_expr y
      | EXor (x, y) ->
          o "(@[%a xor@ %a@])" pp_expr x pp_expr y
      | EAdd (x, y) ->
          o "(@[%a +@ %a@])" pp_expr x pp_expr y
      | ESub (x, y) ->
          o "(@[%a -@ %a@])" pp_expr x pp_expr y
      | EDiv (x, y) ->
          o "(@[ComputerDivision.div@ %a @ %a@])" pp_expr x pp_expr y
      | EMul (x, y) ->
          o "(@[%a *@ %a@])" pp_expr x pp_expr y
      | EMod (x, y) ->
          o "(@[%a mod@ %a@])" pp_expr x pp_expr y
      | ENot x ->
          o "(@[not %a@])" pp_expr x
      | ENeg x ->
          o "(@[- %a@])" pp_expr x
      | EDeref x ->
          o "!%a" identifier x
      | EIf (a, b, c) ->
          o "(@[<hv>@[<hv 2>if %a then@ %a@]@ @[<hv 2>else@ %a@]@])"
            pp_expr a pp_expr b pp_expr c
      | ELoop (c, i, v, b) ->
          o "@[<hv>@[<hv 2>while %a do@ invariant { %a%a }@ %a@]@ done@]"
            pp_expr c
            pp_assertion i
            (option
               (fun fmt (t, o) ->
                  fprintf fmt "variant %a%a"
                    pp_term t
                    (option ~left: (string " for ") identifier) o)) v
            (list ~sep: (fun fmt () -> fprintf fmt ";@ ") pp_expr) b
      | ESeq l ->
          o "(@[<hv>%a@])"
            (list ~sep: (fun fmt () -> fprintf fmt ";@ ") pp_expr) l
      | EApp (x, y) ->
          o "(@[%a@ %a@])" pp_expr x pp_expr y
      | ERaise (x, None) ->
          o "(@[raise %a@])" identifier x
      | ERaise (x, Some y) ->
          o "(@[raise (%a %a)@])" identifier x pp_expr y
      | ETry (e, x, a, f) ->
          o "@[<hv>@[<hv 2>try@ %a@]@ @[<hv 2>with %a%a ->@ %a@]@ end@]"
            pp_expr e identifier x
            (option ~left: (string " ") identifier) a
            pp_expr f
      | EFun (args, pre, post, exns, body) ->
          o "@[<hv 2>fun %a ->@ {%a}@ %a@ {%a%a}@]"
            (list
               ~sep: (fun fmt () -> fprintf fmt "@ ")
               (fun fmt (e, t) ->
                  fprintf fmt "(%a: %a)" identifier e pp_type t)) args
            pp_assertion pre
            pp_expr body
            pp_assertion post
            (list
               ~left: (fun fmt () -> fprintf fmt "@ | ")
               ~sep: (fun fmt () -> fprintf fmt "@ | ")
               (fun fmt (e, a) ->
                  fprintf fmt "@[<hov 2>%a =>@ %a@]"
                    identifier e pp_assertion a)) exns
      | EAssert a ->
          o "@[<hv 2>assert@ { %a }@]" pp_assertion a
      | ELabel (x, e) ->
          o "(@[label %a :@ %a@])" capitalized_identifier x pp_expr e
      | EBlackBox (TRef t) ->
          o "@[ref (any %a)@]" pp_type t
      | EBlackBox t ->
          o "@[(any %a)@]" pp_type t
      | EAbsurd ->
          o "absurd"
      | ELet (x, y, z) ->
          o "(@[<hv>@[<hv>@[<hv 2>let %a =@ %a@]@ in@]@ %a@])"
            identifier x pp_expr y pp_expr z
      | EAssign (id, x) ->
          o "(@[<hv 2>%a :=@ %a@])"
            identifier id pp_expr x
      | ELt (x, y) ->
          o "(@[%a <@ %a@])" pp_expr x pp_expr y
      | EGt (x, y) ->
          o "(@[%a >@ %a@])" pp_expr x pp_expr y
      | ELe (x, y) ->
          o "(@[%a <=@ %a@])" pp_expr x pp_expr y
      | EGe (x, y) ->
          o "(@[%a >=@ %a@])" pp_expr x pp_expr y
      | EEq (x, y) ->
          o "(@[%a =@ %a@])" pp_expr x pp_expr y
      | ETuple l ->
          o "(%a)" (list ~sep: comma_space pp_expr) l
      | EProj (n, i, e) ->
          o "@[<hv 2>(let (";
          for j = 1 to i do o "_," done;
          o "x";
          for j = i+1 to n-1 do o ",_" done;
          o ")@ = %a in x)@]" pp_expr e
      | ERef e ->
          o "(@[ref@ %a@])" pp_expr e

  let pp_decl fmt d =
    let o x = fprintf fmt x in
    match d with
      | DLogicType ([], name) ->
          o "{ @[<hov 2>type@ %a@] }@.@." identifier name
      | DLogicType ([a], name) ->
          o "{ @[<hov 2>type@ %a %a@] }@.@."
            identifier name prime_identifier a
      | DLogicType (l, name) ->
          o "{ @[<hov 2>type@ %a %a@] }@.@."
            identifier name
            (list ~sep: space prime_identifier) l
      | DLogic (name, args, res) ->
          o "{ @[<hov 2>logic %a%a:@ %a@] }@.@."
            identifier name
            (list ~left: space ~sep: space
               (fun fmt t -> fprintf fmt "(%a)" pp_logic_type t)) args
            pp_logic_type res
      | DLogicDef (name, args, res, body) ->
          o "{ @[<hov 2>logic %a%a:@ %a =@ %a@] }@.@."
            identifier name
            (list ~left: space ~sep: space
               (fun fmt (id, ty) ->
                 fprintf fmt "(%a: %a)" identifier id pp_logic_type ty)) args
            pp_logic_type res
            pp_term body
      | DParam (name, t) ->
          o "@[<hov 2>parameter %a:@ %a@]@.@."
            identifier name pp_type t
      | DLet (name, e) ->
          o "@[<hv 2>let %a =@ %a@]@.@."
            identifier name pp_expr e
      | DPredicate (name, args, body) ->
          o "{ @[<hv 2>logic %a%a%a@] }@.@."
            identifier name
            (list
               ~left: space
               ~sep: space
               (fun fmt (id, ty) ->
                  fprintf fmt "(%a: %a)" identifier id pp_logic_type ty)) args
            (option ~left: (fun fmt () -> fprintf fmt " =@ ") pp_assertion)
            body
      | DAxiom (is_lemma, name, body) ->
          o "{ @[<hv 2>%s %a:@ %a@]@ }@]@.@."
	    (if is_lemma then "lemma" else "axiom")
            capitalized_identifier name pp_assertion body

  let to_channel ch file =
    let fmt = formatter_of_out_channel ch in
    fprintf fmt "{ use import int.ComputerDivision@\n  \
                   use import capucine_why3.Capucine_ctx }@.@.";
    List.iter (pp_decl fmt) file
end

let to_channel x =
  match !Options.output_kind with
    | Options.Why2 -> use_boron false; Why2.to_channel x
    | Options.Why3 -> Why3.to_channel x
    | Options.Boron -> use_boron true; Why2.to_channel x

let to_file name file =
  let ch = open_out name in
  to_channel ch file;
  close_out ch
